<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
/**
 * Class to handle backend functionality
 */
if ( ! class_exists( 'WWP_Wholesale_Pricing_Backend' ) ) {

	class WWP_Wholesale_Pricing_Backend extends WWP_Wholesale_Reports {

		public function __construct() {
			$settings = get_option( 'wwp_wholesale_pricing_options', true );
			add_action( 'admin_menu', array( $this, 'wwp_register_custom_menu_page' ), 10 );
			add_action( 'admin_menu', array( $this, 'wwp_register_custom_menu_page_2' ), 200 );
			add_action( 'admin_init', array( $this, 'wwp_request_options' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'wwp_admin_script_style' ) );
			add_filter( 'anr_settings_fields', array( $this, 'wwp_wholesale_recaptcha' ) );
			add_action( 'wp_ajax_register_redirect', array( $this, 'register_redirect' ) );
			add_action( 'wp_ajax_nopriv_register_redirect', array( $this, 'register_redirect' ) );

			add_filter( 'woocommerce_product_data_tabs', array( $this, 'wwp_add_wholesale_product_data_tab' ), 99, 1 );
			add_action( 'admin_head', array( $this, 'wcpp_custom_style' ) );
			add_filter( 'woocommerce_screen_ids', array( $this, 'wwp_screen_ids' ), 10, 1 );

			add_action( 'add_meta_boxes', array( $this, 'wwp_register_meta_box' ) );

			add_action( 'save_post_shop_order', array( $this, 'update_order_wwp_form_data_json_value' ) );
			add_action( 'restrict_manage_posts', array( $this, 'shop_order_user_role_filter' ) );
			add_filter( 'pre_get_posts', array( $this, 'shop_order_user_role_posts_where' ) );
			add_action( 'woocommerce_product_data_panels', array( $this, 'wwp_add_wholesale_product_data_fields_multi' ) );
			add_action( 'woocommerce_process_product_meta', array( $this, 'wwp_woo_wholesale_fields_save_multi' ), 99 );
			
			add_filter( 'manage_edit-product_columns', array( $this, 'change_columns_filter'), 10, 1 );
			add_action( 'manage_product_posts_custom_column', array( $this, 'manage_product_posts_custom_column' ), 10, 2 );
			
			add_filter( 'woocommerce_coupon_data_tabs', array( $this, 'woocommerce_coupon_data_tabs' ), 10, 1 );
			add_action( 'woocommerce_coupon_data_panels', array( $this, 'woocommerce_coupon_data_panels') );
			add_action( 'woocommerce_coupon_options_save', array( $this, 'woocommerce_coupon_options_save') );
		}
		public function woocommerce_coupon_data_tabs( $tabs ) {
			$tabs['wwp_wholesale'] = array( 
				'label'  => __( 'Wholesale', 'webdav-coupon-plugin' ), 
				'target' => 'wwp_wholesale_data', 
				'class'  => 'wwp_wholesale_data' 
			); 
			return $tabs; 
		}
		public function woocommerce_coupon_data_panels() { 
			global $post;
			$coupon_id = $post->ID;
			$roles      = array();
			$taxroles   = get_terms( 'wholesale_user_roles', array('hide_empty' => false) );
			if ( ! empty( $taxroles ) ) {
				foreach ( $taxroles as $key => $role ) {
					$roles[ $role->slug ] = $role->name;
				}
			}
			?>
			<div id="wwp_wholesale_data" class="panel woocommerce_options_panel">
				<?php 
				woocommerce_wp_checkbox( 
					array(
						'id' => 'wwp_coupon_enable',
						'label' => esc_html__( 'Enable Wholesale', 'woocommerce-wholesale-pricing' ),
						'description' => esc_html__( 'Enable to apply coupon on restricted user roles.', 'woocommerce')
					) 
				); 
				
				woocommerce_wp_select(
					array(
						'id' => 'wwp_coupon_type',
						'label' => esc_html__( 'Type', 'woocommerce-wholesale-pricing' ),
						'options' => 
									array(
										'included' => 'Included',
										'excluded' => 'Excluded'
									),
						)
				);
				
				$value = get_post_meta( $coupon_id, 'wwp_coupon_roles', true );		 
				woocommerce_wp_select(
					array(
						'id'                => 'wwp_coupon_roles[]',
						'label'             => esc_html__( 'User Roles', 'woocommerce-wholesale-pricing' ),
						'type'              => 'select',
						'class'             => 'wc-enhanced-select',
						'style'             => 'min-width: 50%;',
						'options'           => $roles,
						'value'             => $value,
						'custom_attributes' => array(
							'multiple' => 'multiple',
						),
					)
				); 
				
				woocommerce_wp_textarea_input(
					array(
						'id' => 'wwp_coupon_not_allowed_message',
						'placeholder' => 'You are not allowed to use this coupon code.',
						'label' => __('Invalid user role message', 'woocommerce')
					)
				);
				
				?>
			</div>
			<?php
		}
		
		public function woocommerce_coupon_options_save( $post_id = null ) { 
			$post = wwp_get_post_data('');
			$wwp_coupon_enable = isset( $post['wwp_coupon_enable'] ) ? wc_clean( $post['wwp_coupon_enable'] ) : '';
			update_post_meta( $post_id, 'wwp_coupon_enable', $wwp_coupon_enable );
			
			$wwp_coupon_type = isset( $post['wwp_coupon_type'] ) ? wc_clean( $post['wwp_coupon_type'] ) : '';
			update_post_meta( $post_id, 'wwp_coupon_type', $wwp_coupon_type );	
			
			$wwp_coupon_roles = isset( $post['wwp_coupon_roles'] ) ? wc_clean( $post['wwp_coupon_roles'] ) : '';
			update_post_meta( $post_id, 'wwp_coupon_roles', $wwp_coupon_roles );	
			
			$wwp_coupon_not_allowed_message = isset( $post['wwp_coupon_not_allowed_message'] ) ? wc_clean( $post['wwp_coupon_not_allowed_message'] ) : '';
			update_post_meta( $post_id, 'wwp_coupon_not_allowed_message', $wwp_coupon_not_allowed_message );
		}
		
		public function change_columns_filter( $columns ) {
			$columns['Wholesale'] = 'Wholesale Pricing On';
			return $columns; 
		}
		
		public function manage_product_posts_custom_column( $column, $product_id ) {
			if ( 'Wholesale' == $column ) {
				$allterms = get_terms( 'wholesale_user_roles', array( 'hide_empty' => false ) );
				foreach ( $allterms as $allterm ) { 
					$role = $allterm->term_id;
					$data = get_post_meta( $product_id, 'wholesale_multi_user_pricing', true );
					if ( isset( $data[ $role ] ) ) { 
						?>
						<button class="wholesale_on button">
							<div class="border_on"></div><?php echo esc_attr($allterm->name); ?> Product
						</button>
						<?php 
					}
					$terms = get_the_terms( $product_id, 'product_cat' );
					if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
						foreach ( $terms as $term ) {
							$data = get_term_meta( $term->term_id, 'wholesale_multi_user_pricing', true );
							if ( isset( $data[ $role ] ) && isset( $data[ $role ]['discount_type'] ) && isset( $data[ $role ]['wholesale_price'] ) ) {
								?>
								<button class="wholesale_on button">
									<div class="border_on"></div><?php echo esc_attr($allterm->name); ?> Category
								</button>
								<?php  
							}
						}
					}
					$data = get_option( 'wholesale_multi_user_pricing' );
					if ( isset( $data[ $role ] ) && isset( $data[ $role ]['discount_type'] ) && isset( $data[ $role ]['wholesale_price'] ) ) {
						?>
							<button class="wholesale_on button">
								<div class="border_on"></div><?php echo esc_attr($allterm->name); ?> Global
							</button>						
						<?php 
					}	

				}
			}
		}
		
		public function register_redirect() {
			$args     = array(
				'posts_per_page' => -1,
				'post_type'      => array( 'page', 'product' ),
				'post_status'    => 'publish',
				's'              => wwp_get_get_data( 'name' ),
			);
			$query    = get_posts( $args );
			$response = array();
			foreach ( $query as $r ) {
				$post_title = $r->post_title . ' - (' . $r->ID . ')';
				$response[] = array(
					'value' => $r->ID,
					'label' => $post_title,
				);
			}
			echo wp_kses_post( wwp_get_get_data( 'callback' ) . '(' . json_encode( $response ) . ')' );
			die();
		}

		public function shop_order_user_role_filter() {

			if ( isset( $_GET['post_type'] ) && 'shop_order' != $_GET['post_type'] ) {
				return;
			}

			if ( ! empty( $_GET['_user_role'] ) ) {
				$user_role = sanitize_text_field( $_GET['_user_role'] );
			} else {
				$user_role = '';
			}
			?>
			<select name="_user_role" class="">
				<option value=""><?php esc_html_e( 'All', 'woocommerce-wholesale-pricing' ); ?></option>
				<option <?php selected( $user_role, 'Wholesaler' ); ?> value="Wholesaler"><?php esc_html_e( 'Wholesaler', 'woocommerce-wholesale-pricing' ); ?></option>
				<option <?php selected( $user_role, 'non-Wholesaler' ); ?> value="non-Wholesaler"><?php esc_html_e( 'Non - Wholesaler', 'woocommerce-wholesale-pricing' ); ?></option>
			</select>
			<?php
		}

		public function shop_order_user_role_posts_where( $query ) {

			if ( ! $query->is_main_query() || ! isset( $_GET['_user_role'] ) ) {
				return;
			}
			$wholesale_user_ids = array();
			$all_wholesale_role = array();
			$allterms           = get_terms( 'wholesale_user_roles', array( 'hide_empty' => false ) );
			foreach ( $allterms as $allterm_key => $allterm_value ) {
				array_push( $all_wholesale_role, $allterm_value->slug );
			}

			$wholesale_user_ids = get_users(
				array(
					'role__in' => $all_wholesale_role,
					'fields'   => 'ID',
				)
			);

			switch ( $_GET['_user_role'] ) {
				case 'Wholesaler':
					$query->set(
						'meta_query',
						array(
							array(
								'key'     => '_customer_user',
								'compare' => 'IN',
								'value'   => $wholesale_user_ids,
							),
						)
					);
					break;
				case 'non-Wholesaler':
					$query->set(
						'meta_query',
						array(
							array(
								'key'     => '_customer_user',
								'compare' => 'NOT IN',
								'value'   => $wholesale_user_ids,
							),
						)
					);
					break;
			}
			if ( empty( $wholesale_user_ids ) ) {
				$query->set( 'posts_per_page', 0 );
			}
		}



		public function wwp_woo_wholesale_fields_save_multi( $post_id ) {
			if ( ! isset( $_POST['wwp_product_wholesale_nonce'] ) || ! wp_verify_nonce( wc_clean( $_POST['wwp_product_wholesale_nonce'] ), 'wwp_product_wholesale_nonce' ) ) {
				return;
			}

			// hide product for customer
			$_wwp_hide_for_customer = isset( $_POST['_wwp_hide_for_customer'] ) ? wc_clean( $_POST['_wwp_hide_for_customer'] ) : '';
			update_post_meta( $post_id, '_wwp_hide_for_customer', esc_attr( $_wwp_hide_for_customer ) );

			// hide product for visitor
			$_wwp_hide_for_visitor = isset( $_POST['_wwp_hide_for_visitor'] ) ? wc_clean( $_POST['_wwp_hide_for_visitor'] ) : '';
			update_post_meta( $post_id, '_wwp_hide_for_visitor', esc_attr( $_wwp_hide_for_visitor ) );
			
			$wwp_icon_url = isset( $_POST['wwp_icon_url'] ) ? wc_clean( $_POST['wwp_icon_url'] ) : '';
			update_post_meta( $post_id, 'wwp_icon_url', esc_attr( $wwp_icon_url ) );
			
			$wwp_image_url = isset( $_POST['wwp_image_url'] ) ? wc_clean( $_POST['wwp_image_url'] ) : '';
			update_post_meta( $post_id, 'wwp_image_url', esc_attr( $wwp_image_url ) );
			
			$wwp_attachment_title = isset( $_POST['wwp_attachment_title'] ) ? wc_clean( $_POST['wwp_attachment_title'] ) : '';
			update_post_meta( $post_id, 'wwp_attachment_title', esc_attr( $wwp_attachment_title ) );
			
			$wwp_attachment_text_link = isset( $_POST['wwp_attachment_text_link'] ) ? wc_clean( $_POST['wwp_attachment_text_link'] ) : '';
			update_post_meta( $post_id, 'wwp_attachment_text_link', esc_attr( $wwp_attachment_text_link ) );
		}


		public function update_order_wwp_form_data_json_value( $post_id ) {
			$wwp_form_data_json = wwp_get_post_data( 'wwp_form_data_json' );
			if ( isset( $wwp_form_data_json ) ) {
				$wwp_form_data_json = wc_clean( $wwp_form_data_json );
				update_post_meta( $post_id, 'wwp_form_data_json', $wwp_form_data_json );
			}

		}
		public function wwp_register_meta_box() {

			$registrations = get_option( 'wwp_wholesale_registration_options' );
			if ( isset( $registrations['display_fields_checkout'] ) && 'yes' == $registrations['display_fields_checkout'] ) {
				add_meta_box(
					'wwp_form_builder',
					esc_html__( 'Checkout Extra Fields Data', 'woocommerce-wholesale-pricing' ),
					array( $this, 'wwp_meta_box_callback' ),
					'shop_order',
					'advanced',
					'high'
				);
			}

		}

		public function wwp_meta_box_callback( $order_id ) {
			 echo wp_kses_post( render_form_builder( 'get_post_meta', $order_id->ID ) );
		}

		public function wwp_wholesale_recaptcha( $fields ) {

			$add                                = array( 'wwp_wholesale_recaptcha' => 'Wholesale Registration Form' );
			$fields['enabled_forms']['options'] = array_merge( $fields['enabled_forms']['options'], $add );
			return $fields;

		}

		public function wwp_request_options() {
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_admin_request_notification' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_admin_request_subject' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_admin_request_body' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_request_approve_notification' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_email_request_subject' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_email_request_body' );

			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_user_registration_notification' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_registration_notification_subject' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_registration_notification_body' );

			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_user_rejection_notification' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_rejection_notification_subject' );
			register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_rejection_notification_body' );

			// version 1.3.0 For Subscriptions User Role Update Notification
			if ( in_array( 'woocommerce-subscriptions/woocommerce-subscriptions.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
				register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_subscription_role_notification' );
				register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_subscription_role_subject' );
				register_setting( 'wwp_wholesale_request_notifications', 'wwp_wholesale_subscription_role_body' );
			}
			// ends version 1.3.0
			$this->upgrade_plugin_fixes();
		}

		public function upgrade_plugin_fixes() {
			$settings = get_option( 'wwp_wholesale_pricing_options', true );
			// if ( isset( $settings['enable_upgrade'] ) && 'yes' == $settings['enable_upgrade'] ) {
				// $settings['request_again_submit'] = 'yes';
				// $settings['enable_upgrade'] = 'no';
				// update_option('wwp_wholesale_pricing_options', $settings);
			// }

			if ( isset( $settings['upgrade_btn_label'] ) && ! empty( $settings['upgrade_btn_label'] ) ) {
				$settings['upgrade_tab_text']  = $settings['upgrade_btn_label'];
				$settings['upgrade_btn_label'] = '';
				update_option( 'wwp_wholesale_pricing_options', $settings );
			}
		}

		public function wwp_register_custom_menu_page() {
			$check = apply_filters( 'wwp_wholesales_menus', true );
			if ($check) {
				add_menu_page(
					esc_html__( 'Wholesale Pricing', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Wholesale', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale',
					'wwp_wholesale',
					array( $this, 'wwp_wholesale_reports_callback' ),
					'dashicons-store',
					51
				);
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Wholesale For WooCommerce', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Dashboard', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale_reports',
					'wwp_wholesale',
					array( $this, 'wwp_wholesale_reports_callback' )
				);
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Wholesale Settings', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Settings', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale_settings',
					'wwp_wholesale_settings',
					array( $this, 'wwp_wholesale_page_callback' )
				);
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'User Roles', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'User Roles', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale_user_role',
					'edit-tags.php?taxonomy=wholesale_user_roles'
				);
	
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Notifications', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Notifications', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale_notifications',
					'wwp_wholesale_notifcations',
					array( $this, 'wwp_wholesale_notifications_callback' )
				);
	
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Import', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Import', 'woocommerce-wholesale-pricing' ),
					'manage_wholesale_import',
					'edit.php?post_type=product&page=product_importer'
				);
			}
		}
 
		public  function wwp_wholesale_reports_callback() {
			?>
			<div class="container remove-space">
			<br>
				<h2 class="text-center"><?php esc_html_e( 'Wholesale For WooCommerce', 'woocommerce-wholesale-pricing' ); ?></h2><hr>
			</div>
			<?php 
			// This hook call from reports page
			
			// Reports Section
			do_action( 'wwp_dashboard_reports', $this);
			// User Request Section
			do_action( 'wwp_dashboard_user_requests', $this);
			// Wholesale recent Order
			do_action( 'wwp_dashboard_recent_order', $this);
		}
		
		

		public function wwp_register_custom_menu_page_2() {

			$check = apply_filters( 'wwp_wholesales_menus', true );
	
			if ( $check ) {
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Import', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Import', 'woocommerce-wholesale-pricing' ),
					'manage_options',
					'edit.php?post_type=product&page=product_importer'
				);
	
				add_submenu_page(
					'wwp_wholesale',
					esc_html__( 'Export', 'woocommerce-wholesale-pricing' ),
					esc_html__( 'Export', 'woocommerce-wholesale-pricing' ),
					'manage_options',
					'edit.php?post_type=product&page=product_exporter'
				);
			}
		}

		public static function wwp_wholesale_page_callback() {

			$settings = ! empty( get_option( 'wwp_wholesale_pricing_options' ) ) ? get_option( 'wwp_wholesale_pricing_options', true ) : array();
			if ( isset( $_POST['save-wwp_wholesale'] ) ) {
				if ( isset( $_POST['wwp_wholesale_register_nonce'] ) || wp_verify_nonce( wc_clean( $_POST['wwp_wholesale_register_nonce'] ), 'wwp_wholesale_register_nonce' ) ) {
					$settings                                 = isset( $_POST['options'] ) ? wc_clean( $_POST['options'] ) : '';
					$settings['enable_registration_page']     = isset( $settings['enable_registration_page'] ) ? 'yes' : 'no';
					$settings['wholesaler_allow_minimum_qty'] = isset( $settings['wholesaler_allow_minimum_qty'] ) ? 'yes' : 'no';
					$settings['restrict_store_access']        = isset( $settings['restrict_store_access'] ) ? 'yes' : 'no';
					$settings['over_right_wholesale_form']    = isset( $settings['over_right_wholesale_form'] ) ? 'yes' : 'no';
					$settings['wholesaler_prodcut_only']      = isset( $settings['wholesaler_prodcut_only'] ) ? 'yes' : 'no';
					$settings['enable_upgrade']               = isset( $settings['enable_upgrade'] ) ? 'yes' : 'no';
					$settings['disable_auto_role']            = isset( $settings['disable_auto_role'] ) ? 'yes' : 'no';
					$settings['retailer_disabled']            = isset( $settings['retailer_disabled'] ) ? 'yes' : 'no';
					$settings['save_price_disabled']          = isset( $settings['save_price_disabled'] ) ? 'yes' : 'no';
					$settings['enable_admin_new_order_item']  = isset( $settings['enable_admin_new_order_item'] ) ? 'yes' : 'no';
					$settings['payment_method_name'] = array();
					
					if ( isset( $_POST['options']['payment_method_name'] ) ) {
						$post = $_POST;
						foreach ( ( array ) $post['options']['payment_method_name'] as $key => $value ) {
							if ($value) {
								$settings['payment_method_name'][$value] = 'no';
							}
						}
					}

					update_option( 'wwp_wholesale_pricing_options', $settings );
					$roles         = get_terms(
						'wholesale_user_roles',
						array(
							'hide_empty' => false,
						)
					);
					$data          = array();
					$cart_discount = array();
					if ( ! empty( $roles ) ) {
						foreach ( $roles as $key => $role ) {
							if ( isset( $_POST[ 'role_' . $role->term_id ] ) ) {

								if ( isset( $_POST[ 'role_' . $role->term_id ] ) ) {
									$data[ $role->term_id ]['slug'] = $role->slug;
								}
								if ( isset( $_POST[ 'discount_type_' . $role->term_id ] ) ) {
									$data[ $role->term_id ]['discount_type'] = wc_clean( $_POST[ 'discount_type_' . $role->term_id ] );
								}
								if ( isset( $_POST[ 'wholesale_price_' . $role->term_id ] ) ) {
									$data[ $role->term_id ]['wholesale_price'] = is_numeric( wc_clean( $_POST[ 'wholesale_price_' . $role->term_id ] ) ) ? wc_clean( $_POST[ 'wholesale_price_' . $role->term_id ] ) : '';
								}
								if ( isset( $_POST[ 'min_quatity_' . $role->term_id ] ) ) {
									$data[ $role->term_id ]['min_quatity'] = is_numeric( wc_clean( $_POST[ 'min_quatity_' . $role->term_id ] ) ) ? wc_clean( $_POST[ 'min_quatity_' . $role->term_id ] ) : 1;
								}
							}
							if ( isset( $_POST[ 'role_' . $role->term_id . '_cart' ] ) ) {
								if ( isset( $_POST[ 'role_' . $role->term_id . '_cart' ] ) ) {
									$cart_discount[ $role->term_id ]['slug'] = $role->slug;
								}
								if ( isset($_POST['discount_type_' . $role->term_id . '_cart']) ) {
									$cart_discount[$role->term_id]['discount_type'] = wc_clean($_POST['discount_type_' . $role->term_id . '_cart']);
								}
							}
						}
					}
					if ( isset($_POST['wholesale_multi_user_cart_discount']) ) {
						$post = wwp_get_post_data('');
						$tier_pricing = $post['wholesale_multi_user_cart_discount'];
						update_option( 'cart_tire_preces', $tier_pricing );
					}
					update_option( 'wholesale_multi_user_pricing', $data );
					update_option( 'wholesale_multi_user_cart_discount', $cart_discount );
				}
			}
			?>
			<form action="" id="wwp-global-settings" method="post">
				<?php
					$wwp_wholsale_plg_title = apply_filters( 'wwp_wholsale_plg_title', esc_html__( 'Wholesale For WooCommerce', 'woocommerce-wholesale-pricing' ) );
				?>
				<h2 class="text-center"><?php echo esc_attr( $wwp_wholsale_plg_title ); ?></h2><hr>
				<?php wp_nonce_field( 'wwp_wholesale_register_nonce', 'wwp_wholesale_register_nonce' ); ?>
				
				<div class="tab" role="tabpanel">
					<!-- Nav tabs -->
				<div class="row">
					<div class="col-md-3 col-sm-12">
							<?php 
							$wwp_settings_tab_lists = apply_filters( 'wwp_settings_tab_lists', 
								array(
									'section1' => esc_html__( 'General', 'wwp-wholesale' ),
									'section1_5' => esc_html__( 'Subscription', 'wwp-wholesale' ),
									'section2' => esc_html__( 'Wholesale Price Global', 'wwp-wholesale' ),
									'section9' => esc_html__( 'Cart Total Discount', 'wwp-wholesale' ),
									'section3' => esc_html__( 'Labels', 'wwp-wholesale' ),
									'section4' => esc_html__( 'Login Restrictions', 'wwp-wholesale' ),
									'section5' => esc_html__( 'Product Visibility', 'wwp-wholesale' ),
									'section6' => esc_html__( 'Upgrade Settings', 'wwp-wholesale' ),
									'section10' => esc_html__( 'Payment Methods', 'wwp-wholesale' ),
									'section7' => esc_html__( 'Additional CSS', 'wwp-wholesale' ),
									'section8' => esc_html__( 'Compatible Plugins', 'wwp-wholesale' ),
									
									)
								);
							?>
					<ul class="nav nav-tabs" role="tablist">
						<?php
						foreach ( $wwp_settings_tab_lists as $section_id => $section_name ) {
							?>
							<li role="presentation">
								<a href="#<?php echo esc_attr( $section_id ); ?>" role="tab" data-toggle="tab"><?php echo esc_html( $section_name ); ?></a>
							</li>
							<?php
						}
						?>
					</ul>
					</div>
					<div class="col-md-9 col-sm-12">	
						<div class="tab-content tabs">
							<div role="tabpanel" class="tab-pane fade" id="section1">
								<table class="form-table wwp-main-settings">
									<tbody>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Wholesale Pricing Mode', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<p>
													<input class="inp-cbx" style="display: none" id="single_wholesaler_role" type="radio" value="single" name="options[wholesale_role]" <?php echo ( isset( $settings['wholesale_role'] ) && 'single' == $settings['wholesale_role'] ) ? 'checked' : ''; ?>>
													<label class="cbx" for="single_wholesaler_role">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Single Wholesale Role', 'woocommerce-wholesale-pricing' ); ?></span>												
													</label>
													<span data-tip="Default settings for single user role." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>											
												</p>
												<p>
													<input class="inp-cbx" style="display: none" id="multiple_wholesaler_role" type="radio" value="multiple" name="options[wholesale_role]" <?php echo ( isset( $settings['wholesale_role'] ) && 'multiple' == $settings['wholesale_role'] ) ? 'checked' : ''; ?>>
													<label class="cbx" for="multiple_wholesaler_role">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>													
														<span><?php esc_html_e( ' Multiple Wholesale Roles', 'woocommerce-wholesale-pricing' ); ?></span>													
													</label>
													<span data-tip="Manage prices according to multiple wholesaler user roles." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
												</p>
											</td>
										</tr>
										
										<tr scope="row" id="multiroledropdown">
										
											<th><label for="default_multipe_wholesale_roles"><?php esc_html_e( 'Default Multi Wholesale Roles', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<?php
												$allterms = get_terms( 'wholesale_user_roles', array( 'hide_empty' => false ) );
												?>
												<select id="default_multipe_wholesale_roles" class="regular-text" name="options[default_multipe_wholesale_roles]" >
													<option value=""><?php esc_html_e( 'Select Wholesale Role', 'woocommerce-wholesale-pricing' ); ?></option>
													<?php
													foreach ( $allterms as $allterm ) {
														$selected = '';
														if ( isset( $settings['default_multipe_wholesale_roles'] ) && $settings['default_multipe_wholesale_roles'] == $allterm->slug ) {
																$selected = 'selected';
														}
														?>
														<option value="<?php echo esc_attr( $allterm->slug ); ?>" <?php echo esc_html( $selected ); ?>><?php echo esc_html( $allterm->name ); ?></option>
													<?php } ?> 
												</select>     
												<span data-tip="Define the default wholesaler role for your user." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
										
										<tr scope="row">
											<th>
												<label for=""><?php esc_html_e( 'Disable Auto Approval', 'woocommerce-wholesale-pricing' ); ?></label>
											</th>
											<td>
												<p>
													<input class="inp-cbx" style="display: none" id="disable_auto_role" type="checkbox" value="yes" name="options[disable_auto_role]" <?php echo ( isset( $settings['disable_auto_role'] ) && 'yes' == $settings['disable_auto_role'] ) ? 'checked' : ''; ?>>
													<label class="cbx cbx-square" for="disable_auto_role">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Check this option to disable auto approval for wholesale user role registration requests', 'woocommerce-wholesale-pricing' ); ?></span>
													</label>
												</p>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Enable Registration Link', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<p>
													<input class="inp-cbx" style="display: none" id="enable_registration_page" type="checkbox" value="yes" name="options[enable_registration_page]" <?php echo ( isset( $settings['enable_registration_page'] ) && 'yes' == $settings['enable_registration_page'] ) ? 'checked' : ''; ?>>
													<label class="cbx cbx-square" for="enable_registration_page">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Enable wholesale registration link on my account page (You must enable registration form on myaccount page to work this functionality)', 'woocommerce-wholesale-pricing' ); ?></span>
													</label>
												</p>
											</td>
										</tr>
										<tr scope="row">
											<th><label for="registration_page_for_wholesale"><?php esc_html_e( 'Registration Page', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<?php
												$args  = array(
													'posts_per_page'   => -1,
													'post_type'        => 'page',
													'post_status'      => 'publish',
												);
												$pages = get_posts( $args );
												?>
												<select id="registration_page_for_wholesale" class="regular-text" name="options[registration_page]" >
													<option value=""><?php esc_html_e( 'Select Page', 'woocommerce-wholesale-pricing' ); ?></option>
													<?php
													foreach ( $pages as $page ) {
														$selected = '';
														if ( isset( $settings['registration_page'] ) && $settings['registration_page'] == $page->ID ) {
															$selected = 'selected';
														}
														?>
														<option value="<?php echo esc_attr( $page->ID ); ?>" <?php echo esc_html( $selected ); ?>><?php echo esc_html( $page->post_title ); ?></option>
													<?php } ?> 
												</select>  
												<span data-tip="Select the page on which you want to display your wholesale registration form." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>								
											</td>
										</tr>
										<tr scope="row">
											<th><label for="registration_page_for_wholesale"><?php esc_html_e( 'Registration Page Redirect', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
											<?php
											if ( isset( $settings['register_redirect'] ) && ! empty( $settings['register_redirect'] ) ) {
												$register_redirect_title = get_the_title( $settings['register_redirect'] ) . ' - (' . $settings['register_redirect'] . ')';
											} else {
												$register_redirect_title       = '';
												$settings['register_redirect'] = '';
											}
											?>
												<input id="register_redirect_autocomplete" type="text" class="regular-text" value="<?php echo esc_attr( $register_redirect_title ); ?>" >
												<input id="register_redirect" type="hidden" class="" name="options[register_redirect]" value="<?php echo esc_attr( $settings['register_redirect'] ); ?>" > 
												<span data-tip="Please select a page or product to redirect after a successful registration." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
										
										<tr scope="row" id="attachment_location">
											<th><label for=""><?php esc_html_e( 'Select Attachment location', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<?php
												$attachment_location_value = array(
													'woocommerce_before_add_to_cart_button' => 'Woocommerce before add to cart button',
													'woocommerce_after_add_to_cart_button' => 'Woocommerce after add to cart button',
													'woocommerce_after_single_product_summary' => 'Woocommerce after single product summary'
												);
												?>
												<select id="attachment_location_value" class="regular-text" name="options[wwp_attachment_location]" >
													<option value=""><?php esc_html_e( 'Select Attachment location', 'woocommerce-wholesale-pricing' ); ?></option>
													<?php
													foreach ( $attachment_location_value as $key => $value ) {
														$selected = '';
														if ( isset( $settings['wwp_attachment_location'] ) && $settings['wwp_attachment_location'] == $key ) {
															$selected = 'selected';
														}
														?>
														<option value="<?php echo esc_attr( $key ); ?>" <?php echo esc_html( $selected ); ?>><?php echo esc_html( $value ); ?></option>
													<?php } ?> 
												</select>  
												<span data-tip="Select a product attachment location." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>								
											</td>
										</tr>
										
										
										
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Order Notification Email', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
										  
											<p>
													<input class="inp-cbx" style="display: none" id="emailuserrole" type="radio" value="order_email_user_role" name="options[emailuserrole]" <?php echo ( isset( $settings['emailuserrole'] ) && 'order_email_user_role' == $settings['emailuserrole'] ) ? 'checked' : ''; ?>>
													<label class="cbx" for="emailuserrole">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' User Role', 'woocommerce-wholesale-pricing' ); ?></span>												
													</label>
													
													<input class="inp-cbx" style="display: none" id="order_email_custom" type="radio" value="order_email_custom" name="options[emailuserrole]"<?php echo ( isset( $settings['emailuserrole'] ) && 'order_email_custom' == $settings['emailuserrole'] ) ? 'checked' : ''; ?>>
													<label class="cbx" for="order_email_custom">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Custom Email', 'woocommerce-wholesale-pricing' ); ?></span>												
													</label>
													<span data-tip="Select option to send wholesale order notification email." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>											
											</p>
											
											</td>
										</tr>
										 
										<tr scope="row" id="select_role_wrap">
											<th><label for=""><?php esc_html_e( 'Select Role', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
										   
												<select id="email_user_role_value" class="regular-text" name="options[email_user_role_value]" >
													<option value=""><?php esc_html_e( 'Select Role', 'woocommerce-wholesale-pricing' ); ?></option>
													<?php
													foreach ( get_editable_roles() as $role_name => $role_info ) {
														$selected = '';
														if ( isset( $settings['email_user_role_value'] ) && $settings['email_user_role_value'] == $role_name ) {
															$selected = 'selected';
														}
														?>
														<option value="<?php echo esc_attr( $role_name ); ?>" <?php echo esc_html( $selected ); ?>><?php echo esc_html( $role_info['name'] ); ?></option>
													<?php } ?> 
												</select>  
												<span data-tip="Select a role of email notifications." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>								
											</td>
										</tr>

										<?php 
										if ( !isset($settings['order_custom_email_value']) ) { 
											$settings['order_custom_email_value'] = '';
										} 
										?>
										<tr scope="row" id="select_email_custom_wrap">
											<th><label for=""><?php esc_html_e( 'Recipient(s)', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
											<input id="order_custom_email_value"  name="options[order_custom_email_value]"  type="text" class="regular-text" value="<?php echo esc_attr( $settings['order_custom_email_value'] ); ?>" >
											<span data-tip="Enter recipients (comma separated)" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>								
											</td>
										</tr>
										 
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Manual Wholesale Order', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<p>
													<input class="inp-cbx" style="display: none" id="enable_admin_new_order_item" type="checkbox" value="yes" name="options[enable_admin_new_order_item]" <?php echo ( isset( $settings['enable_admin_new_order_item'] ) && 'yes' == $settings['enable_admin_new_order_item'] ) ? 'checked' : ''; ?>>
													<label class="cbx cbx-square" for="enable_admin_new_order_item">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Allow admin to add wholesale orders', 'woocommerce-wholesale-pricing' ); ?></span>
													</label>
												</p>
											</td>
										</tr>
										
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Enable Coupon', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<p>
													<input class="inp-cbx" style="display: none" id="enable_coupon" type="checkbox" value="yes" name="options[enable_coupon]" <?php echo ( isset( $settings['enable_coupon'] ) && 'yes' == $settings['enable_coupon'] ) ? 'checked' : ''; ?>>
													<label class="cbx cbx-square" for="enable_coupon">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' Enable Wholesale Coupon', 'woocommerce-wholesale-pricing' ); ?></span>
													</label>
												</p>
											</td>
										</tr>
										 
									</tbody>
								</table>							
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section1_5">
								<table class="form-table wwp-main-settings">
									<tbody>
										<?php
										if ( in_array( 'woocommerce-subscriptions/woocommerce-subscriptions.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) && isset( $settings['wholesale_role'] ) ) {
											?>
											<tr scope="row">
												<th><label for=""><?php esc_html_e( 'Enable Wholesale For Subscription', 'woocommerce-wholesale-pricing' ); ?></label></th>
												<td>
													<input class="inp-cbx" style="display: none" id="enable_subscription" type="checkbox" value="yes" name="options[enable_subscription]" <?php echo ( isset( $settings['enable_subscription'] ) && 'yes' == $settings['enable_subscription'] ) ? 'checked' : ''; ?>>
													<label class="cbx cbx-square" for="enable_subscription">
														<span>
															<svg width="12px" height="9px" viewbox="0 0 12 9">
																<polyline points="1 5 4 8 11 1"></polyline>
															</svg>
														</span>
														<span><?php esc_html_e( ' if checked Wholesaler role will be assigned on the respective wholesale subscription.', 'woocommerce-wholesale-pricing' ); ?></span>
													</label>
												</td>
											</tr>
											<?php
											$args     = array(
												'post_type'     => array( 'product' ),
												'post_status' => 'publish',
												'posts_per_page' => -1,
												'tax_query' => array(
													array(
														'taxonomy' => 'product_type',
														'field'    => 'slug',
														'terms'    => 'variable-subscription',
													),
												),
											);
											$products = get_posts( $args );
											?>
											<tr scope="row">
												<th><label for=""><?php esc_html_e( 'Select Variable Subscription', 'woocommerce-wholesale-pricing' ); ?></label></th>
												<td>
													<label for="wholesale_subscription">
														<select id="wholesale_subscription" class="" name="options[wholesale_subscription]">
														<option disabled><?php esc_html_e( 'Select variable', 'woocommerce-wholesale-pricing' ); ?></option> 
														<?php
														if ( ! empty( $products ) ) {
															foreach ( $products as $key => $product ) {
																$selected = ( isset( $settings['wholesale_subscription'] ) && $product->ID == $settings['wholesale_subscription'] ) ? 'selected' : '';
																echo '<option value="' . esc_attr( $product->ID ) . '" ' . esc_attr( $selected ) . ' >' . esc_html( $product->post_title ) . '</option>';
															}
														}
														?>
														</select>
														<p><?php esc_html_e( 'Select the variable subscription product.', 'woocommerce-wholesale-pricing' ); ?></p>
													</label>
												</td>
											</tr>
											<?php
										} else {
											echo esc_html__( 'You need to enable Woocommerce Subscription plugin.', 'woocommerce-wholesale-pricing' );
										}
										?>
									</tbody>
								</table>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section2">
								<?php
								$roles = get_terms(
									'wholesale_user_roles',
									array(
										'hide_empty' => false,
									)
								);
								if ( ! empty( $roles ) ) {
									?>
									<div id="accordion">
										<?php
										$data = get_option( 'wholesale_multi_user_pricing' );
										foreach ( $roles as $key => $role ) :
											$min      = 1;
											$price    = '';
											$discount = '';
											if ( isset( $settings['wholesale_role'] ) && 'single' == $settings['wholesale_role'] && 'default_wholesaler' != $role->slug ) {
												continue;
											}
											if ( isset( $data[ $role->term_id ] ) ) {
												$min      = $data[ $role->term_id ]['min_quatity'];
												$price    = $data[ $role->term_id ]['wholesale_price'];
												$discount = $data[ $role->term_id ]['discount_type'];
											}
											?>
											<div class="card">
												<button onclick="return false;" class="btn btn-link collapsed" data-toggle="collapse" data-target="#collapse_<?php esc_attr_e( $role->term_id ); ?>" aria-expanded="false" aria-controls="collapse_<?php esc_attr_e( $role->term_id ); ?>">
													<?php esc_html_e( $role->name ); ?>
													<div class="wwp_signal">
														<?php
														$wwp_on_active  = '';
														$wwp_off_active = '';
														if ( isset( $data[ $role->term_id ] ) && ! empty( $data[ $role->term_id ] ) ) {
															$wwp_on_active = 'active';
														} else {
															$wwp_off_active = 'active';
														}

														?>
														<div class="wwp_circle wwp_circle_off <?php echo esc_attr( @$wwp_off_active ); ?> ">&nbsp;</div>
														<div class="wwp_circle wwp_circle_on <?php echo esc_attr( @$wwp_on_active ); ?> ">&nbsp;</div>
													</div>
												</button>
												<div id="collapse_<?php esc_attr_e( $role->term_id ); ?>" class="collapse" aria-labelledby="heading_<?php esc_attr_e( $role->term_id ); ?>" data-parent="#accordion" style="">
													<div class="card-body">
														<table class="form-table wwp-main-settings">
															<tbody>
																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Role Activation', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<input class="inp-cbx wwp-checbox" style="display: none" type="checkbox" value="<?php esc_attr_e( $role->slug ); ?>" id="role_<?php esc_attr_e( $role->term_id ); ?>" name="role_<?php esc_attr_e( $role->term_id ); ?>" <?php echo isset( $data[ $role->term_id ] ) ? 'checked' : ''; ?> >
																		<label class="cbx cbx-square" for="role_<?php esc_attr_e( $role->term_id ); ?>">
																			<span>
																				<svg width="12px" height="9px" viewbox="0 0 12 9">
																					<polyline points="1 5 4 8 11 1"></polyline>
																				</svg>
																			</span>
																			<span><?php esc_html_e( 'Enable Role', 'woocommerce-wholesale-pricing' ); ?></span>
																		</label>
																	</td>
																</tr>

																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Discount Type', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<select class="regular-text" name="discount_type_<?php esc_attr_e( $role->term_id ); ?>" value="">
																			<option value="percent" <?php selected( $discount, 'percent' ); ?> > <?php esc_html_e( 'Percent', 'woocommerce-wholesale-pricing' ); ?> </option>
																			<option value="fixed"  <?php selected( $discount, 'fixed' ); ?> > <?php esc_html_e( 'Fixed', 'woocommerce-wholesale-pricing' ); ?> </option>
																		</select>
																		<span data-tip="Price type for wholesale products" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																	</td>
																</tr>

																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Wholesale Value', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<input class="regular-text wwp-price" type="text" name="wholesale_price_<?php esc_attr_e( $role->term_id ); ?>" value="<?php esc_attr_e( $price ); ?>">
																		<span data-tip="Enter the value you would like to change the Wholesale User" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																	</td>
																</tr>

																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Min Quantity', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<input class="regular-text " type="text" name="min_quatity_<?php esc_attr_e( $role->term_id ); ?>" value="<?php esc_attr_e( $min ); ?>">
																		<span data-tip="Enter Wholesale minimum quantity to apply discount" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																	</td>
																</tr>
																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Global Tier Pricing', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<button name="save-wwp_wholesale" data-toggle="modal" data-target="#global_tier_pricing_Modal<?php esc_attr_e( $role->term_id ); ?>" class="wwp-button-primary" type="button" value="Save changes">Add Tier Pricing</button>
																		<span data-tip="Global Tier Pricing" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																		<?php
																		if (!isset($settings['tier_pricing'])) {
																			$settings['tier_pricing'] = '';
																		}
																		
																		$name = 'options[tier_pricing]';
																		echo wp_kses_post( tier_pricing_modal_popup( 'Global Tier Pricing', 'global_tier_pricing_Modal' . $role->term_id, $role->term_id, $settings['tier_pricing'], $name, '' ) );

																		?>
																	 
																	</td>
																</tr>
															</tbody>
														</table>
													</div>
												</div>
											</div>
											<?php
										endforeach;
										?>
									</div>
									<?php
								}
								?>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section3">
								<table class="form-table wwp-main-settings">
									<tbody>
										<tr scope="row">
											<th><label for="retailer_label"><?php esc_html_e( 'Retailer Price Label', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input type="text" class="regular-text" name="options[retailer_label]" id="retailer_label" value="<?php echo isset( $settings['retailer_label'] ) ? esc_html( $settings['retailer_label'] ) : ''; ?>"> 
												
												<input class="inp-cbx" style="display: none" id="retailer_disabled" type="checkbox" value="yes" name="options[retailer_disabled]" <?php echo ( isset( $settings['retailer_disabled'] ) && 'yes' == $settings['retailer_disabled'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="retailer_disabled">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span for="retailer_disabled"><?php esc_html_e( 'Label Hide', 'woocommerce-wholesale-pricing' ); ?></span>												
												</label>
												<span data-tip="Hide price Label for wholesale user only." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											
											</td>
										</tr>
										<tr scope="row">
											<th><label for="wholesaler_price_label"><?php esc_html_e( 'Wholesaler Price Label', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td><input type="text" class="regular-text" name="options[wholesaler_label]" id="wholesaler_price_label" value="<?php echo isset( $settings['wholesaler_label'] ) ? esc_html( $settings['wholesaler_label'] ) : ''; ?>">
											</td>
										</tr>
										<tr scope="row">
											<th><label for="save_price_label"><?php esc_html_e( 'Save Price Label', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input type="text" class="regular-text" name="options[save_label]" id="save_price_label" value="<?php echo isset( $settings['save_label'] ) ? esc_html( $settings['save_label'] ) : ''; ?>">
												<input class="inp-cbx" style="display: none" id="save_price_disabled" type="checkbox" value="yes" name="options[save_price_disabled]" <?php echo ( isset( $settings['save_price_disabled'] ) && 'yes' == $settings['save_price_disabled'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="save_price_disabled">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span for="save_price_disabled"><?php esc_html_e( 'Label Hide', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
												<span data-tip="Hide price Label for wholesale user only." class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
									</tbody>
								</table>							
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section4">
								<table class="form-table wwp-main-settings">
									<tbody>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Hide price', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="price_hide" type="checkbox" value="yes" name="options[price_hide]" <?php echo ( isset( $settings['price_hide'] ) && 'yes' == $settings['price_hide'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="price_hide">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Hide retail prices until user gets logged in', 'woocommerce-wholesale-pricing' ); ?></span>												
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for="display_link_text"><?php esc_html_e( 'Label for login link', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td><input type="text" class="regular-text" name="options[display_link_text]" id="display_link_text" value="<?php echo isset( $settings['display_link_text'] ) ? esc_html( $settings['display_link_text'] ) : ''; ?>">
											<span data-tip="This login link will appear on every product if Hide price option is checked" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Restrict wholesale store access', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="wholesaler_login_restriction" type="checkbox" value="yes" name="options[wholesaler_login_restriction]" <?php echo ( isset( $settings['wholesaler_login_restriction'] ) && 'yes' == $settings['wholesaler_login_restriction'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="wholesaler_login_restriction">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Enabling this option will allow only approved wholesale users to login.', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for="login_message_waiting_user"><?php esc_html_e( 'Custom message for pending request', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<?php
											if ( empty( $settings['login_message_waiting_user'] ) ) {
												$settings['login_message_waiting_user'] = __( 'You can not access this store, Your request status is in Pending' );
											}
											?>
											<td><input type="text" class="regular-text" name="options[login_message_waiting_user]" id="login_message_waiting_user" value="<?php echo isset( $settings['login_message_waiting_user'] ) ? esc_html( $settings['login_message_waiting_user'] ) : ''; ?>">
											<span data-tip="Enter message to display for pending request" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
										<tr scope="row">
											<th><label for="login_message_rejected_user"><?php esc_html_e( 'Custom message for rejected request', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<?php
											if ( empty( $settings['login_message_rejected_user'] ) ) {
												$settings['login_message_rejected_user'] = __( 'You can not access this store, Your request is Rejected by admin' );
											}
											?>
											<td><input type="text" class="regular-text" name="options[login_message_rejected_user]" id="login_message_rejected_user" value="<?php echo isset( $settings['login_message_rejected_user'] ) ? esc_html( $settings['login_message_rejected_user'] ) : ''; ?>">
											<span data-tip="Enter message to display for rejected request" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Restrict Full Store Access', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="restrict_store_access" type="checkbox" value="yes" name="options[restrict_store_access]" <?php echo ( isset( $settings['restrict_store_access'] ) && 'yes' == $settings['restrict_store_access'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="restrict_store_access">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Restrict or Hide your store with Password', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Override Default registration form', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>									
												<input class="inp-cbx" style="display: none" id="over_right_wholesale_form" type="checkbox" value="yes" name="options[over_right_wholesale_form]" <?php echo ( isset( $settings['over_right_wholesale_form'] ) && 'yes' == $settings['over_right_wholesale_form'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="over_right_wholesale_form">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Override Default registration form to wholesale form', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Restrict Store Access Message', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
											<?php $settings['restrict_store_access_message'] = isset( $settings['restrict_store_access_message'] ) ? $settings['restrict_store_access_message'] : ''; ?>
											<textarea id="restrict_store_access_message" rows="7" name="options[restrict_store_access_message]" class="textarea"><?php echo wp_kses_post( wp_unslash( $settings['restrict_store_access_message'] ) ); ?></textarea> 
											</td>
										</tr>
									</tbody>
								</table>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section5">
								<table class="form-table wwp-main-settings">
									<tbody>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Restrict wholesale products visibility', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="wholesaler_prodcut_only" type="checkbox" value="yes" name="options[wholesaler_prodcut_only]" <?php echo ( isset( $settings['wholesaler_prodcut_only'] ) && 'yes' == $settings['wholesaler_prodcut_only'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="wholesaler_prodcut_only">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'By enabling this option wholesale only products will be visible to wholesaler user roles only.', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Restrict wholesale products globally from non-wholesaler customer.', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="non_wholesale_product_hide" type="checkbox" value="yes" name="options[non_wholesale_product_hide]" <?php echo ( isset( $settings['non_wholesale_product_hide'] ) && 'yes' == $settings['non_wholesale_product_hide'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="non_wholesale_product_hide">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Enable this option to hide wholesale products from retailers and non-logged in user.', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Enforce minimum quantity rules', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="wholesaler_allow_minimum_qty" type="checkbox" value="yes" name="options[wholesaler_allow_minimum_qty]" <?php echo ( isset( $settings['wholesaler_allow_minimum_qty'] ) && 'yes' == $settings['wholesaler_allow_minimum_qty'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="wholesaler_allow_minimum_qty">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Enforce the wholesale customer to purchase with minimum quantity rules', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
									</tbody>
								</table>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section6">
								<table class="form-table wwp-main-settings">
									<tbody>									
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Enable Upgrade Tab', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="enable_upgrade" type="checkbox" value="yes" name="options[enable_upgrade]" <?php echo ( isset( $settings['enable_upgrade'] ) && 'yes' == $settings['enable_upgrade'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="enable_upgrade">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( ' Enable wholesale upgrade tab on my account page for non wholesale users', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Request Again Submit', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="request_again_submit" type="checkbox" value="yes" name="options[request_again_submit]" <?php echo ( isset( $settings['request_again_submit'] ) && 'yes' == $settings['request_again_submit'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="request_again_submit">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Ability to enable submitting request again after rejection.', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Upgrade Tab Text', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<label for="upgrade_tab_text">
													<input type="text" class="regular-text" name="options[upgrade_tab_text]" id="upgrade_tab_text" value="<?php echo isset( $settings['upgrade_tab_text'] ) ? esc_html( $settings['upgrade_tab_text'] ) : ''; ?>" Placeholder="Label for Upgrade to Wholesaler tab">
												</label>
												<span data-tip='Display any text you want on the "Upgrade to Wholesaler" tab.' class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
											</td>
										</tr>
									</tbody>
								</table>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section7">
								<table class="form-table wwp-main-settings">
									<tbody>
										<?php do_action( 'registration_page_css_after', $settings ); ?>
										<tr scope="row">
											<th colspan="2"><h2><?php esc_html_e( 'Additional CSS', 'woocommerce-wholesale-pricing' ); ?></h2></th>
										</tr>
										
										<?php
										if ( empty( $settings['wholesale_css'] ) ) {
											$settings['wholesale_css'] = '/* Enter Your Custom CSS Here */';
										}
										?>
										
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Registration Page CSS', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
											<textarea id="code_editor_page_css" rows="15" name="options[wholesale_css]" class="widefat-100 textarea"><?php echo wp_kses_post( wp_unslash( $settings['wholesale_css'] ) ); ?></textarea>
											<p class="wwwp_help_text"><?php esc_html_e( 'Enter css without <style> tag.', 'woocommerce-wholesale-pricing' ); ?></p>
											</td>
										</tr>
									</tbody>
								</table>
							</div>
							
							<div role="tabpanel" class="tab-pane fade" id="section8">
								
								<div class="wholesaleplugin_container" style="background:#fbfaff;">
									<div class="wholesaleplugin_content">
										<h1>Compatibility & Integrations</h1> </div>
								</div>
								<div class="wholesaleplugin_container">
									<div class="wholesaleplugin_content">
										<div class="wholesaleplugin_content_holder">
											<div class="col_left">
												<h3>Bulk order for WooCommerce </h3>
												<p>Bulk Order Form for WooCommerce is the perfect WooCommerce extension that gives you easy ways to list your products and, at the same time, allows you to customize your product tables without the need of an expert by your side.</p>
												<p>Create a simple, flexible, and responsive product table with Bulk Order for WooCommerce.</p>
												<ul>
													<li> Fully responsive design view; </li>
													<li> Sorting and filtering options; </li>
													<li> Multiple pagination options; </li>
													<li> Display extra columns such as SKU, weight, dimensions, etc.; </li>
													<li> Full product data control; </li>
													<li> List specific products based on category, tag, status custom field value, or date; </li>
													<li> Instant add to cart button; </li>
													<li> Customers can add multiple or bulk quantities to cart </li>
												</ul> <a href="https://woocommerce.com/products/bulk-order-form-for-woocommerce/" class="btn--view">Buy on WooCommerce</a> </div>
											<div class="col_right">
												<iframe width="470" height="315" src="https://www.youtube.com/embed/59JewNgiLLM" frameborder="0" allow="autoplay;" allowfullscreen></iframe>
											</div>
										</div>
									</div>
								</div>
								<div class="wholesaleplugin_container" style="background:#fbfaff;">
									<div class="wholesaleplugin_content">
										<div class="col_left">
											<iframe width="520" height="315" src="https://www.youtube.com/embed/-va24oCJ0LM" frameborder="0" allow=" autoplay; " allowfullscreen></iframe>
											</iframe>
										</div>
										<div class="col_right">
											<h3>Currency Switcher for WooCommerce</h3>
											<p>Currency Switcher for WooCommerce automatically detects the Geo-Location IP of your customer and the country from which they are browsing your store. When any page from your store loads, the extension displays all the prices in the home currency of the customer. </p>
											<p>The easy-to-use extension can be widgetized anywhere on the store from the shop and cart to the checkout page of your WooCommerce website.</p>
											<ul>
												<li> Expand your business to other countries & regions; </li>
												<li> Removes the hassle of currency conversion; </li>
												<li> Reduce Cart Abandonment, Refunds, Chargebacks; </li>
												<li> Ensure pricing display consistency across your store; </li>
												<li> Change currency automatically using customer’s GEOIP; </li>
												<li> Help customers avoid extra fee charges on their credit card statements. </li>
											</ul> <a href="https://woocommerce.com/products/currency-switcher-for-woocommerce/" class="btn--view">Buy on WooCommerce</a> </div>
									</div>
								</div>
							</div>
							
							
							
							
							
							
							
							
							
							
							
							
							
							<div role="tabpanel" class="tab-pane fade" id="section9">
								
							 
							<?php
								$roles = get_terms(
									'wholesale_user_roles',
									array(
										'hide_empty' => false,
									)
								);
							if ( ! empty( $roles ) ) {
								?>
									<div id="accordion">
									<?php
									$data = get_option( 'wholesale_multi_user_cart_discount' );
									$cart_tire_preces = get_option( 'cart_tire_preces' );
									foreach ( $roles as $key => $role ) :

										$min_wholesale_amount = '';
										$wholesale_amount     = '';
										$wholesale_lable      = '';
										$discount             = '';
										if ( isset( $settings['wholesale_role'] ) && 'single' == $settings['wholesale_role'] && 'default_wholesaler' != $role->slug ) {
											continue;
										}
										if ( isset( $data[ $role->term_id ] ) ) {
											if ( isset($data[ $role->term_id ]['wholesale_price']) ) {
												$price                = $data[ $role->term_id ]['wholesale_price'];
											}
											if ( isset($data[ $role->term_id ]['discount_type']) ) {
												$discount                = $data[ $role->term_id ]['discount_type'];
											}
										}
										?>
											<div class="card">
												<button onclick="return false;" class="btn btn-link collapsed" data-toggle="collapse" data-target="#collapse_<?php esc_attr_e( $role->term_id ); ?>_cart" aria-expanded="false" aria-controls="collapse_<?php esc_attr_e( $role->term_id ); ?>_cart">
												<?php esc_html_e( $role->name ); ?>
													<div class="wwp_signal">
													<?php
													$wwp_on_active  = '';
													$wwp_off_active = '';
													if ( isset( $data[ $role->term_id ] ) && ! empty( $data[ $role->term_id ] ) ) {
														$wwp_on_active = 'active';
													} else {
														$wwp_off_active = 'active';
													}
													?>
														<div class="wwp_circle wwp_circle_off <?php echo esc_attr( @$wwp_off_active ); ?> ">&nbsp;</div>
														<div class="wwp_circle wwp_circle_on <?php echo esc_attr( @$wwp_on_active ); ?> ">&nbsp;</div>
													</div>
												</button>
												<div id="collapse_<?php esc_attr_e( $role->term_id ); ?>_cart" class="collapse" aria-labelledby="heading_<?php esc_attr_e( $role->term_id ); ?>" data-parent="#accordion" style="">
													<div class="card-body">
													<table class="form-table wwp-main-settings">
															<tbody>
																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Role Activation', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<input class="inp-cbx wwp-checbox" style="display: none" type="checkbox" value="<?php esc_attr_e( $role->slug ); ?>" id="role_<?php esc_attr_e( $role->term_id ); ?>_cart" name="role_<?php esc_attr_e( $role->term_id ); ?>_cart" <?php echo isset( $data[ $role->term_id ] ) ? 'checked' : ''; ?> >
																		<label class="cbx cbx-square" for="role_<?php esc_attr_e( $role->term_id ); ?>_cart">
																			<span>
																				<svg width="12px" height="9px" viewbox="0 0 12 9">
																					<polyline points="1 5 4 8 11 1"></polyline>
																				</svg>
																			</span>
																			<span><?php esc_html_e( 'Enable Role', 'woocommerce-wholesale-pricing' ); ?></span>
																		</label>
																	</td>
																</tr>

																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Discount Type', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<select class="regular-text" name="discount_type_<?php esc_attr_e( $role->term_id ); ?>_cart" value="">
																			<option value="percent" <?php selected( $discount, 'percent' ); ?> > <?php esc_html_e( 'Percent', 'woocommerce-wholesale-pricing' ); ?> </option>
																			<option value="fixed"  <?php selected( $discount, 'fixed' ); ?> > <?php esc_html_e( 'Fixed', 'woocommerce-wholesale-pricing' ); ?> </option>
																		</select>
																		<span data-tip="Price type for cart discount" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																	</td>
																</tr>
																<tr scope="row">
																	<th>
																		<label for=""><?php esc_html_e( 'Cart Total Discount Range', 'woocommerce-wholesale-pricing' ); ?></label>
																	</th>
																	<td>
																		<button name="cart_total_discount_range" data-toggle="modal" data-target="#cart_total_discount_range<?php esc_attr_e( $role->term_id ); ?>" class="wwp-button-primary" type="button" value="Save changes">Add Discount</button>
																		<span data-tip="Cart Total Discount Range" class="data-tip-top"><span class="woocommerce-help-tip"></span></span>
																		<div class="cart_total_discount_range_wrap">
																		<?php
																		
																		$name = 'wholesale_multi_user_cart_discount[tier_pricing]';
																		if ( !isset( $cart_tire_preces['tier_pricing'] ) ) {
																			$cart_tire_preces['tier_pricing'] = '';
																		}
																		echo wp_kses_post( tier_pricing_modal_popup( 'Cart Total Discount Range', 'cart_total_discount_range' . $role->term_id, $role->term_id, $cart_tire_preces['tier_pricing'], $name, '' ) );
																		?>
																		
																		</div>
																	</td>
																</tr>
															</tbody>
														</table>
													
													</div>
												</div>
											</div>
											<?php
										endforeach;
									?>
									</div>
									<?php
							}
							?>
							</div>
								<div role="tabpanel" class="tab-pane fade" id="section10">
								<table class="form-table wwp-main-settings">
									<tbody>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Payment Method', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td>
												<input class="inp-cbx" style="display: none" id="enable_custom_payment_method" type="checkbox" value="yes" name="options[enable_custom_payment_method]" <?php echo ( isset( $settings['enable_custom_payment_method'] ) && 'yes' == $settings['enable_custom_payment_method'] ) ? 'checked' : ''; ?>>
												<label class="cbx cbx-square" for="enable_custom_payment_method">
													<span>
														<svg width="12px" height="9px" viewbox="0 0 12 9">
															<polyline points="1 5 4 8 11 1"></polyline>
														</svg>
													</span>
													<span><?php esc_html_e( 'Enable Custom Payment Method', 'woocommerce-wholesale-pricing' ); ?></span>
												</label>
											</td>
										</tr>
										<tr scope="row">
											<th><label for=""><?php esc_html_e( 'Payment Method Name', 'woocommerce-wholesale-pricing' ); ?></label></th>
											<td >
											<div id="payment_method_data">
												<?php 
												if (isset($settings['payment_method_name']) && !empty($settings['payment_method_name'])) {
													foreach ( (array) $settings['payment_method_name'] as $key => $payment_method_name_value) { 
														if ( !empty($payment_method_name_value) ) {	
															?>
																<div class="panel_payment_method_name">
																	<label for="payment_method_name">
																		<input type="text" class="regular-text payment_method_name" readonly name="options[payment_method_name][<?php esc_attr_e( $key); ?>]" 
																		value="<?php esc_attr_e( $key); ?>" 
																		Placeholder="Enter Payment Method Name">
																	</label>
																	<span><a class="button removebtn">Remove</a></span>
																	<span><a class="button editbtn wwpeditbtn">Edit</a></span>
																</div>
															<?php
														}
													} 
												}
												?>
											</div>	 
												<div> <a class="button add_row_payment_method_name" >Add Method</a> </div>
											</td>
										</tr>
									 
									</tbody>
								</table>
					
								<script>
								
									jQuery(document).ready(function(){
										jQuery(".add_row_payment_method_name").click(function() {
											template  =	'<div class="panel_payment_method_name " ><label for="payment_method_name">';
											template  += '<input type="text" class="regular-text payment_method_name"  name="options[payment_method_name][]" value="" Placeholder="Enter Payment Method Name">';
											template  += '</label> <span><a class="button removebtn">Remove</a></span></div>';
											console.log(template);
											jQuery("#payment_method_data").append(template);
										});
										jQuery(document).on( 'click','.button.removebtn', function (e) {
											jQuery(this).closest(".panel_payment_method_name").remove();
										});
										
										jQuery(document).on( 'click','.button.wwpeditbtn', function (e) { 
											alert('Editing the title of the gateway will result in disconnection of wholesale roles from this gateway.');
											console.log(jQuery(this).closest(".panel_payment_method_name").find('.payment_method_name'));
											jQuery(this).closest(".panel_payment_method_name").find('.payment_method_name').removeAttr('readonly');
										});
									});
								</script>
							</div>
						</div>
					 
				<div class="sep20px">&nbsp;</div>
				<div class="main-save-settings">
				<button name="save-wwp_wholesale" class="wwp-button-primary" type="submit" value="Save changes"><?php esc_html_e( 'Save changes', 'woocommerce-wholesale-pricing' ); ?></button>
				</div>
				<div class="sep20px">&nbsp;</div>
					</div>

				</div>	
				</div>
				
			
			</form>
			<?php
		}
		public static function wwp_wholesale_notifications_callback() {
			?>
			<div class="wrap">
				<form method="post" action="options.php">
					<?php settings_errors(); ?>
					<?php settings_fields( 'wwp_wholesale_request_notifications' ); ?>
					<?php do_settings_sections( 'wwp_wholesale_request_notifications' ); ?>

					<div class="tab" role="tabpanel">
						<!-- Nav tabs -->
						<ul id="blkTabs" class="nav nav-tabs" role="tablist">
							<li role="presentation">
								<a href="#section_notification1" role="tab" data-toggle="tab">New Request</a>
							</li>
							<li role="presentation">
								<a href="#section_notification2" role="tab" data-toggle="tab">Request Rejection</a>
							</li>
							<li role="presentation">
								<a href="#section_notification3" role="tab" data-toggle="tab">Request Approval</a>
							</li>
							<li role="presentation">
								<a href="#section_notification4" role="tab" data-toggle="tab">Subscription Upgrade</a>
							</li>
						</ul>

						<!-- Tab panes -->
						<div class="tab-content tabs">
							<div role="tabpanel" class="tab-pane fade" id="section_notification1">
								<div id="accordion_notification">
									<!-- NEW REQUEST ADMIN NOTIFICATION -->
									<div class="card">
										<?php $value = get_option( 'wwp_wholesale_admin_request_notification' ); ?>										
										<button onclick="return false;" class="btn btn-link collapsed" data-toggle="collapse" data-target="#collapse_admin_notify" aria-expanded="false" aria-controls="collapse_admin_notify">
											Admin Notification
											<div class="wwp_signal">
											<?php
											if ( 'yes' === $value ) {
												?>
												<div class="wwp_circle wwp_circle_off">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on active">&nbsp;</div>
												<?php
											} else {
												?>
												<div class="wwp_circle wwp_circle_off active">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on">&nbsp;</div>
												<?php
											}
											?>
											</div>
										</button>										
										<div id="collapse_admin_notify" class="collapse" data-parent="#accordion_notification" >
											<div class="card-body">
												<table class="form-table wwp-main-settings">
													<tr>
														<th>
															<label for="wwp_wholesale_admin_request_notification"><?php esc_html_e( 'Role Request Notification', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<input class="inp-cbx" style="display: none" type="checkbox" name="wwp_wholesale_admin_request_notification" value="yes" id="wwp_wholesale_admin_request_notification" <?php echo checked( 'yes', $value ); ?>>
															<label class="cbx cbx-square" for="wwp_wholesale_admin_request_notification">
																<span>
																	<svg width="12px" height="9px" viewbox="0 0 12 9">
																		<polyline points="1 5 4 8 11 1"></polyline>
																	</svg>
																</span>
																<span><?php esc_html_e( 'When checked, an Email will be sent to admin about the new requested User Role.', 'woocommerce-wholesale-pricing' ); ?></span>
															</label>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_admin_request_subject"><?php esc_html_e( 'Email Subject', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php $value = get_option( 'wwp_wholesale_admin_request_subject' ); ?>
															<input type="text" name="wwp_wholesale_admin_request_subject" id="wwp_wholesale_admin_request_subject" value="<?php echo esc_attr( $value ); ?>" class="widefat-100 regular-text"/>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_admin_request_body"><?php esc_html_e( 'Message', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php
																$content = html_entity_decode( get_option( 'wwp_wholesale_admin_request_body' ) );
																echo wp_kses_post(
																	wp_editor(
																		$content,
																		'wwp_wholesale_admin_request_body',
																		array( 'textarea_rows' => 3 )
																	)
																);
															?>
															<p><?php esc_html_e( 'Email body for the new requested user role. Use {first_name}, {last_name}, {username}, {email}, {date}, {time} tag in body to get user email.', 'woocommerce-wholesale-pricing' ); ?></p>
														</td>
													</tr>
												</table>
											</div>
										</div>
									</div>


									<!-- NEW USER REQUEST USER NOTIFICATION-->
									<div class="card">
										<?php $value = get_option( 'wwp_wholesale_user_registration_notification' ); ?>
										<button onclick="return false;" class="btn btn-link collapsed" data-toggle="collapse" data-target="#collapse_user_notify" aria-expanded="false" aria-controls="collapse_user_notify">
											User Notification
											<div class="wwp_signal">
											<?php
											if ( 'yes' === $value ) {
												?>
												<div class="wwp_circle wwp_circle_off">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on active">&nbsp;</div>
												<?php
											} else {
												?>
												<div class="wwp_circle wwp_circle_off active">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on">&nbsp;</div>
												<?php
											}
											?>
											</div>
										</button>										
										<div id="collapse_user_notify" class="collapse" data-parent="#accordion_notification" >
											<div class="card-body">
												<table class="form-table wwp-main-settings">
												<tr>
													<th>
														<label for="wwp_wholesale_user_registration_notification"><?php esc_html_e( 'Registration Notification', 'woocommerce-wholesale-pricing' ); ?></label>
													</th>
													<td>														
														<input class="inp-cbx" style="display: none" type="checkbox" name="wwp_wholesale_user_registration_notification" value="yes" id="wwp_wholesale_user_registration_notification" <?php echo checked( 'yes', $value ); ?>>
														<label class="cbx cbx-square" for="wwp_wholesale_user_registration_notification">
															<span>
																<svg width="12px" height="9px" viewbox="0 0 12 9">
																	<polyline points="1 5 4 8 11 1"></polyline>
																</svg>
															</span>
															<span><?php esc_html_e( 'When checked, an Email will be sent to user registration requested	.', 'woocommerce-wholesale-pricing' ); ?></span>
														</label>
													</td>
												</tr>
												<tr>
													<th>
														<label for="wwp_wholesale_registration_notification_subject"><?php esc_html_e( 'Email Subject', 'woocommerce-wholesale-pricing' ); ?></label>
													</th>
													<td>
														<?php $value = get_option( 'wwp_wholesale_registration_notification_subject' ); ?>
														<input type="text" name="wwp_wholesale_registration_notification_subject" id="wwp_wholesale_registration_notification_subject" value="<?php echo esc_attr( $value ); ?>" class="widefat-100 regular-text"/>
													</td>
												</tr>
												<tr>
													<th>
														<label for="wwp_wholesale_registration_notification_body"><?php esc_html_e( 'Message', 'woocommerce-wholesale-pricing' ); ?></label>
													</th>
													<td>
														<?php
															$content = html_entity_decode( get_option( 'wwp_wholesale_registration_notification_body' ) );
															echo wp_kses_post(
																wp_editor(
																	$content,
																	'wwp_wholesale_registration_notification_body',
																	array( 'textarea_rows' => 3 )
																)
															);
														?>
														<p><?php esc_html_e( 'Email body for the new registration user role. Use {first_name}, {last_name}, {username}, {email}, {date}, {time} tag in body to get user email.', 'woocommerce-wholesale-pricing' ); ?></p>
													</td>
												</tr>
												</table>
											</div>
										</div>
									</div>
								</div>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section_notification2">
								<div id="accordion_notification_rejected">
									<!-- USER REJECTION NOTIFICATION -->
									<div class="card">
										<?php $value = get_option( 'wwp_wholesale_user_rejection_notification' ); ?>
										<button onclick="return false;" class="btn btn-link collapsed" data-toggle="collapse" data-target="#collapse_user_notify_rejected" aria-expanded="true" aria-controls="collapse_user_notify_rejected">
											Request Rejection Notification
											<div class="wwp_signal">
											<?php
											if ( 'yes' === $value ) {
												?>
												<div class="wwp_circle wwp_circle_off">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on active">&nbsp;</div>
												<?php
											} else {
												?>
												<div class="wwp_circle wwp_circle_off active">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on">&nbsp;</div>
												<?php
											}
											?>
											</div>
										</button>										
										<div id="collapse_user_notify_rejected" class="collapse show" data-parent="#accordion_notification_rejected" >
											<div class="card-body">
												<table class="form-table wwp-main-settings">
													<tr>
														<th>
															<label for="wwp_wholesale_user_rejection_notification"><?php esc_html_e( 'Rejection Notification', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>														
															<input class="inp-cbx" style="display: none" type="checkbox" name="wwp_wholesale_user_rejection_notification" value="yes" id="wwp_wholesale_user_rejection_notification" <?php echo checked( 'yes', $value ); ?>>
															<label class="cbx cbx-square" for="wwp_wholesale_user_rejection_notification">
																<span>
																	<svg width="12px" height="9px" viewbox="0 0 12 9">
																		<polyline points="1 5 4 8 11 1"></polyline>
																	</svg>
																</span>
																<span><?php esc_html_e( 'When checked, an Email will be sent to user Rejection requested.', 'woocommerce-wholesale-pricing' ); ?></span>
															</label>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_rejection_notification_subject"><?php esc_html_e( 'Email Subject', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php $value = get_option( 'wwp_wholesale_rejection_notification_subject' ); ?>
															<input type="text" name="wwp_wholesale_rejection_notification_subject" id="wwp_wholesale_rejection_notification_subject" value="<?php echo esc_attr( $value ); ?>" class="widefat-100 regular-text"/>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_rejection_notification_body"><?php esc_html_e( 'Message', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php
																$content = html_entity_decode( get_option( 'wwp_wholesale_rejection_notification_body' ) );
																echo wp_kses_post(
																	wp_editor(
																		$content,
																		'wwp_wholesale_rejection_notification_body',
																		array( 'textarea_rows' => 3 )
																	)
																);
															?>
															<p><?php esc_html_e( 'Email body for the new rejection user role. Use {first_name}, {last_name}, {username}, {email}, {date}, {time} tag in body to get user email.', 'woocommerce-wholesale-pricing' ); ?></p>
														</td>
													</tr>
												</table>
											</div>
										</div>
									</div>
								</div>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section_notification3">
								<div id="accordion_notification_approved">
									<!-- USER REJECTION NOTIFICATION -->
									<div class="card">
										<?php $value = get_option( 'wwp_wholesale_request_approve_notification' ); ?>
										<button onclick="return false;" class="btn btn-link" data-toggle="collapse" data-target="#collapse_user_notify_approved" aria-expanded="true" aria-controls="collapse_user_notify_approved">
											Request Approval Notification
											<div class="wwp_signal">
											<?php
											if ( 'yes' === $value ) {
												?>
												<div class="wwp_circle wwp_circle_off">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on active">&nbsp;</div>
												<?php
											} else {
												?>
												<div class="wwp_circle wwp_circle_off active">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on">&nbsp;</div>
												<?php
											}
											?>
											</div>
										</button>										
										<div id="collapse_user_notify_approved" class="collapse show" data-parent="#accordion_notification_approved" >
											<div class="card-body">
												<table class="form-table wwp-main-settings">
													<tr>
														<th>
															<label for="wwp_wholesale_request_approve_notification"><?php esc_html_e( 'Request Approval Email', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>																													
															<input class="inp-cbx" style="display: none" type="checkbox" name="wwp_wholesale_request_approve_notification" value="yes" id="wwp_wholesale_request_approve_notification" <?php echo checked( 'yes', $value ); ?>>
															<label class="cbx cbx-square" for="wwp_wholesale_request_approve_notification">
																<span>
																	<svg width="12px" height="9px" viewbox="0 0 12 9">
																		<polyline points="1 5 4 8 11 1"></polyline>
																	</svg>
																</span>
																<span><?php esc_html_e( 'When checked, an Email will be sent to user about the approval of their requested User Role.', 'woocommerce-wholesale-pricing' ); ?></span>
															</label>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_email_request_subject"><?php esc_html_e( 'Email Subject', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php $value = get_option( 'wwp_wholesale_email_request_subject' ); ?>
															<input type="text" name="wwp_wholesale_email_request_subject" id="wwp_wholesale_email_request_subject" value="<?php echo esc_attr( $value ); ?>" class="widefat-100 regular-text"/>
														</td>
													</tr>
													<tr>
														<th>
															<label for="wwp_wholesale_email_request_body"><?php esc_html_e( 'Message', 'woocommerce-wholesale-pricing' ); ?></label>
														</th>
														<td>
															<?php
																$content = html_entity_decode( get_option( 'wwp_wholesale_email_request_body' ) );
																echo wp_kses_post(
																	wp_editor(
																		$content,
																		'wwp_wholesale_email_request_body',
																		array( 'textarea_rows' => 3 )
																	)
																);
															?>
															<p><?php esc_html_e( 'Email body for the approval of User Role request. Use {first_name}, {last_name}, {username}, {email}, {date}, {time} tag in body to get user email.', 'woocommerce-wholesale-pricing' ); ?></p>
														</td>
													</tr>
												</table>
											</div>
										</div>
									</div>
								</div>
							</div>

							<div role="tabpanel" class="tab-pane fade" id="section_notification4">
								<div id="accordion_notification_subscription">
									<!-- USER REJECTION NOTIFICATION -->
									<div class="card">
										<?php $value = get_option( 'wwp_wholesale_subscription_role_notification' ); ?>
										<button onclick="return false;" class="btn btn-link" data-toggle="collapse" data-target="#collapse_subscription_notification" aria-expanded="true" aria-controls="collapse_subscription_notification">
											User Role Upgrade Notification
											<div class="wwp_signal">
											<?php
											if ( 'yes' === $value ) {
												?>
												<div class="wwp_circle wwp_circle_off">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on active">&nbsp;</div>
												<?php
											} else {
												?>
												<div class="wwp_circle wwp_circle_off active">&nbsp;</div>
												<div class="wwp_circle wwp_circle_on">&nbsp;</div>
												<?php
											}
											?>
											</div>
										</button>										
										<div id="collapse_subscription_notification" class="collapse show" data-parent="#accordion_notification_subscription" >
											<div class="card-body">
												<table class="form-table wwp-main-settings">
													<!-- version 1.3.0 -->
													<?php
													if ( in_array( 'woocommerce-subscriptions/woocommerce-subscriptions.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
														?>
														<tr>
															<th>
																<label for="wwp_wholesale_subscription_role_notification"><?php esc_html_e( 'Enable Role Upgrade Notification', 'woocommerce-wholesale-pricing' ); ?></label>
															</th>
															<td>
																<input class="inp-cbx" style="display: none" type="checkbox" name="wwp_wholesale_subscription_role_notification" value="yes" id="wwp_wholesale_subscription_role_notification" <?php echo checked( 'yes', $value ); ?>>
																<label class="cbx cbx-square" for="wwp_wholesale_subscription_role_notification">
																	<span>
																		<svg width="12px" height="9px" viewbox="0 0 12 9">
																			<polyline points="1 5 4 8 11 1"></polyline>
																		</svg>
																	</span>
																	<span><?php esc_html_e( 'When checked, an Email will be sent to user on the role upgrade after subscription.', 'woocommerce-wholesale-pricing' ); ?></span>
																</label>
															</td>
														</tr>
														<tr>
															<th>
																<label for="wwp_wholesale_subscription_role_subject"><?php esc_html_e( 'Email Subject', 'woocommerce-wholesale-pricing' ); ?></label>
															</th>
															<td>
																<?php $value = get_option( 'wwp_wholesale_subscription_role_subject' ); ?>
																<input type="text" name="wwp_wholesale_subscription_role_subject" id="wwp_wholesale_subscription_role_subject" value="<?php echo esc_attr( $value ); ?>" class="widefat-100 regular-text"/>
															</td>
														</tr>
														<tr>
															<th>
																<label for="wwp_wholesale_subscription_role_body"><?php esc_html_e( 'Message', 'woocommerce-wholesale-pricing' ); ?></label>
															</th>
															<td>
																<?php
																	$content = html_entity_decode( get_option( 'wwp_wholesale_subscription_role_body' ) );
																	echo wp_kses_post(
																		wp_editor(
																			$content,
																			'wwp_wholesale_subscription_role_body',
																			array( 'textarea_rows' => 3 )
																		)
																	);
																?>
																<p><?php esc_html_e( 'Email body for the role upgrade after subscription. Use {first_name}, {last_name}, {username}, {date}, {time}, {email} & {role} tag in body to get user email.', 'woocommerce-wholesale-pricing' ); ?></p>
															</td>
														</tr>
														<?php
													} else {
														esc_html_e( 'You need to enable Woocommerce Subscription plugin.', 'woocommerce-wholesale-pricing' );
													}
													?>
													<!-- ends version 1.3.0 -->
												</table>
											</div>
											
										</div>
									</div>
								</div>
							</div>
						</div>
					</div>
					<?php /*submit_button();*/ ?>
					<div class="sep20px">&nbsp;</div>
					<input type="submit" name="submit" id="submit" class="wwp-button-primary" value="Save Changes">					
					<div class="sep20px">&nbsp;</div>
				</form>
			</div>
			<?php
		}
		/**
		 * Initialize product wholesale data tab
		 *
		 * @since   1.0
		 * @version 1.0
		 */
		public function wwp_add_wholesale_product_data_tab( $product_data_tabs ) {
			$product_data_tabs['wwp-wholesale-tab'] = array(
				'label'  => esc_html__( 'Wholesale', 'woocommerce-wholesale-pricing' ),
				'target' => 'wwp_wholesale_product_data',
			);
			return $product_data_tabs;
		}
		/**
		 * Initialize product wholesale data tab
		 *
		 * @since   1.0
		 * @version 1.0
		 */
		public function wcpp_custom_style() {
			?>
			<style>
				.wwp-wholesale-tab_tab a:before {
					font-family: Dashicons;
					content: "\f240" !important;
				}
			</style>
			<?php
		}
		/**
		 * Product wholesale data tab multi users
		 *
		 * @since   1.0
		 * @version 1.0
		 */
		public function wwp_add_wholesale_product_data_fields_multi() {
			// version 1.3.0
			global $post;
			$product_id = $post->ID;
			$roles      = array();
			$taxroles   = get_terms(
				'wholesale_user_roles',
				array(
					'hide_empty' => false,
				)
			);
			if ( ! empty( $taxroles ) ) {
				foreach ( $taxroles as $key => $role ) {
					$roles[ $role->slug ] = $role->name;
				}
			}
			?>
			<div id="wwp_wholesale_product_data" class="panel woocommerce_options_panel">
			<?php
			wp_nonce_field( 'wwp_product_wholesale_nonce', 'wwp_product_wholesale_nonce' );
			$wwp_icon_url = get_post_meta( $product_id, 'wwp_icon_url', true );
			$wwp_image_url = get_post_meta( $product_id, 'wwp_image_url', true );
			woocommerce_wp_checkbox(
				array(
					'id'            => '_wwp_hide_for_customer',
					'wrapper_class' => '_wwp_hide_for_customer',
					'label'         => esc_html__( 'Hide Product', 'woocommerce-wholesale-pricing' ),
					'description'   => esc_html__( 'Hide this product from users having customer role', 'woocommerce-wholesale-pricing' ),
				)
			);

			woocommerce_wp_checkbox(
				array(
					'id'            => '_wwp_hide_for_visitor',
					'wrapper_class' => '_wwp_hide_for_visitor',
					'label'         => esc_html__( 'Hide Product', 'woocommerce-wholesale-pricing' ),
					'description'   => esc_html__( 'Hide this product from visitors', 'woocommerce-wholesale-pricing' ),
				)
			);

			$value = get_post_meta( $product_id, 'wholesale_product_visibility_multi', true );
			woocommerce_wp_select(
				array(
					'id'                => 'wholesale_product_visibility_multi[]',
					'label'             => esc_html__( 'Hide Product for Wholesaler Roles', 'woocommerce-wholesale-pricing' ),
					'type'              => 'select',
					'class'             => 'wc-enhanced-select',
					'style'             => 'min-width: 50%;',
					'desc_tip'          => 'true',
					'description'       => esc_html__( 'Choose specific user roles to hide the product.', 'woocommerce-wholesale-pricing' ),
					'options'           => $roles,
					'value'             => $value,
					'custom_attributes' => array(
						'multiple' => 'multiple',
					),
				)
			); // ends version 1.3.0
			?>
				<div id="wwp_wholesale_product_data" class="panel woocommerce_options_panel">
					<div id="variable_product_options" class=" wc-metaboxes-wrapper" style="display: block;">
						<div id="variable_product_options_inner">
							<div id="message" class="inline notice woocommerce-message">
								<p><?php echo sprintf( '%1$s <strong>%2$s</strong> %3$s', esc_html__( 'For', 'woocommerce-wholesale-pricing' ), esc_html__( 'Multi-user wholesale roles', 'woocommerce-wholesale-pricing' ), esc_html__( 'manage price from wholesale metabox', 'woocommerce-wholesale-pricing' ) ); ?></p>
								<p><a class="button-primary" id="wholesale-pricing-pro-multiuser-move"><?php esc_html_e( 'Move', 'woocommerce-wholesale-pricing' ); ?></a></p>
							</div>
						</div>
					</div>
				</div>
		<h2 class=""><b>Select Attachment File</b></h2>	
		<p class="form-field">
			<label for="wwp_icon_url">Upload Icon</label>
			<input type="text" name="wwp_icon_url" id="wwp_icon_url" value="<?php echo esc_attr( $wwp_icon_url ); ?>" class="short regular-text">
			<span class="wwp_media_selection">
				<input type="button" name="www_upload_icon" id="www_upload_icon" class="button-secondary" value="Upload">
				<input type="button" id="www_remove_icon" class="button-secondary" value="Remove">
			</span>
		</p>
		<p class="form-field">
			<label for="wwp_image_url">Upload Attachment</label>
			<input type="text" name="wwp_image_url" id="wwp_image_url" value="<?php echo esc_attr( $wwp_image_url ); ?>"  class="short regular-text">
			<span class="wwp_media_selection">
				<input type="button" name="www_upload_attachment" id="www_upload_attachment" class="button-secondary" value="Upload">
				<input type="button" id="www_remove_attachment" class="button-secondary" value="Remove">
			</span>
		</p>
			<?php 
				woocommerce_wp_text_input(
					array(
						'id' => 'wwp_attachment_title',
						'placeholder' => 'Set Custom Title',
						'label' => __('Set Custom Title', 'woocommerce-wholesale-pricing'),
						'type' => 'text',
						'custom_attributes' => array(
							'step' => 'any',
							'min' => '0'
						)
					)
				);
				woocommerce_wp_text_input(
					array(
						'id' => 'wwp_attachment_text_link',
						'placeholder' => 'Text For Download Link',
						'label' => __('Text For Download Link', 'woocommerce-wholesale-pricing'),
						'type' => 'text',
						'custom_attributes' => array(
							'step' => 'any',
							'min' => '0'
						)
					)
				); 
			?>
		<h2 class=""><b>Current Attachment = <a onclick="wwp_open_attachment()" url="<?php echo esc_attr( $wwp_image_url ); ?>" class="wwp_download_link" target="_blank" >Download Here</a></b></h2>		
		</br></br>
		<script type="text/javascript">
			function wwp_open_attachment () {
				url = jQuery('.wwp_download_link').attr( 'url');
				
				if (url != '' && url  != undefined) {
					window.open(url);
				}
				return false;
				
				}
				jQuery(document).ready(function($) {
					$('#www_upload_icon').click(function(e) {
					e.preventDefault();
					var image = wp.media({ 
						title: 'Upload Icon',
						multiple: false
					}).open().on('select', function(e) {
						var uploaded_image = image.state().get('selection').first();
						var image_url = uploaded_image.toJSON().url;
						$('#wwp_icon_url').val(image_url);
					});
				});
			
				$('#www_upload_attachment').click(function(e) {
					e.preventDefault();
					var image = wp.media({ 
						title: 'Upload Attachment',
						multiple: false
					}).open().on('select', function(e) {
						var uploaded_image = image.state().get('selection').first();
						var image_url = uploaded_image.toJSON().url;
						$('#wwp_image_url').val(image_url);
						$('wwp_download_link').val(image_url);
						$('.wwp_download_link').attr( 'url', image_url );
						console.log(uploaded_image);
					});
				});
	
				$('#www_remove_attachment').click(function(e) {
					$('#wwp_image_url').val('');
					$('.wwp_download_link').attr( 'url', '' );
				});
				$('#www_remove_icon').click(function(e) {
					$('#wwp_icon_url').val('');
				});    
				$('#wwp_image_url').on('input',function(e){
					if ( this.value  ) {
						$('.wwp_download_link').attr( 'url', this.value );
					} else {
						$('.wwp_download_link').attr( 'url', '' );
					}
				});
			});
		</script>
			</div>
			<?php
		}

		public function wwp_screen_ids( $screen_ids ) {
			$custom     = array( 'edit-wholesale_user_roles' );
			$screen_ids = array_merge( $custom, $screen_ids );
			return $screen_ids;
		}
		public function wwp_admin_script_style() {
			wp_enqueue_style( 'jquery-ui-styles' );
			wp_enqueue_script( 'jquery-ui-core' );
			wp_enqueue_script( 'jquery-ui-autocomplete' );
			wp_enqueue_script( 'jquery-ui-sortable' );
			wp_enqueue_script( 'wwp-script', WWP_PLUGIN_URL . 'assets/js/admin-script.js', array( 'jquery' ), '1.0' );
			wp_enqueue_script( 'wwp-chart', WWP_PLUGIN_URL . 'assets/js/chart.js', array( 'jquery' ), '1.0');
			$settings = get_option( 'wwp_wholesale_pricing_options', true );
			unset( $settings['wholesale_css'] );
			wp_localize_script(
				'wwp-script',
				'wwpscript',
				array(
					'ajaxurl'                => admin_url( 'admin-ajax.php' ),
					'admin_url'              => admin_url(),
					'ajax_nonce'             => wp_create_nonce( 'wwp_wholesale_pricing' ),
					'wwp_wholesale_settings' => $settings,
				)
			);
			add_thickbox();

			wp_enqueue_style( 'wwp-style', WWP_PLUGIN_URL . 'assets/css/admin-style.css', array(), '1.0' );
			wp_enqueue_style( 'fontawesome', WWP_PLUGIN_URL . 'assets/css/font-awesome.min.css', array(), '1.0' );
			// wp_enqueue_script('wwp-popper-min-js', WWP_PLUGIN_URL . 'assets/js/popper.min.js', array( 'jquery' ), '1.0.0' );
			// wp_enqueue_script('wwp-bootstrap-js', WWP_PLUGIN_URL . 'assets/js/bootstrap.min.js', array( 'jquery' ), '4.5.3' );

			// wp_enqueue_script('wwp-bootstrap-select-js', WWP_PLUGIN_URL . 'assets/js/bootstrap-select.min.js', array( 'wwp-bootstrap-js' ), '1.13.14' );
			wp_enqueue_script( 'wwp-popper-min-js', WWP_PLUGIN_URL . 'assets/js/popper.min.js', array( 'jquery' ), '1.0.0' );
			wp_enqueue_script( 'wwp-bootstrap-js', WWP_PLUGIN_URL . 'assets/js/bootstrap.min.js', array( 'jquery' ), '4.5.3' );
			wp_enqueue_script( 'wwp-bootstrap-select-js', WWP_PLUGIN_URL . 'assets/js/bootstrap-select.min.js', array( 'wwp-bootstrap-js' ), '1.13.14' );
			global $post_type;
			// wp_die($post_type);

			if ( ( isset( $_GET['page'] ) && ( 'wwp_wholesale' == sanitize_text_field( $_GET['page'] ) || 'wwp_wholesale_settings' == sanitize_text_field( $_GET['page'] ) || 'wwp-registration-setting' == sanitize_text_field( $_GET['page'] ) || 'wwp_wholesale_notifcations' == sanitize_text_field( $_GET['page'] ) || 'wwp-bulk-wholesale-pricing' == sanitize_text_field( $_GET['page'] ) ) || ( ! isset( $_GET['post_type'] ) && 'wwp_requests' === $post_type ) ) ) {
				wp_enqueue_style( 'wwp-data-tip', WWP_PLUGIN_URL . 'assets/css/data-tip.min.css', array(), '1.0' );
				wp_enqueue_style( 'wwp-bootstrap', WWP_PLUGIN_URL . 'assets/css/bootstrap.min.css', array(), '4.5.3', 'all' );
				wp_enqueue_style( 'wwp-bootstrap-select', WWP_PLUGIN_URL . 'assets/css/bootstrap-select.min.css', array(), '1.13.14', 'all' );
				wp_enqueue_style( 'wwp-admin-new', WWP_PLUGIN_URL . 'assets/css/admin-style-new.css', array(), '1.5.0', 'all' );
			}
		}
	}
	new WWP_Wholesale_Pricing_Backend();
}
