<?php if ( !defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( !class_exists( 'WWLC_Dashboard_Widget' ) ) {

	class WWLC_Dashboard_Widget {

        /*
        |--------------------------------------------------------------------------
        | Class Properties
        |--------------------------------------------------------------------------
        */

		/**
         * Property that holds the single main instance of WWLC_Dashboard_Widget.
         *
         * @since 1.8.0
         * @access private
         * @var WWLC_Dashboard_Widget
         */
		private static $_instance;

        /*
        |--------------------------------------------------------------------------
        | Class Methods
        |--------------------------------------------------------------------------
        */

        /**
         * WWLC_Dashboard_Widget constructor.
         *
         * @param array $dependencies Array of instance objects of all dependencies of WWLC_Dashboard_Widget model.
         *
         * @since 1.8.0
         */
        public function __construct( $dependencies ) {}

		/**
		 * Singleton Pattern.
		 *
         * @param array $dependencies Array of instance objects of all dependencies of WWLC_Dashboard_Widget model.
         *
         * @return WWLC_Dashboard_Widget
		 * @since 1.8.0
         */
		public static function instance( $dependencies = null ) {

            if ( !self::$_instance instanceof self )
                self::$_instance = new self( $dependencies );

            return self::$_instance;

		}

		/**
		 * Creates new dashbpard widget.
		 *
		 * @since 1.8.0
         */
		public function wwlc_insert_dashboard_widget() {

			wp_add_dashboard_widget(
             	'wwlc_lead_capture_stats_widget',         			// Widget slug
           		'Lead Capture Statistics',        					// Title
             	array( $this , 'wwlc_lead_capture_statistics' ) 	// Display function
	        );	

		}

		/**
		 * WWLC statistics.
		 *
		 * @return array
		 * @since 1.8.0
         */
		public function calculate_lead_capture_statics() {

			global $wc_wholesale_prices;

			$users 				= get_users();
			$approved_users 	= array();
			$rejected_users 	= array();
			$pending_users 		= array();
			$active_users 		= array();
			$inactive_users 	= array();

			foreach ( $users as $user ) {

				$is_user_approved 	= get_user_meta( $user->ID , 'wwlc_approval_date' , true );
				$is_user_rejected 	= get_user_meta( $user->ID , 'wwlc_rejection_date' , true);
				$user_capabilities 	= get_user_meta( $user->ID, 'wp_capabilities' , true );

				if( !empty( $is_user_approved ) ) {

					// Approved Users
					array_push( $approved_users , $user->ID );

					// WWP / WWPP Custom Role
					if( wwlc_is_wwp_and_wwpp_active() ) {

						$custom_wwpp_roles 		= $wc_wholesale_prices->wwp_wholesale_roles->getAllRegisteredWholesaleRoles();
						$custom_wwpp_roles_arr 	= array();

						if( $custom_wwpp_roles ) {

							foreach ( $custom_wwpp_roles as $role => $data )
								$custom_wwpp_roles_arr[] = $role;

						}

						if( $user_capabilities ) {

							$user_roles = array();

							foreach ( $user_capabilities as $key => $value )
								$user_roles[] = $key;

							// Inactive Users
							if( in_array( 'wwlc_inactive' , $user_roles ) )
								array_push( $inactive_users , $user->ID );
							else {

								$has_wholesale_role = array_intersect( $user_roles , $custom_wwpp_roles_arr );
								$has_wholesale_role = count( $has_wholesale_role );

								// Active Users
								if( $has_wholesale_role > 0 )
									array_push( $active_users , $user->ID );
								// Inactive Users
								else
									array_push( $inactive_users , $user->ID );
								
							}

						}

					}

				}
				// Rejected Users
				else if( !empty( $is_user_rejected ) )
					array_push( $rejected_users , $user->ID );

				else if( !empty( $user_capabilities ) ) {

					foreach ( $user_capabilities as $key => $value ) {

						// Pending Users
						if( $key === 'wwlc_unapproved' ) {
							array_push( $pending_users , $user->ID );
							break;
						}

					}

				}

				// Totals
				$total_pending 			= count( $pending_users );
				$total_approved 		= count( $approved_users );
				$total_rejected 		= count( $rejected_users );
				$total_active 	 		= count( $active_users );
				$total_inactive 	 	= count( $inactive_users );
				$total_registrations 	= count( $pending_users ) + count( $approved_users ) + count( $rejected_users );

			}

			return array(
						'total_registrations' 	=> $total_registrations,
						'total_pending' 		=> $total_pending,
						'total_approved' 		=> $total_approved,
						'total_rejected'		=> $total_rejected,
						'total_active' 			=> $total_active,
						'total_inactive' 		=> $total_inactive
					);

		}

		/**
		 * Handles displaying data inside the dashboard widget.
		 *
		 * @since 1.8.0
         */
		public function wwlc_lead_capture_statistics() {

			$lead_capture_stats = $this->calculate_lead_capture_statics(); ?>

			<table class="wwlc_stats">
				<tr>
					<td><?php 
						_e( 'Total Registrations' , 'woocommerce-wholesale-lead-capture' ); 
						echo wc_help_tip( __( 'Total customers that have registered via the wholesale registration form, including any accounts that are now inactive or have had their user role changed.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
					</td>
					<td><b><?php echo $lead_capture_stats[ 'total_registrations' ]; ?></b></td>
				</tr>
				<tr>
					<td><?php 
						_e( 'Total Pending' , 'woocommerce-wholesale-lead-capture' ); 
						echo wc_help_tip( __( 'Total customers that registered via the wholesale registration form where their status is still pending.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
					</td>
					<td><b><?php echo $lead_capture_stats[ 'total_pending' ]; ?></b></td>
				</tr>
				<tr>
					<td><?php 
						_e( 'Total Approved' , 'woocommerce-wholesale-lead-capture' ); 
						echo wc_help_tip( __( 'Total customers that registered via the wholesale registration form that have been approved, regardless of subsequent user role changes.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
					</td>
					<td><b><?php echo $lead_capture_stats[ 'total_approved' ]; ?></b></td>
				</tr>
				<tr>
					<td><?php 
						_e( 'Total Rejected' , 'woocommerce-wholesale-lead-capture' ); 
						echo wc_help_tip( __( 'Total customers that registered via the wholesale registration form that have been rejected.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
					</td>
					<td><b><?php echo $lead_capture_stats[ 'total_rejected' ]; ?></b></td>
				</tr>

				<?php if( wwlc_is_wwp_and_wwpp_active() ) { ?>

					<tr>
						<td><?php 
							_e( 'Total Active Wholesale Customers' , 'woocommerce-wholesale-lead-capture' );
							echo wc_help_tip( __( 'Total customers that registered via the wholesale registration form that are still active and have a wholesale user role.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
						</td>
						<td><b><?php echo $lead_capture_stats[ 'total_active' ]; ?></b></td>
					</tr>
					<tr>
						<td><?php 
							_e( 'Total Inactive Wholesale Customers' , 'woocommerce-wholesale-lead-capture' ); 
							echo wc_help_tip( __( 'Total customers that registered via the wholesale registration form but are now inactive or have had their user role changed.' , 'woocommerce-wholesale-lead-capture' ) ); ?>:
						</td>
						<td><b><?php echo $lead_capture_stats[ 'total_inactive' ]; ?></b></td>
					</tr>

				<?php } ?>

			</table><?php

		}

	    /**
	     * Stats styles and Lead Capture Statistics Tooltip.
	     *
	     * @since 1.8.0
	     * @access public
	     */
		public function dashboard_widget_scripts_styles() {

			$screen = get_current_screen();

			if( $screen->id == 'dashboard' && $screen->base == 'dashboard' ) { ?>

				<style type="text/css">
					#wwlc_lead_capture_stats_widget table{ width:100%; }
					#wwlc_lead_capture_stats_widget table tr td{ padding: 5px; }
					#wwlc_lead_capture_stats_widget table tr td:first-child{ width: 260px; text-align: right; }
					#wwlc_lead_capture_stats_widget table tr td:last-child{ text-align: left; }
					.wwlc_stats .woocommerce-help-tip{ font-size: 18px; margin: 0px 4px; }
				</style>

				<script type="text/javascript">
		            // qtip tooltip
		            jQuery( document ).ready( function( $ ) {
			            $( '.woocommerce-help-tip' ).tipTip( {
							'attribute': 'data-tip',
							'fadeIn': 50,
							'fadeOut': 50,
							'delay': 200
						} );
					} );
				</script><?php

			}

		}

	    /**
	     * Execute model.
	     *
	     * @since 1.8.0
	     * @access public
	     */
	    public function run() {

	    	add_action( 'wp_dashboard_setup', array( $this , 'wwlc_insert_dashboard_widget' ) );
	    	add_action( 'admin_footer' , array( $this , 'dashboard_widget_scripts_styles' ) );

	    }
	}
}
