<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter for user menu links.
 *
 * @param array $actions
 * @param int   $user_profile_id
 *
 * @return array
 */
function um_verified_admin_user_actions( $actions, $user_profile_id ) {
	if ( UM()->roles()->um_current_user_can( 'edit', $user_profile_id ) ) {

		$status = UM()->Verified_Users_API()->api()->verified_status( $user_profile_id );

		if ( 'unverified' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_be_verified( $user_profile_id ) ) {
				$actions['verify'] = array( 'label' => __( 'Verify', 'um-verified' ) );
			}
		} elseif ( 'pending' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_approve_verification( $user_profile_id ) ) {
				$actions['approve_verification'] = array( 'label' => __( 'Approve verification request', 'um-verified' ) );
			}
			if ( UM()->Verified_Users_API()->api()->can_reject_verification( $user_profile_id ) ) {
				$actions['reject_verification'] = array( 'label' => __( 'Reject verification request', 'um-verified' ) );
			}
		} elseif ( 'verified' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_be_unverified( $user_profile_id ) ) {
				$actions['unverify'] = array( 'label' => __( 'Unverify', 'um-verified' ) );
			}
		}
	}
	return $actions;
}
add_filter( 'um_admin_user_actions_hook', 'um_verified_admin_user_actions', 100, 2 );

/**
 * Extend settings
 *
 * @param array $settings
 *
 * @return array
 */
function um_verified_users_settings( $settings ) {
	$settings_fields = apply_filters( 'um_verified_users_settings_fields', array() );

	if ( ! empty( $settings_fields ) ) {
		$settings['extensions']['sections']['verified'] = array(
			'title'  => __( 'Verified Users', 'um-verified' ),
			'fields' => $settings_fields,
		);
	}

	return $settings;
}
add_filter( 'um_settings_structure', 'um_verified_users_settings' );

/**
 * Add badge to display name
 *
 * @param $name
 * @param $user_id
 * @param $html
 *
 * @return string
 */
function um_verified_add_badge( $name, $user_id, $html ) {
	if ( ! $html ) {
		return $name;
	}

	if ( UM()->Verified_Users_API()->api()->is_verified( $user_id ) ) {
		$name = $name . UM()->Verified_Users_API()->api()->verified_badge();
	}

	return $name;
}
add_filter( 'um_user_display_name_filter', 'um_verified_add_badge', 50, 3 );

/**
 * New tag for activity
 *
 * @param $args
 *
 * @return array
 */
function um_verified_search_tpl( $args ) {
	$args[] = '{verified}';
	return $args;
}
add_filter( 'um_activity_search_tpl', 'um_verified_search_tpl' );

/**
 * New tag replace for activity
 *
 * @param $args
 * @param $array
 *
 * @return array
 */
function um_verified_replace_tpl( $args, $array ) {
	$args[] = isset( $array['verified'] ) ? $array['verified'] : '';
	return $args;
}
add_filter( 'um_activity_replace_tpl', 'um_verified_replace_tpl', 10, 2 );

/**
 * Add new activity action
 *
 * @param $actions
 *
 * @return mixed
 */
function um_verified_activity_action( $actions ) {
	$actions['verified-account'] = __( 'Account Verifications', 'um-verified' );
	return $actions;
}
add_filter( 'um_activity_global_actions', 'um_verified_activity_action' );

/**
 * Adds a notification type
 *
 * @param array $logs
 *
 * @return array
 */
function um_verified_add_notification_type( $logs ) {
	$logs['account_verified'] = array(
		'title'        => __( 'User account is verified', 'um-verified' ),
		'account_desc' => __( 'When my account gets verified', 'um-verified' ),
	);
	return $logs;
}
add_filter( 'um_notifications_core_log_types', 'um_verified_add_notification_type', 200 );

/**
 * Adds a notification icon
 *
 * @param $output
 * @param $type
 *
 * @return string
 */
function um_verified_add_notification_icon( $output, $type ) {
	if ( 'account_verified' === $type ) {
		$output = '<i class="um-icon-ios-checkmark" style="color: #5EA5E7"></i>';
	}

	return $output;
}
add_filter( 'um_notifications_get_icon', 'um_verified_add_notification_icon', 10, 2 );

/**
 * Adds a notification icon
 *
 * @param $fields
 * @param $role
 *
 * @return array
 */
function um_verified_profile_completeness_roles_metabox_fields( $fields, $role ) {
	$fields[] = array(
		'id'          => '_um_profilec_verify',
		'type'        => 'select',
		'label'       => __( 'Verify user account', 'um-verified' ),
		'tooltip'     => __( 'Verify the user\'s account when he/she completes his/her profile.', 'um-verified' ),
		'value'       => ! empty( $role['_um_profilec_verify'] ) ? $role['_um_profilec_verify'] : 0,
		'conditional' => array( '_um_profilec', '=', '1' ),
		'options'     => array(
			0 => __( 'No', 'um-verified' ),
			1 => __( 'Yes', 'um-verified' ),
		),
	);

	return $fields;
}
add_filter( 'um_profile_completeness_roles_metabox_fields', 'um_verified_profile_completeness_roles_metabox_fields', 10, 2 );



if ( UM()->is_new_ui() ) {
	/**
	 * Verify/unverify from backend profile
	 *
	 * @param string  $content
	 * @param WP_User $user
	 *
	 * @return string
	 */
	function um_verification_field( $content, $user ) {
		global $pagenow;
		if ( 'profile.php' === $pagenow ) {
			return $content;
		}

		if ( empty( $user ) ) {
			return $content;
		}

		if ( ! isset( $user->ID ) ) {
			return $content;
		}

		if ( current_user_can( 'edit_users' ) && current_user_can( 'edit_user', $user->ID ) ) {
			$is_verified = UM()->Verified_Users_API()->api()->is_verified( $user->ID );
			ob_start();
			?>
			<table class="form-table">
				<tbody>
				<tr>
					<th>
						<label for="um_set_verification"><?php esc_html_e( 'Account Verification', 'um-verified' ); ?></label>
					</th>
					<td>
						<select name="um_set_verification" id="um_set_verification">
							<option value="0" <?php selected( 0, $is_verified ); ?>><?php esc_html_e( 'Unverified Account', 'um-verified' ); ?></option>
							<option value="1" <?php selected( 1, $is_verified ); ?>><?php esc_html_e( 'Verified Account', 'um-verified' ); ?></option>
						</select>
					</td>
				</tr>
				</tbody>
			</table>
			<?php
		}

		$content .= ob_get_clean();
		return $content;
	}
	add_filter( 'um_user_profile_additional_fields', 'um_verification_field', 2, 2 );

	/**
	 * Verify/unverify from backend profile
	 *
	 * @param string $content
	 *
	 * @return string
	 */
	function um_user_new_form_verification_field( $content ) {
		global $pagenow;
		if ( 'profile.php' === $pagenow ) {
			return $content;
		}

		if ( current_user_can( 'edit_users' ) ) {
			ob_start();
			?>
			<table class="form-table">
				<tbody>
				<tr>
					<th>
						<label for="um_set_verification"><?php esc_html_e( 'Account Verification', 'um-verified' ); ?></label>
					</th>
					<td>
						<select name="um_set_verification" id="um_set_verification">
							<option value="0" selected><?php esc_html_e( 'Unverified Account', 'um-verified' ); ?></option>
							<option value="1"><?php esc_html_e( 'Verified Account', 'um-verified' ); ?></option>
						</select>
					</td>
				</tr>
				</tbody>
			</table>
			<?php
		}

		$content .= ob_get_clean();
		return $content;
	}
	add_filter( 'um_user_new_form_additional_fields', 'um_user_new_form_verification_field', 2 );
} else {
	/**
	 * Verify/unverify from backend profile
	 *
	 * @param string $content
	 * @param $user
	 *
	 * @return string
	 */
	function um_verification_field( $content, $user ) {
		if ( empty( $user ) ) {
			return $content;
		}

		if ( ! isset( $user->ID ) ) {
			return $content;
		}

		global $pagenow;
		if ( 'profile.php' === $pagenow ) {
			return $content;
		}

		if ( current_user_can( 'edit_users' ) && current_user_can( 'edit_user', $user->ID ) ) {
			$user        = get_userdata( $user->ID );
			$is_verified = UM()->Verified_Users_API()->api()->is_verified( $user->ID );
			ob_start(); ?>

			<table class="form-table">
				<tbody>
				<tr>
					<th>
						<label for="um_set_verification"><?php esc_html_e( 'Account Verification', 'um-verified' ); ?></label>
					</th>
					<td>
						<select name="um_set_verification" id="um_set_verification">
							<option value='0' <?php selected( 0, $is_verified ); ?>><?php esc_html_e( 'Unverified Account', 'um-verified' ); ?></option>
							<option value='1' <?php selected( 1, $is_verified ); ?>><?php esc_html_e( 'Verified Account', 'um-verified' ); ?></option>
						</select>
					</td>
				</tr>
				</tbody>
			</table>
			<?php
		}

		$content .= ob_get_clean();
		return $content;
	}
	add_filter( 'um_user_profile_additional_fields', 'um_verification_field', 2, 2 );
}

/**
 * Settings in access widget
 *
 * @param array $fields
 * @param array $data
 * @param object post_type_object
 *
 * @return array
 */
function um_admin_access_settings_fields( $fields, $data, $post_type_object ) {
	$fields[] = array(
		'id'          => '_um_locked_to_verified',
		'type'        => 'checkbox',
		'label'       => __( 'Lock content to verified accounts only?', 'um-verified' ),
		'value'       => ! empty( $data['_um_locked_to_verified'] ) ? $data['_um_locked_to_verified'] : 0,
		'conditional' => array( '_um_accessible', '=', '2' ),
	);

	return $fields;
}
add_filter( 'um_admin_access_settings_fields', 'um_admin_access_settings_fields', 10, 3 );

/**
 * Settings in access widget
 *
 * @param array $fields
 * @param $data
 * @param string $screen
 * @return array
 */
function um_admin_category_access_settings_fields( $fields, $data, $screen ) {
	if ( 'edit' === $screen ) {
		$fields[] = array(
			'id'          => '_um_locked_to_verified',
			'type'        => 'checkbox',
			'class'       => 'form-field',
			'name'        => '_um_locked_to_verified',
			'label'       => __( 'Lock content to verified accounts only?', 'um-verified' ),
			'value'       => ! empty( $data['_um_locked_to_verified'] ) ? $data['_um_locked_to_verified'] : 0,
			'conditional' => array( '_um_accessible', '=', '2' ),
		);
	} elseif ( 'create' === $screen ) {
		$fields[] = array(
			'id'          => '_um_locked_to_verified',
			'type'        => 'checkbox',
			'name'        => '_um_locked_to_verified',
			'label'       => __( 'Lock content to verified accounts only?', 'um-verified' ),
			'value'       => ! empty( $data['_um_locked_to_verified'] ) ? $data['_um_locked_to_verified'] : 0,
			'conditional' => array( '_um_accessible', '=', '2' ),
		);
	}

	return $fields;
}
add_filter( 'um_admin_category_access_settings_fields', 'um_admin_category_access_settings_fields', 10, 3 );

/**
 * Settings in access widget
 *
 * @param bool $has_access
 * @param array $restriction
 * @return bool
 */
function um_verified_users_restriction( $has_access, $restriction ) {
	if ( ! empty( $restriction['_um_locked_to_verified'] ) && ! UM()->Verified_Users_API()->api()->is_verified( get_current_user_id() ) ) {
		return false;
	}

	return $has_access;
}
add_filter( 'um_custom_restriction', 'um_verified_users_restriction', 10, 2 );

/**
 * Lock Gutenberg block to verified accounts only
 *
 * @param bool $display
 * @param array $block
 *
 * @return bool
 */
function um_verified_users_loggedin_block_restriction( $display, $block ) {
	if ( ! empty( $block['attrs']['um_locked_to_verified'] ) ) {
		$is_verified = UM()->Verified_Users_API()->api()->is_verified( get_current_user_id() );
		if ( ! $is_verified ) {
			$display = false;
		}
	}

	return $display;
}
add_filter( 'um_loggedin_block_restriction', 'um_verified_users_loggedin_block_restriction', 10, 2 );

/**
 * Creates options in role page
 *
 * @param $roles_metaboxes
 *
 * @return array
 */
function um_verified_add_role_metabox( $roles_metaboxes ) {

	$roles_metaboxes[] = array(
		'id'       => 'um-admin-form-verified{' . um_verified_users_path . '}',
		'title'    => __( 'Verified Accounts', 'um-verified' ),
		'callback' => array( UM()->metabox(), 'load_metabox_role' ),
		'screen'   => 'um_role_meta',
		'context'  => 'side',
		'priority' => 'default',
	);

	return $roles_metaboxes;
}
add_filter( 'um_admin_role_metaboxes', 'um_verified_add_role_metabox', 10, 1 );

/**
 * Extend UM restiction blocks' options
 *
 * @hooked 'block_type_metadata_settings'
 * @since  2.1.1
 *
 * @param  array $settings
 * @param  array $args
 * @return array
 */
function um_verified_block_type_metadata_settings( $settings, $args ) {
	if ( empty( $settings['attributes']['um_locked_to_verified'] ) ) {
		$settings['attributes']['um_locked_to_verified'] = array(
			'type' => 'boolean',
		);
	}

	return $settings;
}
add_filter( 'block_type_metadata_settings', 'um_verified_block_type_metadata_settings', 10, 2 );

/**
 * Scan templates from extension
 *
 * @param $scan_files
 *
 * @return array
 */
function um_verified_users_extend_scan_files( $scan_files ) {
	$extension_files['um-verified'] = UM()->admin_settings()->scan_template_files( um_verified_users_path . '/templates/' );
	$scan_files                     = array_merge( $scan_files, $extension_files );

	return $scan_files;
}
add_filter( 'um_override_templates_scan_files', 'um_verified_users_extend_scan_files', 10, 1 );

/**
 * Get template paths
 *
 * @param $located
 * @param $file
 *
 * @return array
 */
function um_verified_users_get_path_template( $located, $file ) {
	if ( file_exists( get_stylesheet_directory() . '/ultimate-member/um-verified/' . $file ) ) {
		$located = array(
			'theme' => get_stylesheet_directory() . '/ultimate-member/um-verified/' . $file,
			'core'  => um_verified_users_path . 'templates/' . $file,
		);
	}

	return $located;
}
add_filter( 'um_override_templates_get_template_path__um-verified', 'um_verified_users_get_path_template', 10, 2 );

/**
 * Extend menu fields
 *
 * @param array $fields
 *
 * @return array
 */
function um_wp_nav_menu_fields( $fields ) {
	$fields['um_nav_verified'] = __( 'Verified Lock', 'um-verified' );

	return $fields;
}
add_filter( 'um_wp_nav_menu_fields', 'um_wp_nav_menu_fields' );

/**
 * Verified menu visibility
 *
 * @param bool    $visible
 * @param object  $item
 *
 * @return bool
 */
function um_verified_nav_menu_roles_item_visibility( $visible, $item ) {
	if ( true === (bool) $visible ) {
		$um_nav_verified = get_post_meta( $item->ID, 'menu-item-um_nav_verified', true );
		if ( 1 === absint( $um_nav_verified ) ) {
			$is_verified = UM()->Verified_Users_API()->api()->is_verified( get_current_user_id() );
			if ( ! $is_verified ) {
				$visible = false;
			}
		}
	}

	return $visible;
}
add_filter( 'um_nav_menu_roles_item_visibility', 'um_verified_nav_menu_roles_item_visibility', 1000, 2 );
