<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Block user from adding review
 *
 * @param $action
 * @param $user_id
 */
function um_verified_users_process_user_admin( $action, $user_id ) {
	if ( ! in_array( $action, array( 'verify', 'unverify', 'approve_verification', 'reject_verification' ), true ) ) {
		return;
	}

	if ( ! UM()->common()->users()->can_current_user_edit_user( $user_id ) ) {
		wp_die( esc_html__( 'You do not have permission to edit this user.', 'um-verified' ) );
	}

	if ( 'verify' === $action ) {
		if ( ! wp_verify_nonce( $_REQUEST['nonce'], "verify{$user_id}" ) ) {
			wp_die( esc_html__( 'The link you followed has expired.', 'um-verified' ) );
		}

		$result = UM()->Verified_Users_API()->api()->verify( $user_id, true );
	} elseif ( 'unverify' === $action ) {
		if ( ! wp_verify_nonce( $_REQUEST['nonce'], "unverify{$user_id}" ) ) {
			wp_die( esc_html__( 'The link you followed has expired.', 'um-verified' ) );
		}
		$result = UM()->Verified_Users_API()->api()->unverify( $user_id );
	} elseif ( 'approve_verification' === $action ) {
		if ( ! wp_verify_nonce( $_REQUEST['nonce'], "approve_verification{$user_id}" ) ) {
			wp_die( esc_html__( 'The link you followed has expired.', 'um-verified' ) );
		}

		$result = UM()->Verified_Users_API()->api()->approve_verification( $user_id, true );
	} else {
		if ( ! wp_verify_nonce( $_REQUEST['nonce'], "reject_verification{$user_id}" ) ) {
			wp_die( esc_html__( 'The link you followed has expired.', 'um-verified' ) );
		}

		$result = UM()->Verified_Users_API()->api()->reject_verification( $user_id );
	}

	if ( ! $result ) {
		wp_die( esc_html__( 'Something went wrong.', 'um-verified' ) );
	}

	um_safe_redirect( UM()->permalinks()->get_current_url( true ) );
	exit;
}
add_action( 'um_action_user_request_hook', 'um_verified_users_process_user_admin', 10, 2 );

/**
 * Clear cache data
 */
function um_verified_users_flush_user_status_cache() {
	delete_transient( 'um_count_users_needs-verification' );
}
add_action( 'um_flush_user_status_cache', 'um_verified_users_flush_user_status_cache', 10 );


/**
 * Caching counters
 *
 * @param $meta_ids
 * @param $object_id
 * @param $meta_key
 * @param $_meta_value
 */
function um_verified_users_count_users_transient_update( $meta_ids, $object_id, $meta_key, $_meta_value ) {
	if ( '_um_verified' !== $meta_key ) {
		return;
	}

	$old = get_user_meta( $object_id, $meta_key, true );

	if ( $old === $_meta_value ) {
		return;
	}

	if ( ! ( 'pending' === $old || 'pending' === $_meta_value ) ) {
		return;
	}

	$count = get_transient( 'um_count_users_needs-verification' );
	if ( 'pending' === $old ) {
		if ( false !== $count ) {
			if ( ! is_numeric( $count ) ) {
				delete_transient( 'um_count_users_needs-verification' );
			} else {
				if ( 0 < $count ) {
					$count--;
				} else {
					$count = 0;
				}

				set_transient( 'um_count_users_needs-verification', $count );
			}
		}

		$pending_count = get_transient( 'um_count_users_pending_dot' );
		if ( false !== $pending_count ) {
			if ( ! is_numeric( $pending_count ) ) {
				delete_transient( 'um_count_users_pending_dot' );
			} else {
				if ( 0 < $pending_count ) {
					$pending_count--;
				} else {
					$pending_count = 0;
				}
				set_transient( 'um_count_users_pending_dot', $pending_count );
			}
		}
	} elseif ( 'pending' === $_meta_value ) {
		if ( false !== $count ) {
			if ( is_numeric( $count ) ) {
				$count++;
			} else {
				$count = 1;
			}
		} else {
			$count = 1;
		}

		set_transient( 'um_count_users_needs-verification', $count );

		// add new transient count
		$count = get_transient( 'um_count_users_pending_dot' );
		if ( false !== $count ) {
			if ( is_numeric( $count ) ) {
				$count++;
			} else {
				$count = 1;
			}
		} else {
			$count = 1;
		}
		set_transient( 'um_count_users_pending_dot', $count );
	}
}
add_action( 'update_user_meta', 'um_verified_users_count_users_transient_update', 10, 4 );


/**
 * Caching counters
 *
 * @param $meta_ids
 * @param $object_id
 * @param $meta_key
 * @param $_meta_value
 */
function um_verified_users_count_users_transient_add( $meta_ids, $object_id, $meta_key, $_meta_value ) {
	if ( '_um_verified' !== $meta_key ) {
		return;
	}

	if ( 'pending' !== $_meta_value ) {
		return;
	}

	$count = get_transient( 'um_count_users_needs-verification' );
	if ( false !== $count ) {
		if ( is_numeric( $count ) ) {
			$count++;
		} else {
			$count = 1;
		}
	} else {
		$count = 1;
	}

	set_transient( 'um_count_users_needs-verification', $count );

	// add new transient count
	$count = get_transient( 'um_count_users_pending_dot' );
	if ( false !== $count ) {
		if ( is_numeric( $count ) ) {
			$count++;
		} else {
			$count = 1;
		}
	} else {
		$count = 1;
	}
	set_transient( 'um_count_users_pending_dot', $count );
}
add_action( 'added_user_meta', 'um_verified_users_count_users_transient_add', 10, 4 );


/**
 * Caching counters
 *
 * @param $meta_ids
 * @param $object_id
 * @param $meta_key
 * @param $_meta_value
 */
function um_verified_users_count_users_transient_delete( $meta_ids, $object_id, $meta_key, $_meta_value ) {
	if ( '_um_verified' !== $meta_key ) {
		return;
	}

	$value = ( '' !== $_meta_value ) ? $_meta_value : get_user_meta( $object_id, $meta_key, true );

	if ( 'pending' !== $value ) {
		return;
	}

	$count = get_transient( 'um_count_users_needs-verification' );
	if ( false !== $count ) {
		if ( ! is_numeric( $count ) ) {
			delete_transient( 'um_count_users_needs-verification' );
		} else {
			if ( 0 < $count ) {
				$count--;
			} else {
				$count = 0;
			}

			set_transient( 'um_count_users_needs-verification', $count );
		}
	}

	$count = get_transient( 'um_count_users_pending_dot' );
	if ( false !== $count ) {
		if ( ! is_numeric( $count ) ) {
			delete_transient( 'um_count_users_pending_dot' );
		} else {
			if ( 0 < $count ) {
				$count--;
			} else {
				$count = 0;
			}
			set_transient( 'um_count_users_pending_dot', $count );
		}
	}
}
add_action( 'delete_user_meta', 'um_verified_users_count_users_transient_delete', 10, 4 );

/**
 * Add verification info to profile
 *
 * @param int|array $param1
 * @param int|array $param2
 */
function um_verified_info( $param1, $param2 ) {
	if ( UM()->is_new_ui() ) {
		$user_id = $param2;
		$args    = $param1;
	} else {
		$user_id = $param1;
		$args    = $param2;
	}

	if ( absint( $user_id ) !== get_current_user_id() ) {
		return;
	}

	if ( um_user( 'verified_req_disallowed' ) ) {
		return;
	}

	wp_enqueue_style( 'um-verified' );

	$verified_status = UM()->Verified_Users_API()->api()->verified_status( $user_id );
	$cancel          = UM()->Verified_Users_API()->api()->verify_cancel_url( $user_id, um_user_profile_url( $user_id ) );

	$t_args = compact( 'args', 'cancel', 'user_id', 'verified_status' );
	UM()->get_template( 'profile_info.php', um_verified_users_plugin, $t_args, true );
}
add_action( 'um_after_header_meta', 'um_verified_info', 50, 2 );

/**
 * Add verification info to account
 *
 * @param $shortcode_args
 */
function um_verified_account_info( $shortcode_args ) {
	if ( isset( $shortcode_args['_verified_info'] ) && 0 == $shortcode_args['_verified_info'] ) {
		return;
	}

	$user_id = um_user( 'ID' );

	if ( UM()->Verified_Users_API()->api()->is_verified( $user_id ) ) {
		return;
	}

	if ( um_user( 'verified_req_disallowed' ) ) {
		return;
	}

	wp_enqueue_style( 'um-verified' );

	$verified_status = UM()->Verified_Users_API()->api()->verified_status( $user_id );
	$cancel = UM()->Verified_Users_API()->api()->verify_cancel_url( $user_id, um_get_core_page( 'account' ) );

	$t_args = compact( 'cancel', 'user_id', 'verified_status' );
	UM()->get_template( 'account_info.php', um_verified_users_plugin, $t_args, true );
}
add_action( 'um_after_account_general', 'um_verified_account_info', 10, 1 );


/**
 * Creates user_meta "_um_verified" for sorted
 *
 * @param $user_id
 */
function um_verified_add_meta_um_verified( $user_id ) {
	update_user_meta( $user_id, '_um_verified', 'unverified' );
}
add_action( 'um_before_save_registration_details', 'um_verified_add_meta_um_verified' );


/**
 * Creates user_meta "_um_verified" for sorted
 *
 * @param int $user_id
 */
function um_verified_registration_complete( $user_id ) {
	$user_role   = UM()->roles()->get_priority_user_role( $user_id );
	$permissions = UM()->roles()->role_data( $user_role );
	/** This filter is documented in ultimate-member/includes/core/class-roles-capabilities.php */
	$permissions = apply_filters( 'um_user_permissions_filter', $permissions, $user_id );

	if ( isset( $permissions['verified_by_role'] ) && $permissions['verified_by_role'] ) {
		UM()->Verified_Users_API()->api()->verify( $user_id, true, true );
	} else {
		UM()->Verified_Users_API()->api()->unverify( $user_id, true );
	}
}
add_action( 'um_registration_complete', 'um_verified_registration_complete' );


/**
 *
 */
function um_request_verification() {
	if ( ! is_user_logged_in() ) {
		return;
	}

	if ( isset( $_REQUEST['request_verification'] ) && isset( $_REQUEST['uid'] ) ) {
		$user_id = absint( $_REQUEST['uid'] );

		if ( $user_id != get_current_user_id() || UM()->Verified_Users_API()->api()->verified_status( $user_id ) != 'unverified' ) {
			wp_die( esc_html__( 'Unauthorized.', 'um-verified' ) );
		}

		um_fetch_user( $user_id );

		if ( um_user( 'verified_req_disallowed' ) ) {
			wp_die( esc_html__( 'You are not allowed to do this action.', 'um-verified' ) );
		}

		update_user_meta( $user_id, '_um_verified', 'pending' );
		do_action( 'um_after_user_request_verification', $user_id );

		$emails = um_multi_admin_email();
		if ( ! empty( $emails ) ) {
			foreach ( $emails as $email ) {
				$mail_args = array(
					$email,
					'verification_request',
					array(
						'tags'          => array(
							'{requested_user_link}',
						),
						'tags_replace'  => array(
							UM()->Verified_Users_API()->api()->requested_user_link( $user_id ),
						),
						'fetch_user_id' => $user_id,
					),
				);

				UM()->maybe_action_scheduler()->enqueue_async_action( 'um_dispatch_email', $mail_args );
			}
		}

		um_safe_redirect( esc_url_raw( $_REQUEST['redirect_to'] ) );
	}

	if ( isset( $_REQUEST['request_verification_undo'] ) && isset( $_REQUEST['uid'] ) ) {
		$user_id = absint( $_REQUEST['uid'] );

		if ( $user_id != get_current_user_id() || UM()->Verified_Users_API()->api()->verified_status( $user_id ) != 'pending' ) {
			wp_die( __( 'Unauthorized.', 'um-verified' ) );
		}

		um_fetch_user( $user_id );

		update_user_meta( $user_id, '_um_verified', 'unverified' );
		do_action('um_after_user_undo_request_verification', $user_id );

		um_safe_redirect( esc_url_raw( $_REQUEST['redirect_to'] ) );
	}
}
add_action( 'init', 'um_request_verification', 5 );


/**
 * @param $user_id
 */
function um_update_verification_field( $user_id ) {
	if ( isset( $_POST['um_set_verification'] ) && current_user_can( 'edit_user', $user_id ) ) {
		$new_state = (bool) $_POST['um_set_verification'];
		$state     = UM()->Verified_Users_API()->api()->is_verified( $user_id );

		if ( true === $new_state && false === $state ) {
			UM()->Verified_Users_API()->api()->verify( $user_id, false, true );
			// don't change by role because Verification field has be changed.
			remove_action( 'um_after_user_role_is_updated', 'um_after_user_role_is_updated', 1000 );
		} elseif ( true === $state && false === $new_state ) {
			UM()->Verified_Users_API()->api()->unverify( $user_id, true );
			// don't change by role because Verification field has be changed.
			remove_action( 'um_after_user_role_is_updated', 'um_after_user_role_is_updated', 1000 );
		}
	}
}
add_action( 'edit_user_profile_update', 'um_update_verification_field', 10, 1 );


/**
 * Save user group as verified accounts one time
 *
 * @param $post_id
 * @param $post
 */
function um_admin_before_save_role( $post_id, $post ) {
	$v = absint( $_POST['_um_verified_by_role'] );
	if ( $v == 1 && ! get_option( 'um_verified_' . $post->post_name ) ) {

		$args = array( 'fields' => 'ID', 'number' => 0 );
		$args['meta_query'][] = array( array( 'key' => 'role', 'value' => $post->post_name, 'compare' => '=' ) );

		$users = new WP_User_Query( $args );

		if ( isset( $users->results ) ) {
			foreach ( $users->results as $user_id ) {
				UM()->Verified_Users_API()->api()->verify( $user_id, false, true );
			}
		}

		update_option( 'um_verified_'. $post->post_name, 'done' );
	}
}
add_action( 'um_admin_before_save_role', 'um_admin_before_save_role', 1000, 2 );

/**
 * Auto-verify role's account
 *
 * @param $user_id
 * @param $role
 */
function um_after_user_role_is_updated( $user_id, $role ) {
	$meta = UM()->roles()->role_data( $role );
	/** This filter is documented in ultimate-member/includes/core/class-roles-capabilities.php */
	$meta = apply_filters( 'um_user_permissions_filter', $meta, $user_id );
	if ( isset( $meta['verified_by_role'] ) && $meta['verified_by_role'] ) {
		UM()->Verified_Users_API()->api()->verify( $user_id, false, true );
	} else {
		UM()->Verified_Users_API()->api()->unverify( $user_id, true );
	}
}
add_action( 'um_after_user_role_is_updated', 'um_after_user_role_is_updated', 1000, 2 );

/**
 * Action: Verify user when update profile if complete
 *
 * @hook  um_update_profile_full_name
 *
 * @param array $to_update  An array of submitted data
 * @param int   $user_id    The user ID
 */
function um_verify_user_if_complete( $to_update, $user_id ) {
	if ( ! defined( 'um_profile_completeness_version' ) || UM()->Verified_Users_API()->api()->is_verified( $user_id ) ) {
		return;
	}

	$user_role = UM()->roles()->get_priority_user_role( $user_id );
	$role_data = UM()->roles()->role_data( $user_role );

	if ( empty( $role_data['profilec_verify'] ) ) {
		return;
	}

	$completeness = UM()->Profile_Completeness_API()->shortcode()->profile_progress( $user_id );

	if ( is_array( $completeness ) && $completeness && (int) $completeness['progress'] >= (int) $completeness['req_progress'] ) {
		UM()->Verified_Users_API()->api()->verify( $user_id, true, true );
		// don't change by role because Verification field has be changed.
		remove_action( 'um_after_user_role_is_updated', 'um_after_user_role_is_updated', 1000 );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'UM Log: Verify user from ' . __FUNCTION__ );
		}
	}
}
add_action( 'um_user_after_updating_profile', 'um_verify_user_if_complete', 20, 2 );

/**
 * Add "Lock to verified accounts only" option to menu items
 */
function um_wp_nav_menu_custom_verified_field( $item_id, $um_nav_public ) {
	$show = 'none';
	if ( 2 === absint( $um_nav_public ) ) {
		$show = 'block';
	}

	$um_nav_verified = get_post_meta( $item_id, 'menu-item-um_nav_verified', true );
	echo '<div class="clear"></div>';
	echo '<p class="description description-wide um-wp-nav-verified" style="display: ' . esc_attr( $show ) . '">';
	echo '<label>' . esc_html__( 'Verified options', 'um-verified' ) . '<br /><input ' . checked( 1 === absint( $um_nav_verified ), true, false ) . ' type="checkbox" name="menu-item-um_nav_verified[' . esc_attr( $item_id ) . ']" value="1" />' . esc_html__( 'Lock to verified accounts only', 'um-verified' ) . '</label>';
	echo '</p>';
}
add_action( 'um_wp_nav_menu_custom_fields', 'um_wp_nav_menu_custom_verified_field', 10, 2 );
