<?php
namespace um_ext\um_verified_users\core;

use WP_User_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Verified_Users_Main_API
 * @package um_ext\um_verified_users\core
 */
class Verified_Users_Main_API {

	/**
	 * Number of pending requests
	 */
	public function verified_requests_count() {
		$users_count = get_transient( 'um_count_users_needs-verification' );
		if ( false === $users_count ) {
			$args = array(
				'fields'               => 'ids',
				'number'               => 1,
				'meta_query'           => array(
					array(
						'key'     => '_um_verified',
						'value'   => 'pending',
						'compare' => '=',
					),
				),
				'um_custom_user_query' => true,
			);

			$users = new WP_User_Query( $args );
			if ( empty( $users ) || is_wp_error( $users ) ) {
				$users_count = 0;
			} else {
				$users_count = $users->get_total();
			}

			set_transient( 'um_count_users_needs-verification', $users_count );
		}

		return $users_count;
	}

	/**
	 * URL to verify a user
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function verify_user_url( $user_id ) {
		$url = add_query_arg(
			array(
				'um_adm_action' => 'verify_user',
				'uid'           => $user_id,
				'_wpnonce'      => wp_create_nonce( 'verify_user' . $user_id ),
			),
			admin_url( 'users.php' )
		);
		return $url;
	}

	/**
	 * URL to un-verify a user
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function unverify_user_url( $user_id ) {
		$url = add_query_arg(
			array(
				'um_adm_action' => 'unverify_user',
				'uid'           => $user_id,
				'_wpnonce'      => wp_create_nonce( 'unverify_user' . $user_id ),
			),
			admin_url( 'users.php' )
		);
		return $url;
	}

	/**
	 * URL to verify a user
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function approve_verification_user_url( $user_id ) {
		$url = add_query_arg(
			array(
				'um_adm_action' => 'verification_approved',
				'uid'           => $user_id,
				'_wpnonce'      => wp_create_nonce( 'verification_approved' . $user_id ),
			),
			admin_url( 'users.php' )
		);
		return $url;
	}

	/**
	 * URL to un-verify a user
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function reject_verification_user_url( $user_id ) {
		$url = add_query_arg(
			array(
				'um_adm_action' => 'verification_rejected',
				'uid'           => $user_id,
				'_wpnonce'      => wp_create_nonce( 'verification_rejected' . $user_id ),
			),
			admin_url( 'users.php' )
		);
		return $url;
	}

	/**
	 * URL to un-verify a user
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function requested_user_link( $user_id ) {
		$user = get_userdata( $user_id );
		$url  = admin_url( 'users.php' );
		if ( ! empty( $user ) ) {
			$url = add_query_arg(
				array(
					's' => $user->user_login,
				),
				$url
			);
		}
		return $url;
	}

	/**
	 * Check if user is verified
	 *
	 * @param int $user_id
	 *
	 * @return bool
	 */
	public function is_verified( $user_id ) {
		$is_verified = get_user_meta( $user_id, '_um_verified', true );
		return 'verified' === $is_verified;
	}

	/**
	 * Get user verification status
	 *
	 * @param int $user_id
	 *
	 * @return string (verified||unverified||pending)
	 */
	public function verified_status( $user_id, $format = 'raw' ) {
		$is_verified = get_user_meta( $user_id, '_um_verified', true );
		$is_verified = ! empty( $is_verified ) ? $is_verified : 'unverified';
		if ( 'raw' === $format ) {
			return $is_verified;
		}

		$all_statuses = array(
			'verified'   => __( 'Verified', 'um-verified' ),
			'unverified' => __( 'Unverified', 'um-verified' ),
			'pending'    => __( 'Requested verification', 'um-verified' ),
		);

		if ( array_key_exists( $is_verified, $all_statuses ) ) {
			return $all_statuses[ $is_verified ];
		}

		return __( 'Undefined verification status', 'um-verified' );
	}

	/**
	 * Check if the user can be verified.
	 *
	 * @param int  $user_id User ID
	 * @param bool $force
	 *
	 * @return bool
	 */
	public function can_be_verified( $user_id, $force = false ) {
		if ( ! UM()->common()->users()::user_exists( $user_id ) ) {
			return false;
		}

		if ( ! $force ) {
			$current_user_id = get_current_user_id();
			if ( $current_user_id === $user_id ) {
				return false;
			}
		}

		/*if ( ! UM()->common()->users()->can_current_user_edit_user( $user_id ) ) {
			return false;
		}*/

		return ! $this->is_verified( $user_id );
	}

	/**
	 * Check if the user can be verified.
	 *
	 * @param int $user_id User ID
	 *
	 * @return bool
	 */
	public function can_approve_verification( $user_id ) {
		$current_user_id = get_current_user_id();
		if ( $current_user_id === $user_id || ! UM()->common()->users()::user_exists( $user_id ) ) {
			return false;
		}

		/*if ( ! UM()->common()->users()->can_current_user_edit_user( $user_id ) ) {
			return false;
		}*/

		$status = $this->verified_status( $user_id );

		return 'pending' === $status;
	}

	/**
	 * Check if the user can be verified.
	 *
	 * @param int $user_id User ID
	 *
	 * @return bool
	 */
	public function can_reject_verification( $user_id ) {
		$current_user_id = get_current_user_id();
		if ( $current_user_id === $user_id || ! UM()->common()->users()::user_exists( $user_id ) ) {
			return false;
		}

		/*if ( ! UM()->common()->users()->can_current_user_edit_user( $user_id ) ) {
			return false;
		}*/

		$status = $this->verified_status( $user_id );

		return 'pending' === $status;
	}

	/**
	 * Verify user
	 *
	 * @param int  $user_id
	 * @param bool $sendmail
	 * @param bool $force
	 */
	public function verify( $user_id, $sendmail = false, $force = false ) {
		if ( ! $this->can_be_verified( $user_id, $force ) ) {
			return false;
		}

		do_action( 'um_before_user_is_verified', $user_id );

		$result = update_user_meta( $user_id, '_um_verified', 'verified' );

		if ( false !== $result ) {
			do_action( 'um_after_user_is_verified', $user_id );

			if ( $sendmail ) {
				um_fetch_user( $user_id );

				$user  = get_userdata( $user_id );
				$email = $user->user_email;

				$mail_args = array(
					$email,
					'verified_account',
					array(
						'fetch_user_id' => $user_id,
					),
				);

				UM()->maybe_action_scheduler()->enqueue_async_action( 'um_dispatch_email', $mail_args );
			}

			return true;
		}
		return false;
	}

	/**
	 * Unverify user
	 *
	 * @param int  $user_id
	 * @param bool $force
	 *
	 * @return bool
	 */
	public function unverify( $user_id, $force = false ) {
		if ( ! $this->can_be_unverified( $user_id, $force ) ) {
			return false;
		}

		$result = update_user_meta( $user_id, '_um_verified', 'unverified' );
		if ( false !== $result ) {
			do_action( 'um_after_user_is_unverified', $user_id );
			return true;
		}
		return false;
	}

	/**
	 * Check if the user can be unverified.
	 *
	 * @param int  $user_id User ID
	 * @param bool $force   Ignore current user if `true`
	 *
	 * @return bool
	 */
	public function can_be_unverified( $user_id, $force = false ) {
		if ( ! UM()->common()->users()::user_exists( $user_id ) ) {
			return false;
		}

		if ( ! $force ) {
			$current_user_id = get_current_user_id();
			if ( $current_user_id === $user_id ) {
				return false;
			}
		}

		/*if ( ! UM()->common()->users()->can_current_user_edit_user( $user_id ) ) {
			return false;
		}*/

		return $this->is_verified( $user_id );
	}

	/**
	 * Approve user verification
	 *
	 * @param int $user_id
	 * @param bool $sendmail
	 */
	public function approve_verification( $user_id, $sendmail = false ) {
		if ( ! $this->can_approve_verification( $user_id ) ) {
			return false;
		}

		return $this->verify( $user_id, $sendmail );
	}

	/**
	 * Reject user verification
	 *
	 * @param int $user_id
	 */
	public function reject_verification( $user_id ) {
		if ( ! $this->can_reject_verification( $user_id ) ) {
			return false;
		}

		$result = update_user_meta( $user_id, '_um_verified', 'unverified' );
		if ( false !== $result ) {
			do_action( 'um_after_user_reject_verification', $user_id );
			return true;
		}
		return false;
	}

	/**
	 * Verification request URL
	 *
	 * @param int $user_id
	 * @param string $redirect_to
	 *
	 * @return string
	 */
	public function verify_url( $user_id, $redirect_to = '' ) {
		$args = array(
			'request_verification' => 'true',
			'uid'                  => $user_id,
		);

		if ( $redirect_to ) {
			$args['redirect_to'] = rawurlencode( $redirect_to );
		}
		return add_query_arg( $args, get_bloginfo( 'url' ) );
	}

	/**
	 * Cancel verification request URL
	 *
	 * @param int $user_id
	 * @param string $redirect_to
	 *
	 * @return string
	 */
	public function verify_cancel_url( $user_id, $redirect_to = '' ) {
		$args = array(
			'request_verification_undo' => 'true',
			'uid'                       => $user_id,
		);

		if ( $redirect_to ) {
			$args['redirect_to'] = rawurlencode( $redirect_to );
		}
		return add_query_arg( $args, get_bloginfo( 'url' ) );
	}

	/**
	 * @return string
	 */
	public function verified_badge() {
		wp_enqueue_style( 'um-verified' );
		return '<i class="um-verified um-icon-checkmark-circled um-tip-s" title="' . esc_attr__( 'Verified Account', 'um-verified' ) . '"></i>';
	}
}
