<?php
namespace um_ext\um_verified_users\core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Users_Columns
 * @package um_ext\um_verified_users\core
 */
class Users_Columns {

	/**
	 * Users_Columns constructor.
	 */
	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( &$this, 'users_screen_scripts' ) );
		add_filter( 'um_users_column_account_status', array( $this, 'extends_account_status_column' ), 10, 2 );
		add_filter( 'um_users_column_account_status_row_actions', array( $this, 'extends_account_status_column_row_actions' ), 10, 2 );
		add_filter( 'um_user_statuses_admin_filter_options', array( $this, 'extends_user_statuses_filter' ) );
		add_filter( 'um_admin_pending_queue_filter', array( $this, 'admin_pending_queue_extend' ) );
		add_filter( 'um_skip_filter_users_by_status', array( $this, 'skip_filter_users_by_status' ), 10, 2 );
		add_action( 'pre_user_query', array( $this, 'filter_users_by_status' ) );
		add_filter( 'um_admin_bulk_user_actions_hook', array( $this, 'extend_bulk_actions' ), 100 );
		add_filter( 'um_handle_bulk_actions-users-um_approve_verification_request', array( $this, 'bulk_approve_verification_request' ), 10, 2 );
		add_filter( 'um_handle_bulk_actions-users-um_reject_verification_request', array( $this, 'bulk_reject_verification_request' ), 10, 2 );
		add_filter( 'um_handle_bulk_actions-users-um_verify_accounts', array( $this, 'bulk_verify_accounts' ), 10, 2 );
		add_filter( 'um_handle_bulk_actions-users-um_unverify_accounts', array( $this, 'bulk_unverify_accounts' ), 10, 2 );
		add_filter( 'removable_query_args', array( &$this, 'add_removable_query_args' ) );
	}

	public function users_screen_scripts( $hook ) {
		$suffix = UM()->admin()->enqueue()::get_suffix();
		if ( 'users.php' === $hook ) {
			wp_register_style( 'um_admin_verified_users', um_verified_users_url . 'assets/css/admin/verified-users' . $suffix . '.css', array(), um_verified_users_version );
			wp_enqueue_style( 'um_admin_verified_users' );
		}
	}

	/**
	 * Filter "Users" table, column "Status"
	 *
	 * @since  2.0.5
	 *
	 * @param   string $status
	 * @param   int    $user_id
	 * @return  string
	 */
	public function extends_account_status_column( $status, $user_id ) {
		$status_formatted = UM()->Verified_Users_API()->api()->verified_status( $user_id, 'formatted' );
		return $status . ', ' . $status_formatted;
	}

	/**
	 * Filter "Users" table, column "Status"
	 *
	 * @since  2.0.5
	 *
	 * @param   array $row_actions
	 * @param   int    $user_id
	 * @return  array
	 */
	public function extends_account_status_column_row_actions( $row_actions, $user_id ) {
		$status = UM()->Verified_Users_API()->api()->verified_status( $user_id );

		if ( 'unverified' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_be_verified( $user_id ) ) {
				$verify_url    = UM()->Verified_Users_API()->api()->verify_user_url( $user_id );
				$row_actions[] = '<a href="' . esc_url( $verify_url ) . '" class="um-set-verification-status-verified">' . esc_html__( 'Verify', 'um-verified' ) . '</a>';
			}
		} elseif ( 'pending' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_approve_verification( $user_id ) ) {
				$approve_url   = UM()->Verified_Users_API()->api()->approve_verification_user_url( $user_id );
				$row_actions[] = '<a href="' . esc_url( $approve_url ) . '" class="um-set-verification-status-verified">' . esc_html__( 'Approve verification request', 'um-verified' ) . '</a>';
			}
			if ( UM()->Verified_Users_API()->api()->can_reject_verification( $user_id ) ) {
				$reject_url    = UM()->Verified_Users_API()->api()->reject_verification_user_url( $user_id );
				$row_actions[] = '<a href="' . esc_url( $reject_url ) . '" class="um-set-verification-status-unverified" onclick="return confirm( \'' . esc_js( __( 'Are you sure you want to reject verification request?', 'um-verified' ) ) . '\' );">' . esc_html__( 'Reject verification request', 'um-verified' ) . '</a>';
			}
		} elseif ( 'verified' === $status ) {
			if ( UM()->Verified_Users_API()->api()->can_be_unverified( $user_id ) ) {
				$unverify_url  = UM()->Verified_Users_API()->api()->unverify_user_url( $user_id );
				$row_actions[] = '<a href="' . esc_url( $unverify_url ) . '" class="um-set-verification-status-unverified" onclick="return confirm( \'' . esc_js( __( 'Are you sure you want to mark this user as unverified?', 'um-verified' ) ) . '\' );">' . esc_html__( 'Unverify', 'um-verified' ) . '</a>';
			}
		}

		return $row_actions;
	}

	/**
	 *
	 * @param array $statuses
	 *
	 * @return array
	 *
	 * @since UM 2.8.7
	 */
	public function extends_user_statuses_filter( $statuses ) {
		$statuses['verified']           = __( 'Verified', 'um-verified' );
		$statuses['unverified']         = __( 'Unverified', 'um-verified' );
		$statuses['needs_verification'] = __( 'Requested verification', 'um-verified' );
		return $statuses;
	}

	/**
	 * Modify pending users queue (Admin red dot)
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	public function admin_pending_queue_extend( $args ) {
		$args['meta_query'][] = array(
			'key'     => '_um_verified',
			'value'   => 'pending',
			'compare' => '=',
		);
		return $args;
	}

	/**
	 * @since UM 2.8.7
	 * @param bool   $skip
	 * @param string $status
	 *
	 * @return bool
	 */
	public function skip_filter_users_by_status( $skip, $status ) {
		if ( in_array( $status, array( 'verified', 'needs_verification', 'unverified' ), true ) ) {
			$skip = true;
		}

		return $skip;
	}

	/**
	 * @since UM 2.8.7
	 *
	 * @param $query
	 *
	 * @return void
	 */
	public function filter_users_by_status( $query ) {
		global $wpdb, $pagenow;

		if ( 'users.php' !== $pagenow || ! is_admin() ) {
			return;
		}

		if ( empty( $_REQUEST['um_user_status'] ) ) {
			return;
		}

		$status = sanitize_key( $_REQUEST['um_user_status'] );

		if ( ! in_array( $status, array( 'verified', 'needs_verification', 'unverified' ), true ) ) {
			return;
		}

		if ( 'needs_verification' === $status ) {
			$query->query_where = str_replace(
				'WHERE 1=1',
				"WHERE 1=1 AND
			{$wpdb->users}.ID IN (
				SELECT {$wpdb->usermeta}.user_id
				FROM $wpdb->usermeta
				WHERE {$wpdb->usermeta}.meta_key = '_um_verified' AND
				{$wpdb->usermeta}.meta_value = 'pending'
			)",
				$query->query_where
			);
		} elseif ( 'verified' === $status ) {
			$query->query_where = str_replace(
				'WHERE 1=1',
				"WHERE 1=1 AND
			{$wpdb->users}.ID IN (
				SELECT {$wpdb->usermeta}.user_id
				FROM $wpdb->usermeta
				WHERE {$wpdb->usermeta}.meta_key = '_um_verified' AND
				{$wpdb->usermeta}.meta_value = 'verified'
			)",
				$query->query_where
			);
		} else {
			$query->query_where = str_replace(
				'WHERE 1=1',
				"WHERE 1=1 AND
			{$wpdb->users}.ID IN (
				SELECT {$wpdb->usermeta}.user_id
				FROM $wpdb->usermeta
				WHERE {$wpdb->usermeta}.meta_key = '_um_verified' AND
				{$wpdb->usermeta}.meta_value NOT IN('verified','pending')
			)",
				$query->query_where
			);
		}
	}

	/**
	 * Add bulk actions
	 *
	 * @since UM 2.8.7 changed format
	 *
	 * @param $actions
	 *
	 * @return mixed
	 */
	public function extend_bulk_actions( $actions ) {
		$actions['um_approve_verification_request'] = __( 'Approve verification request', 'um-verified' );
		$actions['um_reject_verification_request']  = __( 'Reject verification request', 'um-verified' );
		$actions['um_verify_accounts']              = __( 'Mark accounts as verified', 'um-verified' );
		$actions['um_unverify_accounts']            = __( 'Mark accounts as unverified', 'um-verified' );
		return $actions;
	}

	/**
	 * @param string $sendback
	 * @param array  $userids
	 *
	 * @return string
	 */
	public function bulk_approve_verification_request( $sendback, $userids ) {
		// 'um-bulk-users' nonce is already verified here
		$verification_approved_count = 0;
		foreach ( $userids as $user_id ) {
			$res = UM()->Verified_Users_API()->api()->approve_verification( $user_id, true );
			if ( $res ) {
				++$verification_approved_count;
			}
		}

		$sendback = add_query_arg(
			array(
				'approved_count' => $verification_approved_count,
				'update'         => 'um_verification_approved',
			),
			UM()->admin()->users_columns()->set_redirect_uri( $sendback )
		);
		return $sendback;
	}

	/**
	 * @param string $sendback
	 * @param array  $userids
	 *
	 * @return string
	 */
	public function bulk_reject_verification_request( $sendback, $userids ) {
		// 'um-bulk-users' nonce is already verified here
		$verification_rejected_count = 0;
		foreach ( $userids as $user_id ) {
			$res = UM()->Verified_Users_API()->api()->reject_verification( $user_id );
			if ( $res ) {
				++$verification_rejected_count;
			}
		}

		$sendback = add_query_arg(
			array(
				'rejected_count' => $verification_rejected_count,
				'update'         => 'um_verification_rejected',
			),
			UM()->admin()->users_columns()->set_redirect_uri( $sendback )
		);
		return $sendback;
	}

	/**
	 * @param string $sendback
	 * @param array  $userids
	 *
	 * @return string
	 */
	public function bulk_verify_accounts( $sendback, $userids ) {
		// 'um-bulk-users' nonce is already verified here
		$verified_count = 0;
		foreach ( $userids as $user_id ) {
			$res = UM()->Verified_Users_API()->api()->verify( $user_id, true );
			if ( $res ) {
				++$verified_count;
			}
		}

		$sendback = add_query_arg(
			array(
				'verified_count' => $verified_count,
				'update'         => 'um_verified',
			),
			UM()->admin()->users_columns()->set_redirect_uri( $sendback )
		);
		return $sendback;
	}

	/**
	 * @param string $sendback
	 * @param array  $userids
	 *
	 * @return string
	 */
	public function bulk_unverify_accounts( $sendback, $userids ) {
		// 'um-bulk-users' nonce is already verified here
		$unverified_count = 0;
		foreach ( $userids as $user_id ) {
			$res = UM()->Verified_Users_API()->api()->unverify( $user_id );
			if ( $res ) {
				++$unverified_count;
			}
		}

		$sendback = add_query_arg(
			array(
				'unverified_count' => $unverified_count,
				'update'           => 'um_unverified',
			),
			UM()->admin()->users_columns()->set_redirect_uri( $sendback )
		);
		return $sendback;
	}

	/**
	 * Add query args to list of query variable names to remove.
	 *
	 * @param array $removable_query_args An array of query variable names to remove from a URL
	 *
	 * @return array
	 */
	public function add_removable_query_args( $removable_query_args ) {
		$removable_query_args[] = 'unverified_count';
		$removable_query_args[] = 'verified_count';
		return $removable_query_args;
	}
}
