<?php
namespace um_ext\um_user_tags\core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class User_Tags_Member_Directory
 *
 * @package um_ext\um_user_tags\core
 */
class User_Tags_Member_Directory {

	/**
	 * User_Tags_Member_Directory constructor.
	 */
	public function __construct() {
		add_action( 'um_pre_directory_shortcode', array( &$this, 'directory_enqueue_scripts' ) );

		add_filter( 'um_search_fields', array( &$this, 'user_tags_filter_dropdown' ) );

		add_filter( 'um_members_directory_custom_field_types_supported_filter', array( &$this, 'custom_field_types_supported_filter' ) );

		add_filter( 'um_general_search_custom_search_where', array( &$this, 'add_user_tags_fields' ), 10, 3 );

		add_filter( 'um_member_directory_meta_general_search_meta_query', array( &$this, 'extends_search_query_meta' ), 10, 2 );
		add_filter( 'um_member_directory_filter_select_options', array( &$this, 'user_tags_filter_options' ), 10, 3 );
		add_filter( 'um_member_directory_filter_select_options_sorted', array( &$this, 'user_tags_filter_options_sort' ), 10, 2 );

		// Change filter query for search at Member Directory.
		add_filter( 'um_query_args_filter_global', array( $this, 'filter_global' ), 10, 4 );
		add_filter( 'um_query_args_filter_global_meta', array( $this, 'filter_global_meta' ), 10, 6 );
	}

	/**
	 * Enqueue scripts on member directory
	 */
	public function directory_enqueue_scripts() {
		wp_enqueue_style( 'um-user-tags' );
		wp_enqueue_script( 'um-user-tags' );
		wp_enqueue_script( 'um-user-tags-members' );
	}

	/**
	 * @param array $options
	 *
	 * @return array
	 */
	public function custom_field_types_supported_filter( $options ) {
		$options[] = 'user_tags';
		return $options;
	}

	/**
	 * @param array $attrs
	 * @return array
	 */
	public function user_tags_filter_dropdown( $attrs ) {
		if ( isset( $attrs['type'] ) && 'user_tags' === $attrs['type'] ) {
			$attrs['options'] = apply_filters( 'um_multiselect_options', array(), $attrs );
			$attrs['custom']  = 1;
		}

		return $attrs;
	}

	/**
	 * Replace search query when term has been found
	 *
	 * @param $search_where
	 * @param $user_query
	 * @param $search
	 *
	 * @return array|mixed|string|string[]
	 */
	public function add_user_tags_fields( $search_where, $user_query, $search ) {
		$term = get_term_by( 'name', $search, 'um_user_tag' );
		if ( ! empty( $term->term_id ) ) {
			$search_where_tags = array(
				$user_query->get_search_sql( serialize( (string) $term->term_id ), array( 'um_search.meta_value' ), 'both' ),
				$user_query->get_search_sql( serialize( $term->term_id ), array( 'um_search.meta_value' ), 'both' ),
			);
			if ( $search !== $term->slug ) {
				$search_where_tags[] = $user_query->get_search_sql( serialize( $term->slug ), array( 'um_search.meta_value' ), 'both' );
			}
			$search_where_tags = array_values( array_unique( $search_where_tags ) );

			$search_where_tags = array_map(
				static function ( $value ) {
					return ' ' . trim( $value, ' AND()' );
				},
				$search_where_tags
			);

			if ( ! empty( $search_where_tags ) ) {
				$search_where_tags = implode( ' OR ', $search_where_tags );
				$search_where      = str_replace( ' AND (', ' AND (' . $search_where_tags . ' OR ', $search_where );
			}
		}
		return $search_where;
	}

	/**
	 * UM metadata compatibility
	 *
	 * @param string $query_string
	 * @param string $search
	 *
	 * @return string
	 */
	public function extends_search_query_meta( $query_string, $search ) {
		$term = get_term_by( 'name', trim( $search ), 'um_user_tag' );

		if ( ! empty( $term->term_id ) ) {
			global $wpdb;
			$query_string .= $wpdb->prepare(
				' OR umm_search.um_value LIKE %s OR umm_search.um_value LIKE %s OR umm_search.um_value LIKE %s',
				'%' . $wpdb->esc_like( serialize( $term->term_id ) ) . '%',
				'%' . $wpdb->esc_like( serialize( (string) $term->term_id ) ) . '%',
				'%' . $wpdb->esc_like( serialize( $term->slug ) ) . '%'
			);
		}

		return $query_string;
	}


	/**
	 * Change filter query for search at Member Directory (default tables).
	 *
	 * @see   \um\core\Member_Directory::filters()
	 * @since 2.2.9
	 *
	 * @param array|bool $field_query Field query.
	 * @param string     $field       Field key.
	 * @param array      $value       Filter value(s).
	 * @param string     $filter_type Filter type.
	 *
	 * @return array The filter meta query array.
	 */
	public function filter_global( $field_query, $field, $value, $filter_type ) {
		if ( is_array( $value ) && 'select' === $filter_type ) {
			$field_type = UM()->fields()->get_field_type( $field );
			if ( 'user_tags' === $field_type ) {
				$field_query = array(
					'relation' => 'OR',
				);

				foreach ( $value as $single_val ) {
					$single_val    = trim( stripslashes( $single_val ) );
					$field_query[] = array(
						'key'       => $field,
						'value'     => serialize( $single_val ),
						'compare'   => 'LIKE',
					);
				}
			}
		}
		return $field_query;
	}


	/**
	 * Change filter query for search at Member Directory (custom table).
	 *
	 * @see   \um\core\Member_Directory_Meta::handle_filter_query()
	 * @since 2.2.9
	 *
	 * @param bool       $skip        Skip default filter handler marker.
	 * @param object     $directory   Member_Directory_Meta class instance.
	 * @param string     $field       Filter's field key.
	 * @param array|bool $value       Filter value.
	 * @param string     $filter_type Filter type.
	 * @param bool       $is_default  If it's admin filtering option then `true`.
	 *
	 * @return bool Skip default filter handler marker.
	 */
	public function filter_global_meta( $skip, $directory, $field, $value, $filter_type, $is_default ) {
		if ( is_array( $value ) && 'select' === $filter_type ) {
			$field_type = UM()->fields()->get_field_type( $field );
			if ( 'user_tags' === $field_type ) {
				global $wpdb;

				$join_alias = esc_sql( 'umm' . $field );

				$directory->joins[] = $wpdb->prepare(
					"INNER JOIN {$wpdb->prefix}um_metadata %i ON (%i.user_id = u.ID AND %i.um_key = %s)",
					$join_alias,
					$join_alias,
					$join_alias,
					$field
				);

				$values_array = array();
				foreach ( $value as $single_val ) {
					$single_val     = trim( stripslashes( $single_val ) );
					$values_array[] = $wpdb->prepare(
						"%i.um_value LIKE %s",
						$join_alias,
						'%' . $wpdb->esc_like( serialize( $single_val ) ) . '%'
					);
				}

				$directory->where_clauses[] = '(' . implode( ' OR ', $values_array ) . ')';

				if ( ! $is_default ) {
					$directory->custom_filters_in_query[ $field ] = $value;
				}

				$skip = true;
			}
		}
		return $skip;
	}


	/**
	 * @param $options
	 * @param $values_array
	 * @param $attrs
	 *
	 * @return mixed
	 */
	public function user_tags_filter_options( $options, $values_array, $attrs ) {
		if ( 'user_tags' !== $attrs['type'] ) {
			return $options;
		}

		$fields = UM()->builtin()->all_user_fields;
		$attrs  = $fields[ $attrs['metakey'] ];
		$attrs  = apply_filters( 'um_search_fields', $attrs, $attrs['metakey'] );

		if ( ! empty( $values_array ) ) {
			$values_array = array_map( 'maybe_unserialize', $values_array );
			$temp_values  = array();
			$temp_values2 = array();
			foreach ( $values_array as $values ) {
				if ( is_array( $values ) ) {
					$temp_values2[] = $values;
				} else {
					$temp_values[] = $values;
				}
			}
			$temp_values2 = array_merge( ...$temp_values2 );
			$temp_values  = array_merge( $temp_values, $temp_values2 );
			$values_array = array_unique( $temp_values );
		}

		return array_intersect_key( array_map( 'trim', $attrs['options'] ), array_flip( $values_array ) );
	}

	/**
	 * @param array $options
	 * @param array $attrs
	 *
	 * @return array
	 */
	public function user_tags_filter_options_sort( $options, $attrs ) {
		if ( 'user_tags' !== $attrs['type'] ) {
			return $options;
		}

		asort( $options );

		return $options;
	}
}
