<?php
namespace um_ext\um_user_tags\core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class User_Tags_Admin
 * @package um_ext\um_user_tags\core
 */
class User_Tags_Admin {

	/**
	 * User_Tags_Admin constructor.
	 */
	public function __construct() {
		add_action( 'um_extend_admin_menu', array( &$this, 'um_extend_admin_menu' ), 5 );

		add_filter( 'um_settings_structure', array( &$this, 'extend_settings' ) );

		add_filter( 'um_admin_role_metaboxes', array( &$this, 'add_role_metabox' ), 10, 1 );

		add_filter( 'um_override_templates_scan_files', array( &$this, 'um_user_tags_extend_scan_files' ), 10, 1 );
		add_filter( 'um_override_templates_get_template_path__um-user-tags', array( &$this, 'um_user_tags_get_path_template' ), 10, 2 );

		add_action( 'admin_menu', array( $this, 'prepare_metabox' ), 20 );
		add_action( 'um_admin_do_action__user_tags_cache', array( $this, 'recalculate_tags' ) );
		add_filter( 'um_adm_action_custom_update_notice', array( &$this, 'recalculate_tags_notice' ), 10, 2 );

		add_filter( 'manage_edit-um_user_tag_columns', array( &$this, 'user_tag_columns' ) );
		add_filter( 'manage_um_user_tag_custom_column', array( &$this, 'user_tag_custom_column' ), 10, 3 );
		add_filter( 'manage_edit-um_user_tag_sortable_columns', array( &$this, 'user_tag_sortable_columns' ) );
	}

	/**
	 * Add metabox User Tags
	 */
	public function prepare_metabox() {
		add_action( 'load-toplevel_page_ultimatemember', array( &$this, 'load_metabox' ) );
	}

	/**
	 * Load metabox User Tags
	 */
	public function load_metabox() {
		add_meta_box( 'um-metaboxes-user-tags', __( 'Ultimate Member: User Tags', 'um-mailchimp' ), array( &$this, 'metabox_content' ), 'toplevel_page_ultimatemember', 'core', 'core' );
	}

	/**
	 * Render metabox MailChimp
	 */
	public function metabox_content() {
		include_once um_user_tags_path . 'includes/admin/templates/dashboard.php';
	}

	/**
	 * Flush the tags counters.
	 */
	public function recalculate_tags() {
		global $wpdb;

		$terms      = get_terms(
			'um_user_tag',
			array(
				'hide_empty' => 0,
			)
		);
		$tags       = get_option( 'um_user_tags_filters', array() );
		$tags_names = array_keys( $tags );

		foreach ( $terms as $term ) {
			$meta_query = array(
				'relation' => 'OR',
			);

			foreach ( $tags_names as $tag ) {
				$meta_query[] = array(
					'key'     => $tag,
					'compare' => 'LIKE',
					'value'   => serialize( (string) $term->term_id ),
				);
				$meta_query[] = array(
					'key'     => $tag,
					'compare' => 'LIKE',
					'value'   => serialize( $term->term_id ),
				);
				$meta_query[] = array(
					'key'     => $tag,
					'compare' => 'LIKE',
					'value'   => serialize( $term->slug ),
				);
			}

			$users = get_users(
				array(
					'meta_query' => $meta_query,
					'fields'     => 'ids',
				)
			);
			$count = count( $users );

			$wpdb->update(
				$wpdb->term_taxonomy,
				array( 'count' => $count ),
				array( 'term_id' => $term->term_id )
			);
		}

		$url = add_query_arg(
			array(
				'page'   => 'ultimatemember',
				'update' => 'um_cleared_user_tags_cache',
			),
			admin_url( 'admin.php' )
		);
		wp_safe_redirect( $url );
		exit;
	}

	public function recalculate_tags_notice( $messages, $update ) {
		if ( 'um_cleared_user_tags_cache' === $update ) {
			$messages[0]['content'] = __( 'Your user tags cache is now removed.', 'um-user-tags' );
		}
		return $messages;
	}

	/**
	 * Change the User Tags table columns.
	 *
	 * @since 2.2.4
	 *
	 * @param array $column_headers The column header labels keyed by column ID.
	 *
	 * @return array
	 */
	public function user_tag_columns( $column_headers ) {
		if ( array_key_exists( 'posts', $column_headers ) ) {
			unset( $column_headers['posts'] );
		}
		$column_headers['um_count'] = __( 'Count', 'um-user-tags' );
		return $column_headers;
	}

	/**
	 * Filters the displayed columns in the User Tags table.
	 *
	 * @since 2.2.4
	 *
	 * @param string $string      Custom column output. Default empty.
	 * @param string $column_name Name of the column.
	 * @param int    $term_id     Term ID.
	 *
	 * @return string The count of users who selected this user tag.
	 */
	public function user_tag_custom_column( $string, $column_name, $term_id ) {
		if ( 'um_count' !== $column_name ) {
			return $string;
		}

		if ( ! UM()->options()->get( 'members_page' ) ) {
			return '-';
		}

		$user_tag_term = get_term( $term_id );
		if ( 0 === $user_tag_term->parent ) {
			return '-';
		}

		$user_tag_fields = get_option( 'um_user_tags_filters', array() );
		$user_tag_fields = array_flip( $user_tag_fields );
		if ( empty( $user_tag_fields ) || empty( $user_tag_fields[ $user_tag_term->parent ] ) ) {
			return '-';
		}

		$tag_field = $user_tag_fields[ $user_tag_term->parent ];

		$link = get_term_link( $user_tag_term, 'um_user_tag' );
		$link = add_query_arg( array( 'tag_field' => $tag_field ), $link );

		$count = number_format_i18n( $user_tag_term->count );
		if ( 0 < $count ) {
			$string = '<a href="' . esc_url( $link ) . '" target="_blank" title="' . esc_attr__( 'View members', 'um-user-tags' ) . '">' . esc_html( $count ) . '</a>';
		} else {
			$string = '0';
		}

		return $string;
	}

	/**
	 * Filters the sortable columns in the User Tags table.
	 *
	 * @since 2.2.4
	 *
	 * @param array $sortable_columns An array of sortable columns.
	 *
	 * @return array
	 */
	public function user_tag_sortable_columns( $sortable_columns ) {
		$sortable_columns['um_count'] = 'count';
		return $sortable_columns;
	}

	/**
	 * Creates options in Role page
	 *
	 * @param array $roles_metaboxes
	 *
	 * @return array
	 */
	public function add_role_metabox( $roles_metaboxes ) {
		$roles_metaboxes[] = array(
			'id'       => 'um-admin-form-user-tags{' . um_user_tags_path . '}',
			'title'    => __( 'User Tags', 'um-user-tags' ),
			'callback' => array( UM()->metabox(), 'load_metabox_role' ),
			'screen'   => 'um_role_meta',
			'context'  => 'normal',
			'priority' => 'default',
		);

		return $roles_metaboxes;
	}

	/**
	 * Add User Tags submenu
	 */
	public function um_extend_admin_menu() {
		add_submenu_page( 'ultimatemember', __( 'User Tags', 'um-user-tags' ), __( 'User Tags', 'um-user-tags' ), 'manage_options', 'edit-tags.php?taxonomy=um_user_tag', '' );
	}

	/**
	 * Extend settings
	 *
	 * @param array $settings
	 *
	 * @return mixed
	 */
	public function extend_settings( $settings ) {
		$settings['licenses']['fields'][] = array(
			'id'        => 'um_user_tags_license_key',
			'label'     => __( 'User Tags License Key', 'um-user-tags' ),
			'item_name' => 'User Tags',
			'author'    => 'Ultimate Member',
			'version'   => um_user_tags_version,
		);

		$forms_query        = new \WP_Query();
		$member_directories = $forms_query->query(
			array(
				'post_type'      => 'um_directory',
				'posts_per_page' => -1,
				'fields'         => array( 'ID', 'post_title' ),
			)
		);

		$directories = array(
			'' => __( '(None)', 'um-user-tags' ),
		);
		if ( ! empty( $member_directories ) && ! is_wp_error( $member_directories ) ) {
			foreach ( $member_directories as $directory ) {
				$directories[ $directory->ID ] = $directory->post_title;
			}
		}

		$fields = array(
			array(
				'id'          => 'user_tags_slug',
				'type'        => 'text',
				'label'       => __( 'User tag slug', 'um-user-tags' ),
				'tooltip'     => __( 'Base permalink for user tag', 'um-user-tags' ),
				'description' => __( 'Once this setting is changed you should update WordPress rewrite rules. Go to the page [Settings > Permalinks] and click the "Save Changes" button.', 'um-user-tags' ),
				'size'        => 'small',
			),
			array(
				'id'       => 'user_tags_max_num',
				'type'     => 'text',
				'label'    => __( 'Maximum number of tags to display in user profile', 'um-user-tags' ),
				'tooltip'  => __( 'Remaining tags will appear by clicking on a link', 'um-user-tags' ),
				'validate' => 'numeric',
				'size'     => 'small',
			),
		);

		if ( UM()->options()->get( 'members_page' ) ) {
			$fields[] = array(
				'id'      => 'user_tags_base_directory',
				'type'    => 'select',
				'label'   => __( 'Base member directory', 'um-user-tags' ),
				'tooltip' => __( 'Select base member directory to use its settings for displaying users with this tag', 'um-user-tags' ),
				'options' => $directories,
				'size'    => 'small',
			);
		}

		$settings['extensions']['sections']['user_tags'] = array(
			'title'  => __( 'User Tags', 'um-user-tags' ),
			'fields' => $fields,
		);

		return $settings;
	}

	/**
	 * Scan templates from extension
	 *
	 * @param $scan_files
	 *
	 * @return array
	 */
	public function um_user_tags_extend_scan_files( $scan_files ) {
		$extension_files['um-user-tags'] = UM()->admin_settings()->scan_template_files( um_user_tags_path . '/templates/' );
		$scan_files                      = array_merge( $scan_files, $extension_files );

		return $scan_files;
	}

	/**
	 * Get template paths
	 *
	 * @param $located
	 * @param $file
	 *
	 * @return array
	 */
	public function um_user_tags_get_path_template( $located, $file ) {
		if ( file_exists( get_stylesheet_directory() . '/ultimate-member/um-user-tags/' . $file ) ) {
			$located = array(
				'theme' => get_stylesheet_directory() . '/ultimate-member/um-user-tags/' . $file,
				'core'  => um_user_tags_path . 'templates/' . $file,
			);
		}

		return $located;
	}
}
