<?php
/**
 * Account Hooks
 *
 * @package UM_Social_Login_API
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Custom error
 *
 * @param string $msg Error message.
 * @param string $error_type Error Type.
 *
 * @return string
 */
function um_social_login_custom_error( $msg, $error_type ) {
	$providers = UM()->Social_Login_API()->available_networks();

	foreach ( $providers as $key => $info ) {
		if ( strstr( $error_type, $key ) && $error_type === $key . '_exist' ) {
			/* translators: %s is a social network name */
			$msg = sprintf( esc_attr__( ' This %s account is already linked to another user.', 'um-social-login' ), $info['name'] );
		} elseif ( strstr( $error_type, $key ) && $error_type === $key . '_linked' ) {
			/* translators: %s is a social network name */
			$msg = sprintf( esc_attr__( ' This %s account is already linked to your account.', 'um-social-login' ), $info['name'] );
		}
	}

	return $msg;
}
add_filter( 'um_custom_error_message_handler', 'um_social_login_custom_error', 10, 2 );

/**
 * Custom Success message in Account > Social
 *
 * @param string $msg Error message.
 * @param string $success_type Error Type.
 *
 * @return string
 */
function um_social_login_custom_success( $msg, $success_type ) {
	if ( 'social' === $success_type ) {
		 //phpcs:disable  WordPress.Security.NonceVerification
		if ( isset( $_REQUEST['return_provider'] ) ) {
			$provider = sanitize_key( $_REQUEST['return_provider'] );
		} elseif ( isset( $_REQUEST['linked_provider'] ) ) {
			$provider = sanitize_key( $_REQUEST['linked_provider'] );
		}
 		//phpcs:enable  WordPress.Security.NonceVerification

		$provider_name = UM()->Social_Login_API()->get_network_name( $provider );

		/* translators: %s is a social network name */
		$msg = sprintf( esc_attr__( 'Your %s account has been linked to your account.', 'um-social-login' ), $provider_name );
	}

	return $msg;
}
add_filter( 'um_custom_success_message_handler', 'um_social_login_custom_success', 10, 2 );

/**
 * Sync user profile photo
 *
 * @param string  $url Avatar URL.
 * @param integer $user_id User ID.
 *
 * @return mixed
 */
function um_social_login_synced_profile_photo( $url, $user_id ) {
	$synced_profile_photo_url = get_user_meta( $user_id, 'synced_profile_photo', true );
	if ( $synced_profile_photo_url ) {
		$url = $synced_profile_photo_url;
		// ssl enabled?
		if ( is_ssl() ) {
			$url = str_replace( 'http://', 'https://', $url );
		}
	}
	return $url;
}
add_filter( 'um_user_avatar_url_filter', 'um_social_login_synced_profile_photo', 100, 2 );

/**
 * Add tab to account page
 *
 * @param array $tabs Tabs settings.
 *
 * @return array
 */
function um_social_login_account_tab( $tabs ) {
	$tabs[450]['social']['icon']        = 'um-faicon-sign-in';
	$tabs[450]['social']['title']       = __( 'Social Connect', 'um-social-login' );
	$tabs[450]['social']['show_button'] = false;

	if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
		$tabs[450]['social']['custom'] = true;
	}

	return $tabs;
}
add_filter( 'um_account_page_default_tabs_hook', 'um_social_login_account_tab', 100 );

/**
 * Add content to account tab
 *
 * @param string $output HTML output.
 *
 * @return string
 */
function um_account_content_hook_social( $output ) {

	// important to only show available networks.
	$providers = UM()->Social_Login_API()->available_networks();

	if ( empty( $providers ) ) {
		return $output;
	}

	wp_enqueue_script( 'um-social-login' );
	wp_enqueue_style( 'um-social-login' );

	$user_id = get_current_user_id();

	ob_start(); ?>

	<div class="um-field  um-social-login-btns" data-key="">

		<?php foreach ( $providers as $provider => $array ) { ?>

			<?php
			if ( ! isset( $array['icon_position'] ) ) {
				$array['icon_position'] = 'left';
			}
			?>
			<div class="um-provider">
				<div class="um-provider-title">
					<?php /* translators: %s is a social network name */ ?>
					<?php echo esc_attr( sprintf( __( 'Connect to %s', 'um-social-login' ), $array['name'] ) ); ?>
					<?php do_action( 'um_social_login_after_provider_title', $provider, $array ); ?>
				</div>

				<div class="um-provider-conn">

					<?php if ( UM()->Social_Login_API()->user_connect()->is_connected( $user_id, $provider ) ) { ?>

						<div class="um-provider-user-photo"><a href="<?php echo esc_url( UM()->Social_Login_API()->user( $user_id )->get_profile_url( $provider ) ); ?>" target="_blank" title="<?php echo esc_attr( UM()->Social_Login_API()->user( $user_id )->get_display_name( $provider ) ); ?>"><img src="<?php echo esc_url( UM()->Social_Login_API()->user( $user_id )->get_profile_photo( $provider ) ); ?>" class="um-provider-photo small" /></a></div>

						<div class="um-provider-user-handle"><a href="<?php echo esc_url( UM()->Social_Login_API()->user( $user_id )->get_profile_url( $provider ) ); ?>" target="_blank"><?php echo esc_attr( UM()->Social_Login_API()->user( $user_id )->get_display_name( $provider ) ); ?></a></div>

						<div class="um-provider-disconnect">(<a href="<?php echo esc_url( UM()->Social_Login_API()->hybridauth()->get_disconnect_url( $provider ) ); ?>"><?php esc_attr_e( 'Disconnect', 'um-social-login' ); ?></a>)</div>

						<span class="um-provider-date-linked">
							<?php $date_linked = UM()->Social_Login_API()->user( $user_id )->get_date_linked( $provider ); ?>
							<?php echo esc_attr( human_time_diff( strtotime( $date_linked ), gmdate( 'U' ) ) ); ?> <?php esc_attr_e( 'ago', 'um-social-login' ); ?>
						</span>

					<?php } else { ?>
						<?php /* translators: %s is a social network name */ ?>
						<a title="<?php echo esc_attr( sprintf( __( 'Connect to %s', 'um-social-login' ), $array['name'] ) ); ?>" href="<?php echo esc_url( UM()->Social_Login_API()->hybridauth()->get_connect_url( $provider ) ); ?>" class="um-social-btn um-social-btn-<?php echo esc_attr( $provider ); ?>" onclick="um_social_login_oauth_window( this.href,'authWindow', 'width=600,height=600,scrollbars=yes' );return false;">
							<?php
							if ( 'left' === $array['icon_position'] ) {
								?>
									<i class="<?php echo esc_attr( $array['icon'] ); ?>"></i>
									<?php /* translators: %s is a social network name */ ?>
									<span> <?php echo esc_attr( sprintf( __( 'Connect to %s', 'um-social-login' ), $array['name'] ) ); ?></span>
								<?php
							}
							?>
							<?php
							if ( 'right' === $array['icon_position'] ) {
								?>
									<span> <?php echo esc_attr( __( 'Connect to ', 'um-social-login' ) ); ?></span>
									<i class="<?php echo esc_attr( $array['icon'] ); ?>"></i>
								<?php
							}
							?>
						</a>

					<?php } ?>

				</div>

				<div class="um-clear"></div>

			</div>

		<?php } ?>

		<style type="text/css">
			<?php foreach ( $providers as $provider => $arr ) { ?>
				.um-social-btn.um-social-btn-<?php echo esc_attr( $provider ); ?> {background-color: <?php echo esc_attr( $arr['bg'] ); ?>!important}
				.um-social-btn.um-social-btn-<?php echo esc_attr( $provider ); ?>:hover {background-color: <?php echo esc_attr( $arr['bg_hover'] ); ?>!important}
				.um-social-btn.um-social-btn-<?php echo esc_attr( $provider ); ?> {color: <?php echo esc_attr( $arr['color'] ); ?>!important}
			<?php } ?>
		</style>

	</div>

	<?php
	$output .= ob_get_clean();
	return $output;
}
add_filter( 'um_account_content_hook_social', 'um_account_content_hook_social' );
