<?php
namespace um_ext\um_social_activity\core;

use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Activity_Main_API
 *
 * @package um_ext\um_social_activity\core
 */
class Activity_Main_API {

	/**
	 * @var array
	 */
	public $global_actions = array();

	/**
	 * Activity_Main_API constructor.
	 */
	public function __construct() {
		add_filter( 'init', array( &$this, 'init_global_actions' ) );

		add_filter( 'um_profile_tabs', array( &$this, 'add_tab' ), 5, 1 );
		add_filter( 'um_user_profile_tabs', array( &$this, 'add_user_tab' ), 5, 1 );
		add_filter( 'pre_kses', array( &$this, 'allow_get_params' ), 10, 1 );
		add_action( 'um_profile_content_activity', array( &$this, 'show_wall' ) );

		add_filter( 'um_image_upload_validation', array( &$this, 'wall_uploader_validation' ), 10, 2 );
	}

	/**
	 * @param null|bool $custom_validation
	 * @param string    $id
	 *
	 * @return null|bool
	 */
	public function wall_uploader_validation( $custom_validation, $id ) {
		if ( ! isset( $_POST['set_mode'] ) ) {
			return $custom_validation;
		}

		$mode = sanitize_key( $_POST['set_mode'] );
		if ( 'wall_img_upload' === $id && 'wall' === $mode ) {
			/** This filter is documented in includes/core/class-files.php */
			$um_image_upload_nonce = apply_filters( 'um_image_upload_nonce', true );
			if ( $um_image_upload_nonce ) {
				$timestamp = absint( $_POST['timestamp'] );
				$nonce     = sanitize_text_field( $_POST['_wpnonce'] );
				if ( ! wp_verify_nonce( $nonce, "um_upload_nonce-{$timestamp}" ) && is_user_logged_in() ) {
					// This nonce is not valid.
					$ret['error'] = esc_html__( 'Invalid nonce', 'um-activity' );
					wp_send_json_error( $ret );
				}
			}

			// Ignore UM core image upload validation
			$custom_validation = true;
		}

		return $custom_validation;
	}

	public function init_global_actions() {
		$actions = array(
			'status'               => __( 'New wall post', 'um-activity' ),
			'new-user'             => __( 'New user', 'um-activity' ),
			'new-post'             => __( 'New blog post', 'um-activity' ),
			'new-product'          => __( 'New product', 'um-activity' ),
			'new-gform'            => __( 'New Gravity Form', 'um-activity' ),
			'new-gform-submission' => __( 'New Gravity Form Answer', 'um-activity' ),
			'new-follow'           => __( 'New follow', 'um-activity' ),
		);

		$this->global_actions = apply_filters( 'um_activity_global_actions', $actions );
	}

	/**
	 * API to automate activity posts
	 *
	 * @param array $array
	 * @param bool $update_post
	 * @param null $update_post_id
	 *
	 * @return int|null|\WP_Error
	 */
	public function save( $array = array(), $update_post = false, $update_post_id = null ) {
		$args = array(
			'post_title'  => '',
			'post_type'   => 'um_activity',
			'post_status' => 'publish',
			'post_author' => $array['author'],
		);

		ob_start();

		$file       = ( isset( $array['custom_path'] ) ) ? $array['custom_path'] : um_activity_path . 'templates/html/' . $array['template'] . '.php';
		$theme_file = get_stylesheet_directory() . '/ultimate-member/templates/activity/' . $array['template'] . '.php';
		if ( file_exists( $theme_file ) ) {
			$file = $theme_file;
		}
		if ( file_exists( $file ) ) {
			include $file;
		}
		$args['post_content'] = ob_get_clean();

		$search = array(
			'{author_name}',
			'{author_profile}',
			'{user_name}',
			'{user_profile}',
			'{user_photo}',
			'{post_title}',
			'{post_url}',
			'{post_excerpt}',
			'{post_image}',
			'{price}',
		);
		$search = apply_filters( 'um_activity_search_tpl', $search );

		$replace_def = array(
			'author_name'    => '',
			'author_profile' => '',
			'user_name'      => '',
			'user_profile'   => '',
			'user_photo'     => '',
			'post_title'     => '',
			'post_url'       => '',
			'post_excerpt'   => '',
			'post_image'     => '',
			'price'          => '',
		);

		$replace = array_merge( $replace_def, array_intersect_key( $array, $replace_def ) );
		$replace = apply_filters( 'um_activity_replace_tpl', $replace, $array );

		$args['post_content'] = str_replace( $search, $replace, $args['post_content'] );

		$args['post_content'] = html_entity_decode( strip_shortcodes( trim( $args['post_content'] ) ) );

		// Update post content
		if ( $update_post && $update_post_id ) {
			$args['ID']         = $update_post_id;
			$args['post_title'] = $array['post_title'];
			$post_id            = wp_update_post( $args );
		} else { // Add a new post
			$post_id = wp_insert_post( $args );
			wp_update_post(
				array(
					'ID'         => $post_id,
					'post_title' => $post_id,
					'post_name'  => $post_id,
				)
			);

			update_post_meta( $post_id, '_wall_id', $array['wall_id'] );
			update_post_meta( $post_id, '_action', $array['template'] );
			update_post_meta( $post_id, '_likes', 0 );
			update_post_meta( $post_id, '_comments', 0 );
		}

		if ( isset( $array['author'] ) ) {
			update_post_meta( $post_id, '_user_id', absint( $array['author'] ) );
		}
		if ( isset( $array['related_id'] ) ) {
			update_post_meta( $post_id, '_related_id', absint( $array['related_id'] ) );
		}

		do_action( 'um_social_activity_post_saved', $post_id );

		return $post_id;
	}

	/**
	 * Grab followed user IDs
	 *
	 * @return array|null
	 */
	public function followed_ids() {
		$array = array();

		if ( ! $this->followed_activity() ) {
			return null;
		}

		if ( ! is_user_logged_in() ) {
			return array( 0 );
		}

		$array[] = get_current_user_id();

		$following = UM()->Followers_API()->api()->following( get_current_user_id() );
		if ( $following ) {
			$array = array_merge( $array, $following );
		}

		if ( isset( $array ) ) {
			return $array;
		}

		return null;
	}

	/***
	 ***    @Check if enabled followed activity only
	 ***/
	public function followed_activity() {
		if ( class_exists( 'UM_Followers_API' ) && UM()->options()->get( 'activity_followed_users' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Return to activity post after login
	 *
	 * @param $post_id
	 *
	 * @return string
	 */
	public function login_to_interact( $post_id = null ) {

		if ( UM()->is_request( 'ajax' ) ) {
			$curr_page = wp_get_referer();
		} else {
			$curr_page = UM()->permalinks()->get_current_url();
		}
		if ( ! empty( $post_id ) ) {
			$curr_page = add_query_arg( 'wall_post', $post_id, $curr_page );
		}

		$pattern = stripslashes( UM()->options()->get( 'activity_need_to_login' ) );

		$text = str_replace(
			array(
				'{current_page}',
				'{login_page}',
				'{register_page}',
			),
			array(
				$curr_page,
				add_query_arg( 'redirect_to', $curr_page, um_get_core_page( 'login' ) ),
				add_query_arg( 'redirect_to', $curr_page, um_get_core_page( 'register' ) ),
			),
			$pattern
		);

		return $text;
	}

	/**
	 * Add Profile Tab
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function add_tab( $tabs ) {
		$tabs['activity'] = array(
			'name' => __( 'Activity', 'um-activity' ),
			'icon' => 'um-icon-compose',
		);

		return $tabs;
	}


	/**
	 * Hide Profile Tab if no capabilities
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function add_user_tab( $tabs ) {
		if ( empty( $tabs['activity'] ) ) {
			return $tabs;
		}

		if ( um_user( 'activity_wall_off' ) ) {
			unset( $tabs['activity'] );
		}

		return $tabs;
	}


	/***
	 ***    @get comment content
	 ***/
	public function commentcontent( $content ) {
		$content = convert_smilies( $content );
		//$content = preg_replace('$(\s|^)(https?://[a-z0-9_./?=&-]+)(?![^<>]*>)$i', ' <a class="um-link" href="$2" target="_blank" rel="nofollow">$2</a> ', $content." ");
		//$content = preg_replace('$(\s|^)(www\.[a-z0-9_./?=&-]+)(?![^<>]*>)$i', '<a class="um-link" target="_blank" href="http://$2"  target="_blank" rel="nofollow">$2</a> ', $content." ");
		$content = $this->make_links_clickable( $content );
		$content = $this->hashtag_links( $content );

		return $content;
	}

	/***
	 ***    @shorten any string based on word count
	 ***/
	public function shorten_string( $string ) {
		$retval        = $string;
		$wordsreturned = UM()->options()->get( 'activity_post_truncate' );
		if ( ! $wordsreturned ) {
			return $string;
		}
		$array = explode( ' ', $string );
		if ( count( $array ) <= $wordsreturned ) {
			$retval = $string;
		} else {
			$res    = array_splice( $array, $wordsreturned );
			$retval = implode( ' ', $array ) . ' <span class="um-activity-seemore">(<a href="" class="um-link">' . esc_html__( 'See more', 'um-activity' ) . '</a>)</span> <span class="um-activity-hiddentext">' . implode( ' ', $res ) . '</span>';
		}

		return $retval;
	}

	/**
	 * Can edit a user comment.
	 *
	 * @param int $comment_id
	 * @param int $user_id
	 *
	 * @return bool
	 */
	public function can_edit_comment( $comment_id, $user_id ) {
		if ( ! $user_id ) {
			return false;
		}
		$comment = get_comment( $comment_id );

		return absint( $comment->user_id ) === absint( $user_id );
	}

	/**
	 * Get a summarized content length
	 *
	 * @param int $post_id
	 *
	 * @return string
	 */
	public function get_content( $post_id = 0 ) {
		if ( empty( $post_id ) ) {
			$loop_post_id = get_the_ID();
			if ( empty( $loop_post_id ) ) {
				return '';
			}

			$post_id = $loop_post_id;
		}

		$post = get_post( $post_id );
		if ( empty( $post ) ) {
			return '';
		}
		$content = $post->post_content;

		$has_oembed  = get_post_meta( $post_id, '_oembed', true );
		$shared_link = get_post_meta( $post_id, '_shared_link', true );
		$video_url   = get_post_meta( $post_id, '_video_url', true );

		if ( $has_oembed ) {
			$content = str_replace( $has_oembed, '', $content );
		}
		if ( $shared_link ) {
			$content = str_replace( $shared_link, '', $content );
		}
		if ( $video_url ) {
			$content = str_replace( $video_url, '', $content );
		}

		$content = trim( $content );
		if ( '' === $content ) {
			return '';
		}

		if ( 'status' === $this->get_action_type( $post_id ) ) {
			$content = $this->shorten_string( $content );
		}
		$content = $this->make_links_clickable( $content );
		$content = $this->hashtag_links( $content );

		// strip avatars
		if ( preg_match( '/\<img src=\"([^\"]+)\" class="(gr)?avatar/', $content, $matches ) ) {
			$src   = $matches[1];
			$found = @getimagesize( $src );
			if ( ! $found ) {
				$content = str_replace( $src, um_get_default_avatar_uri(), $content );
			}
		}

		$content = $this->remove_vc_from_excerpt( $content );

		if ( $has_oembed ) {
			$content .= $has_oembed;
		}

		$author_id = $this->get_author( $post_id );
		if ( $author_id ) {
			$author_data = get_userdata( $author_id );

			if ( ! empty( $author_data ) ) {
				$search = array(
					'{author_name}',
					'{author_profile}',
				);

				$replace = array(
					$author_data->display_name,
					um_user_profile_url( $author_id ),
				);

				$content = nl2br( str_replace( $search, $replace, $content ) );
			}
		}

		// Replace emojis codes
		$content = convert_smilies( $content );
		if ( isset( UM()->shortcodes()->emoji ) ) {
			$content = UM()->shortcodes()->emotize( $content );
		}

		// Add related image if no image
		if ( ! strpos( $content, '<span class="post-image">' ) ) {
			$related_id = get_post_meta( $post_id, '_related_id', true );
			if ( ! empty( $related_id ) ) {
				$post_image_url = $this->get_post_image_url( $related_id );
				if ( $post_image_url ) {
					$post_image = '<span class="post-image"><img src="' . esc_url( $post_image_url ) . '" alt="' . esc_attr( basename( $post_image_url ) ) . '" title="#' . esc_attr( get_the_title( $related_id ) ) . '" class="um-activity-featured-img" /></span>';
					$content    = str_replace( '<span class="post-title">', $post_image . '<span class="post-title">', $content );
				}
			}
		}

		$content = apply_filters( 'um-activity-post-content', $content, $post );

		/**
		 * Filter change content for user wall.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_post_content
		 *
		 * @param {string}  $content  post content.
		 * @param {object}  $post     post object.
		 *
		 * @example <caption>Change post content.</caption>
		 * function my_um_activity_get_photo_content( $content, $post ) {
		 *     // your code here
		 *    return $content;
		 * }
		 * add_filter( 'um_activity_post_content', 'my_um_activity_post_content', 10, 2 );
		 */
		return apply_filters( 'um_activity_post_content', $content, $post );
	}

	/***
	 ***    @Get content link
	 ***/
	public function get_content_link( $content ) {

		$arr_urls = wp_extract_urls( $content );
		if ( isset( $arr_urls ) && ! empty( $arr_urls ) ) {
			foreach ( $arr_urls as $key => $url ) {
				if ( ! strstr( $url, 'vimeo' ) && ! strstr( $url, 'youtube' ) && ! strstr( $url, 'youtu.be' ) ) {

					/**
					 * Filter change content link.
					 *
					 * @since 2.3.6
					 *
					 * @hook um_activity_content_link
					 *
					 * @param {string}  $url      content link.
					 * @param {string } $content  content.
					 *
					 * @example <caption>Change content link.</caption>
					 * function my_um_activity_content_link( $url, $content ) {
					 *     // your code here
					 *    return $url;
					 * }
					 * add_filter( 'um_activity_content_link', 'my_um_activity_content_link', 10, 2 );
					 */
					$url = apply_filters( 'um_activity_content_link', $url, $content );
					return $url;
				}
			}
		}

		return null;
	}

	/**
	 * Check if URL is oEmbed supported
	 *
	 * @param $url
	 *
	 * @return bool|false|string
	 */
	public function is_oembed( $url ) {
		if ( empty( $url ) ) {
			return false;
		}

		$providers = array(
			'mixcloud.com'   => array( 'height' => 200 ),
			'soundcloud.com' => array( 'height' => 200 ),
			'instagram.com'  => array(
				'height' => 500,
				'width'  => 500,
			),
			'twitter.com'    => array(
				'height' => 500,
				'width'  => 700,
			),
			't.co'           => array(
				'height' => 500,
				'width'  => 700,
			),
		);

		$providers = apply_filters( 'um_activity_oembed_providers', $providers );
		foreach ( $providers as $provider => $size ) {
			if ( false !== strpos( $url, $provider ) ) {
				return wp_oembed_get( $url, $size );
			}
		}

		return false;
	}

	/**
	 * Set url meta
	 *
	 * @param $url
	 * @param $post_id
	 *
	 * @return string
	 */
	public function set_url_meta( $url, $post_id ) {

		$request = wp_remote_get( $url );

		// Try to get remote page using request with headers if simple request fails
		if ( ! is_array( $request ) || empty( $request['response'] ) || empty( $request['response']['code'] ) || 200 !== $request['response']['code'] ) {
			$user_agent = empty( $_SERVER['HTTP_USER_AGENT'] ) ? 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/85.0.4183.121 Safari/537.36' : $_SERVER['HTTP_USER_AGENT'];

			$request = wp_remote_get(
				$url,
				array(
					'headers' => array(
						'accept'                    => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
						'accept-encoding'           => 'gzip, deflate, br',
						'accept-language'           => 'en-US,en;q=0.5',
						'cache-control'             => 'max-age=0',
						'upgrade-insecure-requests' => 1,
						'user-agent'                => $user_agent,
					),
				)
			);
		}

		$response = wp_remote_retrieve_body( $request );

		$html   = new \DOMDocument();
		$source = mb_convert_encoding( $response, 'HTML-ENTITIES', 'UTF-8' );
		if ( empty( $source ) ) {
			return '';
		}

		@$html->loadHTML( $source );
		$tags = null;

		$title         = $html->getElementsByTagName( 'title' );
		$tags['title'] = $title->item( 0 )->nodeValue;

		foreach ( $html->getElementsByTagName( 'meta' ) as $meta ) {
			if ( 'og:image' === $meta->getAttribute( 'property' ) ) {
				$tags['image'] = $src = trim( str_replace( '\\', '/', $meta->getAttribute( 'content' ) ) );
				$data          = $this->is_image( $src );
				if ( is_array( $data ) ) {
					$tags['image']        = $src;
					$tags['image_width']  = $data[0];
					$tags['image_height'] = $data[1];
				}
			}
			if ( 'og:image:width' === $meta->getAttribute( 'property' ) ) {
				$tags['image_width'] = trim( $meta->getAttribute( 'content' ) );
			}
			if ( 'og:image:height' === $meta->getAttribute( 'property' ) ) {
				$tags['image_height'] = trim( $meta->getAttribute( 'content' ) );
			}
			if ( 'description' === $meta->getAttribute( 'name' ) ) {
				$tags['description'] = trim( str_replace( '\\', '/', $meta->getAttribute( 'content' ) ) );
			}
		}

		if ( ! isset( $tags['image'] ) ) {
			foreach ( $html->getElementsByTagName( 'img' ) as $img ) {
				$src = esc_url( $img->getAttribute( 'src' ) );
				if ( false !== strpos( $src, '\\' ) ) {
					$src = str_replace( '\\', '/', $src );
				}
				if ( 0 === strpos( $src, '//' ) ) {
					$src = 'http:' . $src;
				}
				$tags['image'] = $src;
				$data          = $this->is_image( $src );
				if ( is_array( $data ) ) {
					$tags['image_width']  = $data[0];
					$tags['image_height'] = $data[1];
					break;
				}
			}
		}

		/* Display the meta now */

		if ( isset( $tags['image_width'] ) && $tags['image_width'] <= 400 ) {
			$content = '<span class="post-meta" style="position:relative;min-height: ' . ( absint( $tags['image_height'] / 2 ) - 10 ) . 'px;padding-left:' . $tags['image_width'] / 2 . 'px;"><a href="{post_url}" target="_blank">{post_image} {post_title} {post_excerpt} {post_domain}</a></span>';
		} else {
			$content = '<span class="post-meta"><a href="{post_url}" target="_blank">{post_image} {post_title} {post_excerpt} {post_domain}</a></span>';
		}

		if ( isset( $tags['description'] ) ) {
			if ( isset( $tags['image_width'] ) && 400 >= $tags['image_width'] ) {
				$content = str_replace( '{post_excerpt}', '', $content );
			} else {
				$content = str_replace( '{post_excerpt}', '<span class="post-excerpt">' . $tags['description'] . '</span>', $content );
			}
		} else {
			$content = str_replace( '{post_excerpt}', '', $content );
		}

		if ( isset( $tags['title'] ) ) {
			$content = str_replace( '{post_title}', '<span class="post-title">' . mb_convert_encoding( $tags['title'], 'HTML-ENTITIES', 'UTF-8' ) . '</span>', $content );
		} else {
			$content = str_replace( '{post_title}', '<span class="post-title">' . esc_html__( 'Untitled', 'um-activity' ) . '</span>', $content );
		}

		if ( isset( $tags['image'] ) ) {
			if ( isset( $tags['image_width'] ) && 400 >= $tags['image_width'] ) {
				$content = str_replace( '{post_image}', '<span class="post-image" style="position:absolute;left:0;top:0;width:' . $tags['image_width'] / 2 . 'px;"><img src="' . $tags['image'] . '" alt="" title="" class="um-activity-featured-img" /></span>', $content );
			} else {
				$content = str_replace( '{post_image}', '<span class="post-image"><img src="' . $tags['image'] . '" alt="" title="" class="um-activity-featured-img" /></span>', $content );
			}
		} else {
			$content = str_replace( '{post_image}', '', $content );
		}

		$parse = wp_parse_url( $url );

		$content = str_replace( '{post_url}', $url, $content );
		$content = str_replace( '{post_domain}', '<span class="post-domain">' . strtoupper( $parse['host'] ) . '</span>', $content );

		update_post_meta( $post_id, '_shared_link', trim( $content ) );

		return trim( $content );
	}

	/***
	 ***    @Checks if image is valid
	 ***/
	public function is_image( $url ) {

		$allow_types = array(
			'jpeg' => 'image/jpeg',
		);

		/**
		 * UM hook
		 *
		 * @type filter
		 * @title um_allow_mime
		 * @description Extend mime types for images
		 * @input_vars
		 * [{"var":"$allow_types","type":"array","desc":"Allowed Types"}]
		 * @change_log
		 * ["Since: 2.1.8"]
		 * @usage add_filter( 'um_allow_mime', 'function_name', 10, 1 );
		 * @example
		 * <?php
		 * add_filter( 'um_allow_mime', 'my_um_allow_mime', 10, 1 );
		 * function my_um_allow_mime( $allow_types ) {
		 *     // your code here
		 *     return $allow_types;
		 * }
		 * ?>
		 */
		$allow_types = apply_filters( 'um_allow_mime', $allow_types );

		$filetype = wp_check_filetype( $url );
		if ( ! in_array( $filetype['type'], $allow_types, true ) ) {
			return 0;
		}

		$size = @getimagesize( $url );
		if ( ! is_array( $size ) ) {
			return 0;
		}

		if ( isset( $size['mime'] ) && strstr( $size['mime'], 'image' ) && in_array( $size['mime'], $allow_types ) && isset( $size[0] ) && absint( $size[0] ) > 100 && isset( $size[1] ) && ( $size[0] / $size[1] >= 1 ) && ( $size[0] / $size[1] <= 3 ) ) {
			return $size;
		}

		return 0;
	}


	/**
	 * Convert hashtags
	 *
	 * @param $content
	 *
	 * @return mixed
	 */
	public function hashtag_links( $content ) {
		// hashtag must have space or start line before and space or end line after. Hashtag can contain digits, letters, underscore. Not space or dash "-".
		$content = preg_replace_callback( '/(^|\s)(#([\p{Pc}\p{N}\p{L}\p{Mn}]+))/um', array( $this, 'hashtag_replace_links_cb' ), $content );
		return $content;
	}


	/**
	 * @param array $matches
	 *
	 * @return string
	 */
	public function hashtag_replace_links_cb( $matches ) {
		return $matches[1] . '<strong onclick="window.location.assign(\'' . add_query_arg( 'hashtag', $matches[3], um_get_core_page( 'activity' ) ) . '\'); return false;" class="um-link">' . $matches[2] . '</strong>';
	}


	/**
	 * Add hashtags
	 *
	 * @param int $post_id
	 * @param string $content
	 * @param bool $append
	 */
	public function hashtagit( $post_id, $content, $append = false ) {
		// hashtag must have space or start line before and space or end line after. Hashtag can contain digits, letters, underscore. Not space or dash "-".
		preg_match_all( '/(^|\s)#([\p{Pc}\p{N}\p{L}\p{Mn}]+)/um', $content, $matches, PREG_SET_ORDER, 0 );

		$terms = array();
		if ( isset( $matches[0] ) && is_array( $matches[0] ) ) {
			foreach ( $matches as $match ) {
				if ( isset( $match[2] ) ) {
					$terms[] = $match[2];
				}
			}
		}

		wp_set_post_terms( $post_id, $terms, 'um_hashtag', $append );
	}


	/**
	 * Get a possible photo
	 *
	 * @param int $post_id
	 * @param string $class
	 * @param null|int $author_id
	 *
	 * @return string
	 */
	public function get_photo( $post_id = 0, $class = '', $author_id = null ) {
		$photo_url = $this->get_download_link( $post_id, $author_id );
		if ( empty( $photo_url ) ) {
			return '';
		}
		$photo_url = esc_attr( $photo_url );

		$content = '';
		if ( 'backend' === $class ) {
			$uri = get_post_meta( $post_id, '_photo', true );
			if ( ! $uri ) {
				return '';
			}
			$uri           = wp_basename( $uri );
			$user_base_dir = UM()->uploader()->get_upload_user_base_dir( $author_id );

			if ( file_exists( $user_base_dir . DIRECTORY_SEPARATOR . $uri ) ) {
				$content = "<a href=\"{$photo_url}\" target=\"_blank\"><img src=\"{$photo_url}\" alt=\"\" style=\"width: 100%;\" /></a>";
			}
		} else {
			$content = "<a href=\"#\" class=\"um-photo-modal\" data-src=\"{$photo_url}\"><img src=\"{$photo_url}\" alt=\"\" /></a>";
		}

		/**
		 * Filter change photo content for user wall.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_get_photo_content
		 *
		 * @param {string} $content    Photo content.
		 * @param {int}    $post_id    activity post id.
		 * @param {string} $class      Class name.
		 * @param {int}    $author_id  author id.
		 *
		 * @example <caption>Direct URL to a photo.</caption>
		 * function my_um_activity_get_photo_content( $content, $post_id, $class, $author_id ) {
		 *     // your code here
		 *    $photo_url = get_post_meta( $wall_post_id, '_photo', true );
		 *    $photo_url = home_url( "/wp-content/uploads/ultimatemember/{$author_id}/{$photo_url}" );
		 *    $photo_url = esc_attr( $photo_url );
		 *    $content   = "<a href=\"#\" class=\"um-photo-modal\" data-src=\"{$photo_url}\"><img src=\"{$photo_url}\" alt=\"\" /></a>";
		 *    return $content;
		 * }
		 * add_filter( 'um_activity_get_photo_content', 'my_um_activity_get_photo_content', 10, 4 );
		 */
		return apply_filters( 'um_activity_get_photo_content', $content, $post_id, $class, $author_id );
	}

	/**
	 * @param int $post_id
	 * @param int $author_id
	 *
	 * @return string
	 */
	public function get_download_link( $post_id, $author_id ) {
		$uri = get_post_meta( $post_id, '_photo', true );
		if ( ! $uri ) {
			return '';
		}

		$uri      = wp_basename( $uri );
		$userdir  = UM()->uploader()->get_upload_user_base_dir( $author_id );
		$filename = wp_normalize_path( "$userdir/$uri" );
		if ( ! file_exists( $filename ) ) {
			return '';
		}
		$filetype = wp_check_filetype( $filename );
		$filetime = filemtime( $filename );

		if ( UM()->is_permalinks ) {
			$nonce = wp_create_nonce( $author_id . $post_id . 'um-download-nonce' );
			$url   = home_url( "/um-activity-download/{$post_id}/{$author_id}/{$nonce}/{$filetime}.{$filetype['ext']}" );
		} else {
			$nonce = wp_create_nonce( $author_id . $post_id . 'um-download-nonce' );
			$url   = add_query_arg(
				array(
					'um_action'   => 'um-activity-download',
					'um_post'     => $post_id,
					'um_author'   => $author_id,
					'um_verify'   => $nonce,
					'um_filename' => $filetime . '.' . $filetype['ext'],
				),
				home_url()
			);
		}

		/**
		 * Filter change photo download URL.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_get_download_link
		 *
		 * @param {string} $url        photo URL.
		 * @param {int}    $post_id    activity post id.
		 * @param {int}    $author_id  author id.
		 *
		 * @example <caption>Direct URL to a photo.</caption>
		 * function my_um_activity_get_download_link( $url, $post_id, $author_id ) {
		 *     // your code here
		 *    $url = get_post_meta( $wall_post_id, '_photo', true );
		 *    $url = home_url( "/wp-content/uploads/ultimatemember/{$author_id}/{$photo_url}" );
		 *    $url = esc_attr( $photo_url );
		 *    return $url;
		 * }
		 * add_filter( 'um_activity_get_download_link', 'my_um_activity_get_download_link', 10, 3 );
		 */
		$url = apply_filters( 'um_activity_get_download_link', $url, $post_id, $author_id );

		return $url;
	}

	/**
	 * Get a possible video
	 *
	 * @param int $post_id
	 * @param array $args
	 *
	 * @return false|string
	 */
	public function get_video( $post_id = 0, $args = array() ) {
		$uri = get_post_meta( $post_id, '_video_url', true );
		if ( ! $uri ) {
			return '';
		}

		$content = wp_oembed_get( $uri, $args );
		/**
		 * Filter change video content.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_get_video
		 *
		 * @param {string} $content Video content.
		 * @param {int}    $post_id activity post id.
		 * @param {array}  $args    args.
		 *
		 * @example <caption>Change video content.</caption>
		 * function my_um_activity_get_video( $content, $post_id, $args ) {
		 *    // your code here
		 *    return $content;
		 * }
		 * add_filter( 'um_activity_get_video', 'my_um_activity_get_video', 10, 3 );
		 */
		$content = apply_filters( 'um_activity_get_video', $content, $post_id, $args );

		return $content;
	}

	/**
	 * Strip video URLs as we need to convert them.
	 *
	 * @param string $content
	 * @param int    $post_id
	 */
	private function setup_video( $content, $post_id ) {
		$urls = wp_extract_urls( $content );

		if ( ! empty( $urls ) ) {
			foreach ( $urls as $url ) {
				$oembed        = new \WP_oEmbed();
				$provider_data = $oembed->get_data( $url );
				if ( ! empty( $provider_data ) && in_array( $provider_data->provider_name, array( 'YouTube', 'Vimeo' ), true ) ) {
					$videos[]['url']         = trim( $url );
					$videos[]['oembed_data'] = wp_json_encode( $provider_data );
				}
			}
		}

		if ( isset( $videos ) ) {
			$content = str_replace( $videos[0]['url'], '', $content );
			update_post_meta( $post_id, '_video_url', $videos[0]['url'] );
			if ( isset( $videos[0]['oembed_data'] ) ) {
				update_post_meta( $post_id, '_video_oembed_data', $videos[0]['oembed_data'] );
			}
		} else {
			delete_post_meta( $post_id, '_video_url' );
			delete_post_meta( $post_id, '_video_oembed_data' );
		}
	}

	/**
	 * Can post on that wall.
	 *
	 * @return mixed|null
	 */
	public function can_write() {
		$res = 1;

		if ( um_user( 'activity_wall_off' ) ) {
			$res = 0;
		}

		if ( UM()->roles()->um_user_can( 'activity_posts_off' ) ) {
			$res = 0;
		}

		if ( ! is_user_logged_in() ) {
			$res = 0;
		}

		return apply_filters( 'um_activity_can_post_on_wall', $res );
	}

	/**
	 * Can comment current user on wall
	 *
	 * @return bool
	 */
	public function can_comment() {
		$res = true;

		if ( UM()->roles()->um_user_can( 'activity_comments_off' ) ) {
			$res = false;
		}

		if ( ! is_user_logged_in() ) {
			$res = false;
		}

		return apply_filters( 'um_activity_can_post_comment_on_wall', $res );
	}

	/**
	 * User Profile Activity Wall
	 */
	public function show_wall() {
		$profile_id = um_profile_id();
		$can_view   = $this->can_view_wall( um_profile_id() );

		UM()->Activity_API()->enqueue()->enqueue_scripts();

		if ( true === $can_view ) {
			echo apply_shortcodes( '[ultimatemember_wall user_id="' . $profile_id . '"]' );
		} else {
			?>
			<div class="um-profile-note">
				<span>
					<i class="um-faicon-lock"></i>
					<?php echo esc_html( $can_view ); ?>
				</span>
			</div>
			<?php
		}
	}


	/**
	 * @param int $profile_id
	 *
	 * @return bool|string
	 */
	public function can_view_wall( $profile_id ) {
		$can_view = true;

		if ( ! UM()->options()->get( 'activity_enable_privacy' ) ) {
			return $can_view;
		}

		$privacy = get_user_meta( $profile_id, 'wall_privacy', true );

		if ( ! is_user_logged_in() ) {
			if ( UM()->options()->get( 'activity_require_login' ) ) {
				$can_view = __( 'You must login to view this user activity', 'um-activity' );
			} elseif ( 1 === absint( $privacy ) ) {
				$can_view = __( 'Please login to view this user\'s activity', 'um-activity' );
			} elseif ( 2 === absint( $privacy ) ) {
				$can_view = __( 'This user wall is private', 'um-activity' );
			}
		} else {
			if ( absint( $profile_id ) !== get_current_user_id() && 2 === absint( $privacy ) ) {
				$can_view = __( 'This user wall is private', 'um-activity' );
			}
		}

		return apply_filters( 'um_wall_can_view', $can_view, $profile_id );
	}

	/**
	 * Format post to time difference
	 *
	 * @deprecated 2.3.3
	 *
	 * @param $from
	 * @param $to
	 * @return string
	 */
	public function human_time_diff( $from, $to = '' ) {
		_deprecated_function( __METHOD__, '2.3.3', 'UM()->datetime()_>time_diff()' );
		return UM()->datetime()->time_diff( $from, $to );
	}

	/**
	 * Get faces of people who liked
	 *
	 * @param $post_id
	 * @param int $num
	 *
	 * @return string
	 */
	public function get_faces( $post_id, $num = 10 ) {
		$res   = '';
		$limit = UM()->options()->get( 'activity_max_faces' );
		if ( ! $limit ) {
			$limit = 10;
		}

		$i     = 0;
		$users = get_post_meta( $post_id, '_liked', true );
		if ( $users && is_array( $users ) ) {
			$users = array_reverse( $users );
			$users = array_slice( $users, 0, $num );
			foreach ( $users as $user_id ) {
				if ( absint( $user_id ) && $user_id ) {

					$res .= get_avatar( $user_id, 80 );

					$i++;
					if ( $i >= $limit ) {
						break;
					}
				}
			}
		}

		$content = '<a href="#" data-post_id="' . esc_attr( $post_id ) . '" class="um-activity-show-likes um-tip-s" title="' . esc_attr__( 'People who like this', 'um-activity' ) . '">' . $res . '</a>';

		/**
		 * Filter change content for faces who like post.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_get_faces
		 *
		 * @param {string}  $content   User faces.
		 * @param {int}     $post_id   post ID.
		 * @param {int}     $num       number of faces.
		 *
		 * @example <caption>Change description.</caption>
		 * function my_um_activity_get_faces( $content, $post_id, $num ) {
		 *    // your code here
		 *    $content = '<a href="#" data-post_id="' . esc_attr( $post_id ) . '" class="um-activity-show-likes um-tip-s" title="' . esc_attr__( 'Likes', 'um-activity' ) . '">' . $res . '</a>';
		 *    return $content;
		 * }
		 * add_filter( 'um_activity_get_faces', 'my_um_activity_get_faces', 10, 3 );
		 */
		$content = apply_filters( 'um_activity_get_faces', $content, $post_id, $num );

		return $content;
	}


	/**
	 * Hide a comment for user
	 *
	 * @param $comment_id
	 */
	public function user_hide_comment( $comment_id ) {
		$user_id = get_current_user_id();

		//hide comment replies
		$comment_data = get_comment( $comment_id );
		if ( 0 === absint( $comment_data->comment_parent ) ) {
			$replies = get_comments(
				array(
					'post_id' => $comment_data->comment_post_ID,
					'parent'  => $comment_id,
					'number'  => 10000,
					'offset'  => 0,
					'fields'  => 'ids',
				)
			);

			if ( ! empty( $replies ) && ! is_wp_error( $replies ) ) {
				foreach ( $replies as $reply_id ) {
					$this->user_hide_comment( $reply_id );
				}
			}
		}

		$users = get_comment_meta( $comment_id, '_hidden_from', true );

		if ( empty( $users ) || ! is_array( $users ) ) {
			$users = array();
		}

		$users[ $user_id ] = current_time( 'timestamp' );

		update_comment_meta( $comment_id, '_hidden_from', $users );
	}


	/**
	 * Unhide a comment for user
	 *
	 * @param $comment_id
	 */
	public function user_unhide_comment( $comment_id ) {
		$users = get_comment_meta( $comment_id, '_hidden_from', true );

		$user_id = get_current_user_id();

		if ( isset( $users[ $user_id ] ) ) {
			unset( $users[ $user_id ] );
		}

		if ( ! $users ) {
			delete_comment_meta( $comment_id, '_hidden_from' );
		} else {
			update_comment_meta( $comment_id, '_hidden_from', $users );
		}
	}


	/**
	 * Checks if user hidden comment
	 *
	 * @param $comment_id
	 *
	 * @return int
	 */
	public function user_hidden_comment( $comment_id ) {
		$users   = get_comment_meta( $comment_id, '_hidden_from', true );
		$user_id = get_current_user_id();

		if ( $users && is_array( $users ) && isset( $users[ $user_id ] ) ) {
			return 1;
		}

		return 0;
	}


	/***
	 ***    @Checks if user liked specific wall comment
	 ***/
	public function user_liked_comment( $comment_id ) {
		$res   = '';
		$users = get_comment_meta( $comment_id, '_liked', true );
		if ( $users && is_array( $users ) && in_array( get_current_user_id(), $users ) ) {
			return true;
		}

		return false;
	}

	/***
	 ***    @Checks if user liked specific wall post
	 ***/
	public function user_liked( $post_id ) {
		$res   = '';
		$users = get_post_meta( $post_id, '_liked', true );
		if ( $users && is_array( $users ) && in_array( get_current_user_id(), $users ) ) {
			return true;
		}

		return false;
	}

	/***
	 ***    @Checks if post is reported
	 ***/
	public function reported( $post_id, $reporter_id = null ) {
		$reported = get_post_meta( $post_id, '_reported', true );
		if ( $reporter_id ) {
			$reported_by = get_post_meta( $post_id, '_reported_by', true );
			if ( isset( $reported_by[ $reporter_id ] ) ) {
				return 1;
			}

			return 0;
		}

		return ( $reported ) ? 1 : 0;
	}

	/***
	 ***    @Gets action name
	 ***/
	public function get_action( $post_id ) {
		$action = (string) get_post_meta( $post_id, '_action', true );
		$action = ( $action ) ? $action : 'status';

		return isset( $this->global_actions[ $action ] ) ? $this->global_actions[ $action ] : '';
	}

	/***
	 ***    @Gets action type
	 ***/
	public function get_action_type( $post_id ) {
		$action = (string) get_post_meta( $post_id, '_action', true );
		$action = ( $action ) ? $action : 'status';

		return $action;
	}

	/**
	 * Get comment time.
	 *
	 * @param string $time
	 *
	 * @return string
	 */
	public function get_comment_time( $time ) {
		return UM()->datetime()->time_diff( strtotime( $time ) );
	}

	/**
	 * Get comment link
	 *
	 * @param string $post_link
	 * @param int $comment_id
	 *
	 * @return string
	 */
	public function get_comment_link( $post_link, $comment_id ) {
		$link = add_query_arg( 'wall_comment_id', $comment_id, $post_link );
		return $link;
	}

	/**
	 * Gets activity in nice time format.
	 *
	 * @param int $post_id
	 *
	 * @return string
	 */
	public function get_post_time( $post_id ) {
		$unix_published_date = get_post_datetime( $post_id, 'date', 'gmt' );

		$time = UM()->datetime()->time_diff( $unix_published_date->getTimestamp() );

		/**
		 * Filter change post time.
		 *
		 * @hook um_activity_human_post_time
		 *
		 * @param {string}  $time     post time.
		 * @param {int}     $post_id  post id.
		 *
		 * @example <caption>Change post time.</caption>
		 * function my_um_activity_human_post_time( $time, $post_id ) {
		 *     // your code here
		 *     return $time;
		 * }
		 * add_filter( 'um_activity_human_post_time', 'my_um_activity_human_post_time', 10, 2 );
		 */
		return apply_filters( 'um_activity_human_post_time', $time, $post_id );
	}

	/**
	 * Gets post permalink
	 *
	 * @param int $post_id
	 *
	 * @return string
	 */
	public function get_permalink( $post_id ) {
		$url = um_get_core_page( 'activity' );
		return add_query_arg( 'wall_post', $post_id, $url );
	}


	/**
	 * Gets post author
	 *
	 * @param int $post_id
	 *
	 * @return int
	 */
	public function get_author( $post_id ) {
		$author = get_post_meta( $post_id, '_user_id', true );
		if ( empty( $author ) ) {
			$post = get_post( $post_id );
			if ( empty( $post ) ) {
				return 0;
			}
			$author = $post->post_author;
		}
		return ! empty( $author ) ? absint( $author ) : 0;
	}


	/**
	 * Gets post wall ID
	 *
	 * @param int $post_id
	 *
	 * @return int
	 */
	public function get_wall( $post_id ) {
		$wall = absint( get_post_meta( $post_id, '_wall_id', true ) );
		return ( $wall ) ? $wall : 0;
	}


	/**
	 * Get likes count
	 *
	 * @param $post_id
	 *
	 * @return int
	 */
	public function get_likes_number( $post_id ) {
		return absint( get_post_meta( $post_id, '_likes', true ) );
	}


	/**
	 * Get comment count
	 *
	 * @param int $post_id
	 *
	 * @return int
	 */
	public function get_comments_number( $post_id ) {
		$comments_all = get_comments(
			array(
				'post_id' => $post_id,
				'parent'  => 0,
				'number'  => 10000,
				'offset'  => 0,
			)
		);
		return count( $comments_all );
	}


	/**
	 * Get replies count
	 *
	 * @param int $post_id
	 * @param int $comment_id
	 *
	 * @return int
	 */
	public function get_replies_number( $post_id, $comment_id ) {
		$replies_all = get_comments(
			array(
				'post_id' => $post_id,
				'parent'  => $comment_id,
				'number'  => 10000,
				'offset'  => 0,
			)
		);
		return count( $replies_all );
	}


	/**
	 * Make links clickable
	 *
	 * @param $content
	 *
	 * @return mixed|null|string|string[]
	 */
	public function make_links_clickable( $content ) {
		$has_iframe = preg_match( '/<iframe.*src=\"(.*)\".*><\/iframe>/isU', $content, $matches );

		if ( $has_iframe ) {
			$content = preg_replace( '/<iframe.*?\/iframe>/i', '[um_activity_iframe]', $content );
		}

		$attributes = apply_filters(
			'um_activity_make_links_clickable_attrs',
			array(
				'target' => '_blank',
				'class'  => 'um-link',
			)
		);

		$attribute_string = '';

		foreach ( $attributes as $key => $value ) {
			$attribute_string .= esc_html( $key ) . '="' . esc_attr( $value ) . '" ';
		}

		$content = preg_replace( '/(<a\b[^><]*)>/i', '$1 ' . trim( $attribute_string ) . '>', make_clickable( $content ) );

		if ( $has_iframe && isset( $matches[0] ) ) {
			$content = str_replace( '[um_activity_iframe]', $matches[0], $content );
		}

		return $content;
	}


	/**
	 * Removes Visual Composer's shortcodes
	 *
	 * @param  string $excerpt
	 *
	 * @return string
	 */
	public function remove_vc_from_excerpt( $excerpt ) {
		$patterns     = '/\[[\/]?vc_[^\]]*\]|[[\/]?nectar_[^\]]*\]|[[\/]?cspm_[^\]]*\]/';
		$replacements = '';

		return preg_replace( $patterns, $replacements, $excerpt );
	}

	/**
	 * Check if enabled friends activity only.
	 * @return bool
	 */
	public function friends_activity() {
		if ( class_exists( 'UM_Friends_API' ) && UM()->options()->get( 'activity_friends_users' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Grab friends user ids
	 *
	 * @return array|null
	 */
	public function friends_ids() {
		$array = array();

		if ( ! $this->friends_activity() ) {
			return null;
		}

		if ( ! is_user_logged_in() ) {
			return array( 0 );
		}

		$array[] = get_current_user_id();

		$friends = UM()->Friends_API()->api()->friends( get_current_user_id() );
		if ( $friends ) {
			foreach ( $friends as $arr ) {
				if ( absint( $arr['user_id1'] ) === get_current_user_id() ) {
					$array[] = $arr['user_id2'];
				} else {
					$array[] = $arr['user_id1'];
				}
			}
		}

		if ( isset( $array ) ) {
			return $array;
		}

		return null;
	}

	/**
	 * Retrieves an array of posts IDs based on the wall arguments.
	 *
	 * @param array $args {
	 *   Array of the wall arguments.
	 *
	 *   @type string $hashtag    A hashtag.
	 *   @type int    $offset     The number of posts to offset before retrieval. Default 0.
	 *   @type int    $user_id    Profile ID. Default 0.
	 *   @type bool   $user_wall  Show a wall for the specific profile if true. Default false.
	 * }
	 *
	 * @return int[] An array of post IDs.
	 */
	public function get_wall_posts( $args ) {
		global $wpdb;

		$user_wall_id = empty( $args['user_id'] ) || empty( $args['user_wall'] ) ? null : absint( $args['user_id'] );

		$query = array(
			'fields'      => 'ids',
			'post_type'   => 'um_activity',
			'post_status' => 'publish',
			'meta_query'  => array(),
		);

		// If $user_wall == 0 - Loads Global Site Activity.
		// If $user_wall == 1 - Loads User Wall and $user_id.
		if ( ! empty( $args['user_id'] ) && ! empty( $args['user_wall'] ) ) {
			$query['meta_query'][] = array(
				'relation' => 'OR',
				array(
					'key'     => '_wall_id',
					'value'   => absint( $args['user_id'] ),
					'compare' => '=',
				),
				array(
					'key'     => '_user_id',
					'value'   => absint( $args['user_id'] ),
					'compare' => '=',
				),
			);
		} elseif ( is_user_logged_in() ) {
			$in_users     = array();
			$followed_ids = $this->followed_ids();
			if ( $followed_ids ) {
				$in_users = array_merge( $in_users, $followed_ids );
			}
			$friends_ids = $this->friends_ids();
			if ( $friends_ids ) {
				$in_users = array_merge( $in_users, $friends_ids );
			}

			if ( UM()->options()->get( 'activity_enable_privacy' ) ) {
				$private_walls = $wpdb->get_col(
					$wpdb->prepare(
						"SELECT DISTINCT user_id
						FROM {$wpdb->usermeta}
						WHERE meta_key='wall_privacy' AND
							  meta_value='2' AND
							  user_id != %d",
						get_current_user_id()
					)
				);

				$private_walls = apply_filters( 'um_activity_wall_private_walls', $private_walls );

				if ( ! empty( $private_walls ) ) {
					if ( ! empty( $in_users ) ) {
						$in_users[] = get_current_user_id();
						$in_users   = array_unique( $in_users );
						$in_users   = array_diff( $in_users, $private_walls );

						$query['meta_query'][] = array(
							'relation' => 'AND',
							array(
								'key'     => '_wall_id',
								'value'   => array_unique( $private_walls ),
								'compare' => 'NOT IN',
							),
							array(
								'key'     => '_user_id',
								'value'   => $in_users,
								'compare' => 'IN',
							),
						);
					} else {
						$query['meta_query'][] = array(
							'relation' => 'AND',
							array(
								'key'     => '_wall_id',
								'value'   => array_unique( $private_walls ),
								'compare' => 'NOT IN',
							),
							array(
								'key'     => '_user_id',
								'value'   => array_unique( $private_walls ),
								'compare' => 'NOT IN',
							),
						);
					}
				} else {
					if ( ! empty( $in_users ) ) {
						$in_users[]            = get_current_user_id();
						$query['meta_query'][] = array(
							'key'     => '_user_id',
							'value'   => array_unique( $in_users ),
							'compare' => 'IN',
						);
					}
				}
			} else {
				if ( ! empty( $in_users ) ) {
					$in_users[]            = get_current_user_id();
					$query['meta_query'][] = array(
						'key'     => '_user_id',
						'value'   => array_unique( $in_users ),
						'compare' => 'IN',
					);
				}
			}
		} elseif ( ! is_user_logged_in() ) {
			if ( UM()->options()->get( 'activity_enable_privacy' ) ) {
				$private_walls = $wpdb->get_col(
					"SELECT DISTINCT user_id
					FROM {$wpdb->usermeta}
					WHERE meta_key='wall_privacy' AND
						  (meta_value='1' OR meta_value='2' OR meta_value='3' OR meta_value='4')"
				);
				$private_walls = apply_filters( 'um_activity_wall_private_walls', $private_walls );

				if ( ! empty( $private_walls ) ) {
					$query['meta_query'][] = array(
						'relation' => 'AND',
						array(
							'key'     => '_wall_id',
							'value'   => array_unique( $private_walls ),
							'compare' => 'NOT IN',
						),
						array(
							'key'     => '_user_id',
							'value'   => array_unique( $private_walls ),
							'compare' => 'NOT IN',
						),
					);
				}
			}
		}

		if ( ! empty( $args['hashtag_id'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			$query['tax_query'] = array(
				array(
					'taxonomy' => 'um_hashtag',
					'field'    => 'term_id',
					'terms'    => explode( ',', $args['hashtag_id'] ),
				),
			);
		}

		$query['offset']         = ! empty( $args['offset'] ) ? absint( $args['offset'] ) : 0;
		$query['posts_per_page'] = absint( $this->get_posts_per_page() );

		// Get posts.
		$query     = apply_filters( 'um_activity_wall_args', $query, $user_wall_id );
		$query_obj = new \WP_Query( $query );

		return array(
			'result'      => $query_obj->get_posts(),
			'total_posts' => $query_obj->found_posts,
		);
	}

	/**
	 * @param int $post_id
	 *
	 * @return bool
	 */
	public function can_view_post( $post_id ) {
		$can_view = true;

		$activity_post = get_post( $post_id );
		if ( empty( $activity_post ) || 'publish' !== $activity_post->post_status ) {
			$can_view = false;
		}

		$wall_id       = $this->get_wall( $post_id );
		$can_view_wall = $this->can_view_wall( $wall_id );
		if ( true !== $can_view_wall ) {
			$can_view = false;
		}

		$in_users     = array();
		$followed_ids = $this->followed_ids();
		if ( $followed_ids ) {
			$in_users = array_merge( $in_users, $followed_ids );
		}
		$friends_ids = $this->friends_ids();
		if ( $friends_ids ) {
			$in_users = array_merge( $in_users, $friends_ids );
		}

		if ( ! empty( $in_users ) ) {
			$in_users[] = get_current_user_id();
			$in_users   = array_unique( $in_users );
			$in_users   = array_map( 'absint', $in_users );
			$author_id  = $this->get_author( $post_id );
			if ( ! in_array( $author_id, $in_users, true ) ) {
				$can_view = false;
			}
		}

		return apply_filters( 'um_activity_post_can_view', $can_view, $post_id );
	}

	/**
	 * Load wall posts
	 */
	public function ajax_load_wall() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() && UM()->options()->get( 'activity_require_login' ) ) {
			wp_send_json_error( __( 'You must login to view activity wall', 'um-activity' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		$hashtag_id = '';
		if ( ! empty( $_POST['hashtag'] ) ) {
			$hashtag    = str_replace( '#', '', sanitize_text_field( $_POST['hashtag'] ) );
			$term       = get_term_by( 'name', $hashtag, 'um_hashtag' );
			$hashtag_id = isset( $term->term_id ) ? $term->term_id : '';
		}

		$args = array(
			'hashtag_id' => $hashtag_id,
			'user_wall'  => false,
			'offset'     => empty( $_POST['offset'] ) ? 0 : absint( $_POST['offset'] ),
		);

		if ( ! empty( $_POST['user_wall'] ) ) {
			$args['user_wall'] = true;
			$args['user_id']   = empty( $_POST['user_id'] ) ? 0 : absint( $_POST['user_id'] );

			$can_view = $this->can_view_wall( $args['user_id'] );
			if ( true !== $can_view ) {
				wp_send_json_error( $can_view );
			}
		}
		// phpcs:enable WordPress.Security.NonceVerification

		$query = $this->get_wall_posts( $args );

		$new_offset = $args['offset'] + $this->get_posts_per_page();

		UM()->Activity_API()->shortcode()->args = array(
			'allowed_html' => $this->get_allowed_html(),
			'offset'       => $new_offset,
			'wall_posts'   => $query['result'],
			'end'          => $new_offset >= $query['total_posts'],
		);
		$t_args                                 = UM()->Activity_API()->shortcode()->args;

		// @todo: Remove 'safe_style_css' as soon as Requires at least >= 6.2.
		add_filter( 'safe_style_css', array( &$this, 'add_extra_safe_style_css' ) );
		UM()->get_template( 'user-wall.php', um_activity_plugin, $t_args, true );
		remove_filter( 'safe_style_css', array( &$this, 'add_extra_safe_style_css' ) );

		die();
	}

	/**
	 * Returns an array of allowed HTML tags and attributes for the Activity wall.
	 *
	 * @return array An array of allowed HTML elements and attributes.
	 */
	public function get_allowed_html() {
		// Adds iframe and onclick to allowed tags and attributes only for activity wall.
		add_filter( 'um_late_escaping_allowed_tags', array( &$this, 'add_extra_kses_allowed_tags' ), 10, 2 );
		$allowed_html = UM()->get_allowed_html( 'templates' );
		remove_filter( 'um_late_escaping_allowed_tags', array( &$this, 'add_extra_kses_allowed_tags' ) );

		return $allowed_html;
	}

	public function add_extra_kses_allowed_tags( $allowed_html, $context ) {
		if ( 'templates' === $context ) {
			$allowed_html['iframe'] = array(
				'allow'          => true,
				'frameborder'    => true,
				'loading'        => true,
				'name'           => true,
				'referrerpolicy' => true,
				'sandbox'        => true,
				'src'            => true,
				'srcdoc'         => true,
				'title'          => true,
			);

			$allowed_html['strong']['onclick'] = true;
		}
		return $allowed_html;
	}

	/**
	 * Filter the list of allowed CSS attributes. Needed to restore position for WP < 6.2.0.
	 *
	 * @param array $allowed_attr
	 * @return array
	 */
	public function add_extra_safe_style_css( $allowed_attr ) {
		if ( version_compare( get_bloginfo( 'version' ), '6.2', '>=' ) ) {
			return $allowed_attr;
		}

		return array_merge(
			$allowed_attr,
			array(
				'position',
				'top',
				'right',
				'bottom',
				'left',
			)
		);
	}

	/**
	 * Get user suggestions
	 */
	public function ajax_get_user_suggestions() {
		UM()->check_ajax_nonce();
		// phpcs:ignore WordPress.Security.NonceVerification
		$data = apply_filters( 'um_activity_ajax_get_user_suggestions', array(), sanitize_text_field( $_GET['term'] ) );

		$current_user = get_current_user_id();

		$data = array_filter(
			$data,
			function( $v ) use ( $current_user ) {
				static $user_ids;
				if ( empty( $user_ids ) ) {
					$user_ids = array();
				}

				if ( empty( $v['user_id'] ) ) {
					return false;
				}

				$user_id = absint( $v['user_id'] );

				// Avoid mention yourself.
				if ( $current_user === $user_id ) {
					return false;
				}

				if ( ! in_array( $user_id, $user_ids, true ) ) {
					$user_ids[] = $user_id;
					return true;
				}
				return false;
			}
		);
		$data = array_values( $data );

		wp_send_json( $data );
	}

	/**
	 * Removes a wall post
	 */
	public function ajax_remove_post() {
		UM()->check_ajax_nonce();
		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['post_id'] ) || absint( $_POST['post_id'] ) <= 0 ) {
			die();
		}
		$post_id = absint( $_POST['post_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$author_id = $this->get_author( $post_id );
		if ( current_user_can( 'edit_users' ) || get_current_user_id() === $author_id ) {
			wp_delete_post( $post_id, true );
		}

		die();
	}

	/**
	 * Removes a wall comment via AJAX
	 */
	public function ajax_remove_comment() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['comment_id'] ) || absint( $_POST['comment_id'] ) <= 0 ) {
			die();
		}
		$comment_id = absint( $_POST['comment_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		if ( $this->can_edit_comment( $comment_id, get_current_user_id() ) ) {
			$this->delete_comment( $comment_id );
			die();
		}

		// Post authors can delete spam and malicious comments under their posts.
		$comment   = get_comment( $comment_id );
		$author_id = $this->get_author( $comment->comment_post_ID );
		if ( get_current_user_id() === $author_id ) {
			$this->delete_comment( $comment_id );
		}

		die();
	}

	/**
	 * Remove comment with all replies
	 *
	 * @param $comment_id
	 */
	public function delete_comment( $comment_id ) {
		global $wpdb;

		$comment = get_comment( $comment_id );

		// Remove comment replies
		if ( 0 === absint( $comment->comment_parent ) ) {
			$replies = get_comments(
				array(
					'post_id' => $comment->comment_post_ID,
					'parent'  => $comment_id,
					'number'  => 10000,
					'offset'  => 0,
					'fields'  => 'ids',
				)
			);

			if ( ! empty( $replies ) && ! is_wp_error( $replies ) ) {
				foreach ( $replies as $reply_id ) {
					$this->delete_comment( $reply_id );
				}
			}
		}

		// remove comment
		wp_delete_comment( $comment_id, true );

		// Remove hashtag(s) from the trending list if it's totally remove from posts / comments.
		$content = $comment->comment_content;
		$post_id = $comment->comment_post_ID;
		preg_match_all( '/(?<!\&)#([^\s\<]+)/', $content, $matches );
		if ( isset( $matches[1] ) && is_array( $matches[1] ) ) {
			foreach ( $matches[1] as $hashtag ) {
				$post_count    = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT COUNT(*)
						FROM {$wpdb->posts}
						WHERE ID = %d AND
							  post_content LIKE %s",
						$post_id,
						"%>#{$hashtag}<%"
					)
				);
				$comment_count = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT COUNT(*)
						FROM {$wpdb->comments}
						WHERE comment_post_ID = %d AND
							  comment_content LIKE %s",
						$post_id,
						"%>#{$hashtag}<%"
					)
				);

				if ( empty( $post_count ) && empty( $comment_count ) ) {
					$term = get_term_by( 'name', $hashtag, 'um_hashtag' );
					wp_remove_object_terms( $post_id, $term->term_id, 'um_hashtag' );
				}
			}
		}
	}

	/**
	 * Load post likes via AJAX
	 */
	public function ajax_get_post_likes() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['post_id'] ) || 0 >= absint( $_POST['post_id'] ) ) {
			die();
		}

		$item_id = absint( $_POST['post_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		if ( ! $item_id ) {
			die();
		}

		$users = get_post_meta( $item_id, '_liked', true );
		if ( ! $users || ! is_array( $users ) ) {
			die();
		}

		$users = array_reverse( $users );

		ob_start();

		$file       = um_activity_path . 'templates/likes.php';
		$theme_file = get_stylesheet_directory() . '/ultimate-member/templates/activity/likes.php';

		if ( file_exists( $theme_file ) ) {
			$file = $theme_file;
		}

		if ( file_exists( $file ) ) {
			include $file;
		}

		$output = ob_get_contents();
		ob_end_clean();
		die( $output );
	}

	/***
	 ***    @load comment likes
	 ***/
	public function ajax_get_comment_likes() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['comment_id'] ) || absint( $_POST['comment_id'] ) <= 0 ) {
			die();
		}

		$item_id = absint( $_POST['comment_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		if ( ! $item_id ) {
			die();
		}

		$users = get_comment_meta( $item_id, '_liked', true );
		if ( ! $users || ! is_array( $users ) ) {
			die();
		}

		$users = array_reverse( $users );

		ob_start();

		$file       = um_activity_path . 'templates/likes.php';
		$theme_file = get_stylesheet_directory() . '/ultimate-member/templates/activity/likes.php';

		if ( file_exists( $theme_file ) ) {
			$file = $theme_file;
		}

		if ( file_exists( $file ) ) {
			include $file;
		}

		$output = ob_get_contents();
		ob_end_clean();
		die( $output );
	}


	/**
	 * Hide a comment via AJAX
	 */
	public function ajax_hide_comment() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			die();
		}

		$comment_id = absint( $_POST['comment_id'] ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( $comment_id <= 0 ) {
			die();
		}
		$this->user_hide_comment( $comment_id );
		die();
	}


	/**
	 * Unhide a comment via AJAX
	 */
	public function ajax_unhide_comment() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			die();
		}

		$comment_id = absint( $_POST['comment_id'] ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( $comment_id <= 0 ) {
			die();
		}

		$this->user_unhide_comment( $comment_id );
		die();
	}

	/***
	 ***    @report a post
	 ***/
	public function ajax_report_post() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			die();
		}

		$post_id = absint( $_POST['post_id'] ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( $post_id <= 0 ) {
			die();
		}

		$user_id = get_current_user_id();

		$users_reported = get_post_meta( $post_id, '_reported_by', true );
		if ( empty( $users_reported ) ) {
			$users_reported = array();
		}
		if ( ! isset( $users_reported[ $user_id ] ) ) {
			$users_reported[ $user_id ] = current_time( 'timestamp' );
			update_post_meta( $post_id, '_reported_by', $users_reported );
		}

		if ( ! get_post_meta( $post_id, '_reported', true ) ) {
			$count = (int) get_option( 'um_activity_flagged' );
			update_option( 'um_activity_flagged', $count + 1 );
		}

		$new_r = (int) get_post_meta( $post_id, '_reported', true );
		update_post_meta( $post_id, '_reported', $new_r + 1 );

		die();

	}


	/***
	 ***    @un-report a post
	 ***/
	public function ajax_unreport_post() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			die();
		}

		$post_id = absint( $_POST['post_id'] ); // phpcs:ignore WordPress.Security.NonceVerification
		$user_id = get_current_user_id();

		if ( 0 >= $post_id ) {
			die();
		}

		$users_reported = get_post_meta( $post_id, '_reported_by', true );
		if ( is_array( $users_reported ) && isset( $users_reported[ $user_id ] ) ) {
			unset( $users_reported[ $user_id ] );
		}

		if ( ! $users_reported ) {
			$user_reported = '';
		}

		update_post_meta( $post_id, '_reported_by', $users_reported );

		if ( get_post_meta( $post_id, '_reported', true ) ) {

			$new_r = (int) get_post_meta( $post_id, '_reported', true );
			-- $new_r;
			if ( $new_r < 0 ) {
				$new_r = 0;
			}
			update_post_meta( $post_id, '_reported', $new_r );

			if ( 0 === $new_r ) {
				$count = absint( get_option( 'um_activity_flagged' ) );
				update_option( 'um_activity_flagged', absint( $count - 1 ) );
			}
		}

		die();

	}


	/**
	 * Load wall comments via AJAX
	 */
	public function ajax_load_more_comments() {
		UM()->check_ajax_nonce();

		$number    = UM()->options()->get( 'activity_load_comments_count' );
		$offset    = absint( $_POST['offset'] ); // phpcs:ignore WordPress.Security.NonceVerification
		$post_id   = absint( $_POST['post_id'] ); // phpcs:ignore WordPress.Security.NonceVerification
		$post_link = $this->get_permalink( $post_id );

		$comments     = get_comments(
			array(
				'post_id' => $post_id,
				'parent'  => 0,
				'number'  => $number,
				'offset'  => $offset,
				'order'   => UM()->options()->get( 'activity_order_comment' ),
			)
		);
		$comments_all = $this->get_comments_number( $post_id );

		UM()->Activity_API()->shortcode()->args = $t_args = array(
			'comments'  => $comments,
			'post_id'   => $post_id,
			'post_link' => $post_link,
		);
		UM()->get_template( 'comment.php', um_activity_plugin, $t_args, true );

		if ( $comments_all > ( $offset + $number ) ) {
			?>
			<span class="um-activity-commentload-end"></span>
			<?php
		}

		die();
	}


	/**
	 * Load wall replies via AJAX
	 */
	public function ajax_load_more_replies() {
		UM()->check_ajax_nonce();

		$number = UM()->options()->get( 'activity_load_comments_count' );

		// phpcs:disable WordPress.Security.NonceVerification
		$offset     = absint( $_POST['offset'] );
		$post_id    = absint( $_POST['post_id'] );
		$comment_id = absint( $_POST['comment_id'] );
		$post_link  = $this->get_permalink( $post_id );
		// phpcs:enable WordPress.Security.NonceVerification

		$child     = get_comments(
			array(
				'post_id' => $post_id,
				'parent'  => $comment_id,
				'number'  => $number,
				'offset'  => $offset,
				'order'   => UM()->options()->get( 'activity_order_comment' ),
			)
		);
		$child_all = $this->get_replies_number( $post_id, $comment_id );

		ob_start();

		foreach ( $child as $commentc ) {
			um_fetch_user( $commentc->user_id );

			UM()->Activity_API()->shortcode()->args = $t_args = array(
				'commentc'  => $commentc,
				'post_id'   => $post_id,
				'post_link' => $post_link,
			);
			UM()->get_template( 'comment-reply.php', um_activity_plugin, $t_args, true );
		}

		if ( $child_all > ( $offset + $number ) ) {
			?>
			<span class="um-activity-ccommentload-end"></span>
			<?php
		}

		ob_get_flush();
		die();
	}

	/**
	 * Like wall comment.
	 *
	 */
	public function ajax_like_comment() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( __( 'You must login to like', 'um-activity' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['commentid'] ) || ! is_numeric( $_POST['commentid'] ) ) {
			wp_send_json_error( __( 'Invalid comment', 'um-activity' ) );
		}

		$comment_id = absint( $_POST['commentid'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$liked = get_comment_meta( $comment_id, '_liked', true );
		if ( is_array( $liked ) && in_array( get_current_user_id(), $liked, true ) ) {
			wp_send_json_error( __( 'You already liked this comment', 'um-activity' ) );
		}

		$increase_likes = false;
		$likes          = get_comment_meta( $comment_id, '_likes', true );
		$likes          = absint( $likes );

		if ( empty( $liked ) || ! is_array( $liked ) ) {
			$liked          = array( get_current_user_id() );
			$increase_likes = true;
		} else {
			if ( ! in_array( get_current_user_id(), $liked, true ) ) {
				$liked[]        = get_current_user_id();
				$increase_likes = true;
			}
		}

		if ( $increase_likes ) {
			update_comment_meta( $comment_id, '_liked', $liked );
			$likes ++;
			update_comment_meta( $comment_id, '_likes', $likes );
		}

		wp_send_json_success( array( 'likes' => $likes ) );
	}

	/**
	 * Unlike wall comment.
	 *
	 */
	public function ajax_unlike_comment() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( __( 'You must login to unlike', 'um-activity' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['commentid'] ) || ! is_numeric( $_POST['commentid'] ) ) {
			wp_send_json_error( __( 'Invalid comment', 'um-activity' ) );
		}

		$comment_id = absint( $_POST['commentid'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$liked = get_comment_meta( $comment_id, '_liked', true );
		if ( empty( $liked ) || ! is_array( $liked ) ) {
			wp_send_json_error( __( 'Invalid comment data', 'um-activity' ) );
		}

		if ( ! in_array( get_current_user_id(), $liked, true ) ) {
			wp_send_json_error( __( 'You didn\'t like this comment', 'um-activity' ) );
		}

		$likes = get_comment_meta( $comment_id, '_likes', true );
		$likes = absint( $likes );

		$liked = array_diff( $liked, array( get_current_user_id() ) );
		update_comment_meta( $comment_id, '_liked', $liked );

		$likes --;
		$likes = 0 < $likes ? $likes : 0;
		update_comment_meta( $comment_id, '_likes', $likes );

		wp_send_json_success( array( 'likes' => $likes ) );
	}

	/**
	 * Like wall post.
	 *
	 */
	public function ajax_like_post() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( __( 'You must login to like', 'um-activity' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['postid'] ) || ! is_numeric( $_POST['postid'] ) ) {
			wp_send_json_error( __( 'Invalid wall post', 'um-activity' ) );
		}

		$post_id = absint( $_POST['postid'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$liked = get_post_meta( $post_id, '_liked', true );
		if ( is_array( $liked ) && in_array( get_current_user_id(), $liked, true ) ) {
			wp_send_json_error( __( 'You already liked this post', 'um-activity' ) );
		}

		$increase_likes = false;
		$likes          = get_post_meta( $post_id, '_likes', true );
		$likes          = absint( $likes );

		if ( empty( $liked ) || ! is_array( $liked ) ) {
			$liked          = array( get_current_user_id() );
			$increase_likes = true;
		} else {
			if ( ! in_array( get_current_user_id(), $liked, true ) ) {
				$liked[]        = get_current_user_id();
				$increase_likes = true;
			}
		}

		if ( $increase_likes ) {
			update_post_meta( $post_id, '_liked', $liked );
			$likes ++;
			update_post_meta( $post_id, '_likes', $likes );
		}

		do_action( 'um_activity_after_wall_post_liked', $post_id, get_current_user_id() );

		wp_send_json_success( array( 'likes' => $likes ) );
	}

	/**
	 * Unlike wall post
	 *
	 */
	public function ajax_unlike_post() {
		UM()->check_ajax_nonce();

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( __( 'You must login to unlike', 'um-activity' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['postid'] ) || ! is_numeric( $_POST['postid'] ) ) {
			wp_send_json_error( __( 'Invalid wall post', 'um-activity' ) );
		}

		$post_id = absint( $_POST['postid'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$liked = get_post_meta( $post_id, '_liked', true );
		if ( empty( $liked ) || ! is_array( $liked ) ) {
			wp_send_json_error( __( 'Invalid post data', 'um-activity' ) );
		}

		if ( ! in_array( get_current_user_id(), $liked, true ) ) {
			wp_send_json_error( __( 'You didn\'t like this post', 'um-activity' ) );
		}

		$likes = get_post_meta( $post_id, '_likes', true );
		$likes = absint( $likes );

		$liked = array_diff( $liked, array( get_current_user_id() ) );
		update_post_meta( $post_id, '_liked', $liked );

		$likes --;
		$likes = 0 < $likes ? $likes : 0;
		update_post_meta( $post_id, '_likes', $likes );

		wp_send_json_success( array( 'likes' => $likes ) );
	}

	/**
	 * Add a new wall post comment via AJAX
	 */
	public function ajax_wall_comment() {
		UM()->check_ajax_nonce();

		$output['error'] = '';

		if ( ! is_user_logged_in() ) {
			$output['error'] = __( 'Login to post a comment', 'um-activity' );
		}

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['postid'] ) || ! is_numeric( $_POST['postid'] ) ) {
			$output['error'] = __( 'Invalid wall post', 'um-activity' );
		}

		if ( ! isset( $_POST['comment'] ) || '' === trim( $_POST['comment'] ) ) {
			$output['error'] = __( 'Enter a comment first', 'um-activity' );
		}

		/**
		 * Filters change error message.
		 *
		 * @since 2.3.4
		 * @hook  um_activity_comment_error
		 *
		 * @param {string} $error   error message.
		 * @param {int}    $post_id post id.
		 *
		 * @return {string} error message.
		 *
		 * @example <caption>Add error message.</caption>
		 * function my_um_activity_comment_error( $error, $post_id ) {
		 *     $error = __( 'Error message', 'um-activity' );
		 *     return $error;
		 * }
		 * add_filter( 'um_activity_comment_error', 'my_um_activity_comment_error', 10, 4 );
		 */
		$output['error'] = apply_filters( 'um_activity_comment_error', $output['error'], $_POST['postid'] );

		if ( ! $output['error'] ) {

			um_fetch_user( get_current_user_id() );

			$time = current_time( 'mysql' );

			if ( isset( $_POST['postid'] ) ) {
				$post_id = absint( $_POST['postid'] );
			}
			$orig_content    = wp_kses(
				trim( $_POST['comment'] ),
				array(
					'br' => array(),
				)
			);
			$comment_content = apply_filters( 'um_activity_comment_content_new', $orig_content, $post_id );
			// apply hashtag
			$this->hashtagit( $post_id, $comment_content, true );

			$comment_content = $this->hashtag_links( $comment_content );
			$comment_content = apply_filters( 'um_activity_insert_post_content_filter', $comment_content, get_current_user_id(), absint( $post_id ), 'new' );

			um_fetch_user( get_current_user_id() );

			$data = array(
				'comment_post_ID'      => $post_id,
				'comment_author'       => um_user( 'display_name' ),
				'comment_author_email' => um_user( 'user_email' ),
				'comment_author_url'   => um_user_profile_url(),
				'comment_content'      => $comment_content,
				'user_id'              => get_current_user_id(),
				'comment_approved'     => 1,
				'comment_author_IP'    => um_user_ip(),
				'comment_type'         => 'um-social-activity',
			);

			$comment_content           = $this->make_links_clickable( $comment_content );
			$output['comment_content'] = stripslashes_deep( $comment_content );
			$output['comment_content'] = convert_smilies( $output['comment_content'] );

			if ( isset( $_POST['reply_to'] ) && absint( $_POST['reply_to'] ) ) {
				$data['comment_parent'] = absint( $_POST['reply_to'] );
			} else {
				$data['comment_parent'] = 0;
			}

			if ( ! empty( $_POST['commentid'] ) ) {
				$data['comment_ID'] = $commentid = absint( $_POST['commentid'] );
				wp_update_comment( $data );
			} else {
				$data['comment_date'] = $time;
				$commentid            = wp_insert_comment( $data );
			}

			update_comment_meta( $commentid, 'orig_content', $orig_content );

			if ( isset( $_POST['reply_to'] ) && absint( $_POST['reply_to'] ) ) {
				$comment_parent = $data['comment_parent'];
				do_action( 'um_activity_after_wall_comment_reply_published', $commentid, $comment_parent, $post_id, get_current_user_id() );
			} else {
				$comment_parent = 0;
			}

			$comment_count = get_post_meta( $post_id, '_comments', true );
			update_post_meta( $post_id, '_comments', $comment_count + 1 );

			$likes = get_comment_meta( $commentid, '_likes', true );

			$output['commentid']          = $commentid;
			$output['user_hidden']        = $this->user_hidden_comment( $commentid );
			$output['permalink']          = $this->get_comment_link( $this->get_permalink( $post_id ), $commentid );
			$output['time']               = $this->get_comment_time( $time );
			$output['can_edit_comment']   = $this->can_edit_comment( $commentid, get_current_user_id() );
			$output['user_liked_comment'] = $this->user_liked_comment( $commentid );
			$output['likes']              = empty( $likes ) ? 0 : $likes;

			do_action( 'um_activity_after_wall_comment_published', $commentid, $comment_parent, $post_id, get_current_user_id() );
		}

		$output = json_encode( $output );
		if ( is_array( $output ) ) {
			print_r( $output );
		} else {
			echo $output;
		}
		die;
		// phpcs:enable WordPress.Security.NonceVerification
	}

	/**
	 * Add a new wall post via AJAX
	 */
	public function ajax_activity_publish() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! isset( $_POST['_post_id'] ) ) {
			wp_send_json_error( __( 'Please specify the post ID. It\'s required.', 'um-activity' ) );
		}
		$post_id = absint( $_POST['_post_id'] );

		if ( '' === $_POST['_post_content'] || '' === trim( $_POST['_post_content'] ) ) {
			if ( '' === trim( $_POST['_post_img'] ) ) {
				wp_send_json_error( __( 'You should type something first.', 'um-activity' ) );
			}
		}

		$wall_id = 0;
		if ( ! empty( $_POST['_wall_id'] ) ) {
			$wall_id = absint( $_POST['_wall_id'] );
		}

		$_post_content = trim( $_POST['_post_content'] );
		$_post_img     = trim( $_POST['_post_img'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$has_oembed = false;

		if ( 0 === $post_id ) {
			$args = array(
				'post_title'   => '',
				'post_type'    => 'um_activity',
				'post_status'  => 'publish',
				'post_author'  => get_current_user_id(),
				'post_content' => '',
			);

			$output['link'] = '';

			if ( trim( $_post_content ) ) {
				$orig_content = wp_kses(
					trim( $_post_content ),
					array(
						'br' => array(),
					)
				);

				$safe_content = apply_filters( 'um_activity_new_post', $orig_content, 0 );

				// shared a link
				$shared_link = $this->get_content_link( $safe_content );
				$has_oembed  = $this->is_oembed( $shared_link );

				if ( isset( $shared_link ) && $shared_link && ! $_post_img && ! $has_oembed ) {
					$safe_content           = str_replace( $shared_link, '', $safe_content );
					$output['_shared_link'] = $shared_link;
				}

				$args['post_content'] = $safe_content;
			}

			$args = apply_filters( 'um_activity_insert_post_args', $args );

			$post_id = wp_insert_post( $args );

			// shared a link
			if ( isset( $shared_link ) && $shared_link && ! $_post_img && ! $has_oembed ) {
				$output['link'] = $this->set_url_meta( $shared_link, $post_id );
			} else {
				delete_post_meta( $post_id, '_shared_link' );
			}

			$args['post_content'] = apply_filters( 'um_activity_insert_post_content_filter', $args['post_content'], get_current_user_id(), $post_id, 'new' );

			wp_update_post(
				array(
					'ID'           => $post_id,
					'post_title'   => $post_id,
					'post_name'    => $post_id,
					'post_content' => $args['post_content'],
				)
			);

			if ( isset( $safe_content ) ) {
				$this->hashtagit( $post_id, $safe_content );
				$this->setup_video( $orig_content, $post_id );
				update_post_meta( $post_id, '_original_content', $orig_content );
				$output['orig_content'] = stripslashes_deep( $orig_content );
			}

			if ( $wall_id > 0 ) {
				update_post_meta( $post_id, '_wall_id', $wall_id );
			}

			// Save item meta
			update_post_meta( $post_id, '_oembed', $has_oembed );
			update_post_meta( $post_id, '_action', 'status' );
			update_post_meta( $post_id, '_user_id', get_current_user_id() );
			update_post_meta( $post_id, '_likes', 0 );
			update_post_meta( $post_id, '_comments', 0 );

			if ( $_post_img ) {
				$photo_uri = um_is_file_owner( $_post_img, get_current_user_id() ) ? $_post_img : false;

				update_post_meta( $post_id, '_photo', $photo_uri );
				$filename       = wp_basename( $photo_uri );
				$photo_metadata = get_transient( "um_{$filename}" );
				update_post_meta( $post_id, '_photo_metadata', $photo_metadata );
				delete_transient( "um_{$filename}" );

				UM()->uploader()->replace_upload_dir = true;
				UM()->uploader()->move_temporary_files( get_current_user_id(), array( '_photo' => $photo_uri ), true );
				UM()->uploader()->replace_upload_dir = false;

				$output['photo']           = $this->get_download_link( $post_id, get_current_user_id() );
				$output['photo_base']      = $photo_metadata['original_name'];
				$output['photo_orig_url']  = UM()->uploader()->get_upload_base_url() . get_current_user_id() . '/' . $filename;
				$output['photo_orig_base'] = wp_basename( $output['photo_orig_url'] );
			}

			$output['postid']  = $post_id;
			$output['content'] = $this->get_content( $post_id );
			$output['video']   = $this->get_video( $post_id );

			do_action( 'um_activity_after_wall_post_published', $post_id, get_current_user_id(), $wall_id );

		} else {
			$output['link'] = '';

			if ( trim( $_post_content ) ) {
				$orig_content = wp_kses(
					trim( $_post_content ),
					array(
						'br' => array(),
					)
				);

				$safe_content = apply_filters( 'um_activity_edit_post', $orig_content, 0 );

				// shared a link
				$shared_link = $this->get_content_link( $safe_content );
				$has_oembed  = $this->is_oembed( $shared_link );

				if ( isset( $shared_link ) && $shared_link && ! $_post_img && ! $has_oembed ) {
					$safe_content   = str_replace( $shared_link, '', $safe_content );
					$output['link'] = $this->set_url_meta( $shared_link, $post_id );
				} else {
					delete_post_meta( $post_id, '_shared_link' );
				}

				$safe_content = apply_filters( 'um_activity_update_post_content_filter', $safe_content, $this->get_author( $post_id ), $post_id, 'save' );

				$args['post_content'] = $safe_content;
			}

			$args['ID'] = $post_id;
			$args       = apply_filters( 'um_activity_update_post_args', $args );

//			// hash tag replies
//			$args['post_content'] = apply_filters( 'um_activity_insert_post_content_filter', $args['post_content'], get_current_user_id(), $post_id, 'new' );

			wp_update_post( $args );

			if ( isset( $safe_content ) ) {
				$this->hashtagit( $post_id, $safe_content );
				$this->setup_video( $orig_content, $post_id );
				update_post_meta( $post_id, '_original_content', $orig_content );
				$output['orig_content'] = stripslashes_deep( $orig_content );
			}

			if ( '' !== $_post_img ) {

				if ( um_is_temp_file( $_post_img ) ) {
					$photo_uri = um_is_file_owner( $_post_img, get_current_user_id() ) ? $_post_img : false;

					UM()->uploader()->replace_upload_dir = true;
					UM()->uploader()->move_temporary_files( get_current_user_id(), array( '_photo' => $photo_uri ), true );
					UM()->uploader()->replace_upload_dir = false;

					update_post_meta( $post_id, '_photo', $photo_uri );
					$filename       = wp_basename( $photo_uri );
					$photo_metadata = get_transient( "um_{$filename}" );
					update_post_meta( $post_id, '_photo_metadata', $photo_metadata );
					delete_transient( "um_{$filename}" );
				} else {
					$filename = wp_basename( $_post_img );
				}

				if ( ! isset( $photo_metadata ) ) {
					$photo_metadata = get_post_meta( $post_id, '_photo_metadata', true );
				}

				$output['photo']           = $this->get_download_link( $post_id, get_current_user_id() );
				$output['photo_base']      = $photo_metadata['original_name'];
				$output['photo_orig_url']  = UM()->uploader()->get_upload_base_url() . get_current_user_id() . '/' . $filename;
				$output['photo_orig_base'] = wp_basename( $output['photo_orig_url'] );

			} else {

				$photo_uri = get_post_meta( $post_id, '_photo', true );

				UM()->uploader()->replace_upload_dir = true;
				UM()->uploader()->delete_existing_file( $photo_uri );
				UM()->uploader()->replace_upload_dir = false;

				delete_post_meta( $post_id, '_photo' );
				delete_post_meta( $post_id, '_photo_metadata' );

				$filename = wp_basename( $photo_uri );
				delete_transient( "um_{$filename}" );

			}

			$output['postid']  = $post_id;
			$output['content'] = $this->get_content( $post_id );
			$output['video']   = $this->get_video( $post_id );

			do_action( 'um_activity_after_wall_post_updated', $post_id, get_current_user_id(), $wall_id );
		}

		// other output
		$output['permalink']      = $this->get_permalink( $post_id );
		$output['user_id']        = get_current_user_id();
		$output['has_oembed']     = $has_oembed;
		$output['has_text_video'] = get_post_meta( $post_id, '_video_url', true );

		/**
		 * Filter change AJAX post content.
		 *
		 * @since 2.3.6
		 *
		 * @hook um_activity_ajax_publish_output
		 *
		 * @param {string}  $output   output content.
		 *
		 * @example <caption>Change post content on AJAX.</caption>
		 * function my_um_activity_ajax_publish_output( $url, $content ) {
		 *     // your code here
		 *    $output['post_content'] = 'post content';
		 *    return $output;
		 * }
		 * add_filter( 'um_activity_ajax_publish_output', 'my_um_activity_ajax_publish_output' );
		 */
		$output = apply_filters( 'um_activity_ajax_publish_output', $output );

		wp_send_json_success( $output );
	}

	/**
	 * Add a new wall post
	 */
	public function ajax_get_activity_post() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( empty( $_POST['post_id'] ) ) {
			wp_send_json_error( __( 'You should select post first', 'um-activity' ) );
		}
		$post_id = absint( $_POST['post_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$post = get_post( $post_id );
		if ( empty( $post ) || is_wp_error( $post ) ) {
			wp_send_json_error( __( 'Invalid post ID', 'um-activity' ) );
		}

		$output['postid'] = $post_id;

		$photo_meta = get_post_meta( $post_id, '_photo_metadata', true );
		if ( ! empty( $photo_meta ) ) {
			$output['photo_base'] = $photo_meta['original_name'];
		}
		$output['orig_content'] = get_post_meta( $post_id, '_original_content', true );
		$output['photo']        = get_post_meta( $post_id, '_photo', true );
		$output['content']      = $this->get_content( $post_id );
		$output['video']        = $this->get_video( $post_id );

		// other output
		$output['permalink']      = $this->get_permalink( $post_id );
		$output['user_id']        = get_current_user_id();
		$output['has_oembed']     = get_post_meta( $post_id, '_oembed', true );
		$output['has_text_video'] = get_post_meta( $post_id, '_video_url', true );

		wp_send_json_success( $output );
	}

	/**
	 * Get comment info
	 */
	public function ajax_get_activity_comment() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( empty( $_POST['comment_id'] ) ) {
			wp_send_json_error( __( 'You should select comment first', 'um-activity' ) );
		}
		$comment_id = absint( $_POST['comment_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$comment = get_comment( $comment_id );
		if ( empty( $comment ) || is_wp_error( $comment ) ) {
			wp_send_json_error( __( 'You should select comment first', 'um-activity' ) );
		}

		$output['commentid']    = $comment_id;
		$orig_content           = get_comment_meta( $comment_id, 'orig_content', true );
		$output['orig_content'] = ! empty( $orig_content ) ? $orig_content : $comment->comment_content;
		$output['content']      = wp_kses( $comment->comment_content, 'post' );

		// other output
		$output['permalink'] = $this->get_permalink( $comment_id );
		$output['user_id']   = get_current_user_id();

		wp_send_json_success( $output );
	}

	/**
	 * @param $string
	 *
	 * @return mixed
	 */
	public function allow_get_params( $string ) {
		return str_replace( '&amp;', '&', $string );
	}


	/**
	 * Get Activity Posts count per page
	 * @return int
	 */
	public function get_posts_per_page() {
		$per_page = wp_is_mobile() ? UM()->options()->get( 'activity_posts_num_mob' ) : UM()->options()->get( 'activity_posts_num' );
		return absint( $per_page );
	}


	/**
	 * Get post image URL - thumbnail, first image, first cover
	 *
	 * @param int|\WP_Post $post Optional. Post ID or WP_Post object.
	 * @return string URL
	 */
	public function get_post_image_url( $post = null ) {
		$image_url = '';

		if ( has_post_thumbnail( $post ) ) {
			$image_urls = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), 'large' );
			$image_url  = current( $image_urls );
		} else {
			if ( is_numeric( $post ) ) {
				$post = get_post( $post );
			}
			if ( is_a( $post, 'WP_Post' ) ) {
				preg_match( '/[^"]+\.(jpeg|jpg|png)/im', $post->post_content, $matches );
				if ( isset( $matches[0] ) ) {
					$image_url = esc_url_raw( $matches[0] );
				}
			}
		}

		return (string) $image_url;
	}
}
