<?php
namespace um_ext\um_mycred\core;

use um\core\Member_Directory_Meta;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Member_Directory
 *
 * @package um_ext\um_mycred\core
 */
class Member_Directory {

	/**
	 * @var array
	 */
	public $joined = array();

	/**
	 * Member_Directory constructor.
	 */
	public function __construct() {
		add_action( 'um_pre_directory_shortcode', array( &$this, 'directory_enqueue_scripts' ) );

		add_filter( 'um_members_directory_sort_fields', array( &$this, 'sort_dropdown_options' ) );

		add_filter( 'um_members_directory_custom_field_types_supported_filter', array( &$this, 'add_mycred_balance' ) );
		add_filter( 'um_custom_field_filter_type', array( &$this, 'add_mycred_balance_filter_type' ), 10, 2 );

		add_filter( 'um_members_directory_filter_fields', array( &$this, 'um_mycred_members_directory_filter_fields' ) );
		add_filter( 'um_members_directory_filter_types', array( &$this, 'um_mycred_directory_filter_types' ) );
		add_filter( 'um_member_directory_filter_slider_common', array( &$this, 'mycred_slider_range' ), 10, 3 );
		add_filter( 'um_member_directory_filter_slider_range_placeholder', array( &$this, 'mycred_slider_range_placeholder' ), 10, 2 );

		add_filter( 'um_search_fields', array( $this, 'mycred_rank_dropdown' ) );
		add_filter( 'um_query_args_filter_global', array( $this, 'mycred_filter_query' ), 10, 4 );

		add_filter( 'um_admin_extend_directory_options_profile', array( &$this, 'admin_directory_options_profile' ), 11 );

		//for grid
		add_action( 'um_members_just_after_name_tmpl', array( &$this, 'badges_tmpl' ), 1 );

		//for list
		add_action( 'um_members_list_after_user_name_tmpl', array( &$this, 'badges_tmpl' ), 1 );

		add_filter( 'um_ajax_get_members_data', array( &$this, 'extend_ajax_members_data' ), 50, 2 );

		add_filter( 'um_search_fields', array( &$this, 'change_filter_label' ), 10, 2 );

		add_filter( 'um_modify_sortby_parameter', array( &$this, 'sortby_points' ), 100, 2 );

		// UM metadata
		add_action( 'um_pre_users_query', array( &$this, 'add_sortby_custom' ), 10, 3 );
		add_filter( 'um_query_args_filter_global_meta', array( $this, 'mycred_filter_query_meta' ), 10, 6 );
	}

	/**
	 * @param $options
	 *
	 * @return mixed
	 */
	public function add_mycred_balance( $options ) {
		$options[] = 'mycred_balance';
		return $options;
	}

	/**
	 * @param $type
	 * @param $field_data
	 *
	 * @return string
	 */
	public function add_mycred_balance_filter_type( $type, $field_data ) {
		if ( isset( $field_data['type'] ) && 'mycred_balance' === $field_data['type'] ) {
			$type = 'slider';
		}

		return $type;
	}

	/**
	 * @param $query_args
	 * @param $sortby
	 *
	 * @uses mycred_get_types
	 *
	 * @return mixed
	 */
	public function sortby_points( $query_args, $sortby ) {
		$mycred_types = mycred_get_types();

		$order  = ( 0 === strpos( $sortby, 'most_' ) ) ? 'DESC' : 'ASC';
		$sortby = preg_replace( '/^(most_|least_)(.*?)/s', "$3", $sortby );

		if ( ! array_key_exists( $sortby, $mycred_types ) ) {
			return $query_args;
		}

		if ( empty( $query_args['meta_query'] ) ) {
			$query_args['meta_query'] = array();
		}

		$query_args['meta_query'][] = array(
			'relation'       => 'OR',
			array(
				'key'     => $sortby,
				'compare' => 'EXISTS',
				'type'    => 'NUMERIC',
			),
			'_mycred_points' => array(
				'key'     => $sortby,
				'compare' => 'NOT EXISTS',
				'type'    => 'NUMERIC',
			),
		);

		$query_args['orderby'] = array(
			'_mycred_points'  => $order,
			'user_registered' => 'DESC',
		);
		unset( $query_args['order'] );

		return $query_args;
	}

	/**
	 * Enqueue scripts
	 *
	 */
	public function directory_enqueue_scripts() {
		wp_enqueue_script( 'um_mycred' );
		wp_enqueue_style( 'um_mycred' );
	}

	/**
	 * @param $options
	 *
	 * @uses mycred_get_types
	 *
	 * @return mixed
	 */
	public function sort_dropdown_options( $options ) {
		$mycred_types = mycred_get_types();

		if ( ! empty( $mycred_types ) ) {
			foreach ( array_keys( $mycred_types ) as $point_type ) {
				// translators: %s: myCRED point type name.
				$options[ 'most_' . $point_type ] = sprintf( __( 'Most %s', 'um-mycred' ), mycred_get_point_type_name( $point_type, false ) );
				// translators: %s: myCRED point type name.
				$options[ 'least_' . $point_type ] = sprintf( __( 'Least %s', 'um-mycred' ), mycred_get_point_type_name( $point_type, false ) );
			}
		}

		return $options;
	}

	/**
	 * @param $options
	 *
	 * @return mixed
	 */
	public function um_mycred_members_directory_filter_fields( $options ) {
		if ( function_exists( 'mycred_have_ranks' ) ) {
			$options['mycred_rank'] = __( 'myCRED Rank', 'um-mycred' );
		}

		return $options;
	}

	/**
	 * @param array $attrs
	 *
	 * @return array
	 */
	public function change_filter_label( $attrs, $field_key ) {
		if ( 'mycred_rank' === $field_key ) {
			$attrs['label'] = __( 'Rank', 'um-mycred' );
		}
		return $attrs;
	}

	/**
	 * @param $filters
	 *
	 * @return mixed
	 */
	public function um_mycred_directory_filter_types( $filters ) {
		if ( function_exists( 'mycred_have_ranks' ) ) {
			$filters['mycred_rank'] = 'select';
		}

		return $filters;
	}

	public function mycred_slider_range( $range, $directory_data, $filter ) {
		$mycred_types = mycred_get_types();
		if ( ! empty( $mycred_types ) && array_key_exists( $filter, $mycred_types ) ) {
			if ( ! empty( $range ) ) {
				$range[0] = 0;

				if ( $range[0] == $range[1] ) {
					return false;
				}
			}
		}

		return $range;
	}

	/**
	 * @param bool|array $placeholder
	 * @param string     $filter
	 *
	 * @uses mycred_get_point_type_name
	 * @uses mycred_get_types
	 *
	 * @return bool|array
	 */
	public function mycred_slider_range_placeholder( $placeholder, $filter ) {
		$mycred_types = mycred_get_types();
		if ( ! empty( $mycred_types ) && array_key_exists( $filter, $mycred_types ) ) {
			return array(
				'<strong>' . esc_html__( 'Balance', 'um-mycred' ) . ':</strong>&nbsp;{value}&nbsp;<span class="um-points-wrap">' . esc_html( lcfirst( mycred_get_point_type_name( $filter, false ) ) ) . '</span>',
				'<strong>' . esc_html__( 'Balance', 'um-mycred' ) . ':</strong>&nbsp;{min_range} - {max_range}&nbsp;<span class="um-points-wrap">' . esc_html( lcfirst( mycred_get_point_type_name( $filter, false ) ) ) . '</span>',
			);
		}

		return $placeholder;
	}

	/**
	 * @param $attrs
	 *
	 * @return mixed
	 */
	public function mycred_rank_dropdown( $attrs ) {
		if ( isset( $attrs['metakey'] ) && 'mycred_rank' === $attrs['metakey'] ) {
			$all_ranks = mycred_get_ranks( 'publish', '-1' );

			$options = array();
			if ( ! empty( $all_ranks ) ) {
				foreach ( $all_ranks as $rank ) {
					$options[ $rank->post_id ] = $rank->post->post_title;
				}
			}

			$attrs['options'] = $options;
			$attrs['custom']  = true;
		}

		return $attrs;
	}

	/**
	 * @param $query
	 * @param $field
	 * @param $value
	 * @param $filter_type
	 *
	 * @return array
	 */
	public function mycred_filter_query( $query, $field, $value, $filter_type ) {
		$mycred_types = mycred_get_types();

		if ( ! empty( $mycred_types ) && array_key_exists( $field, $mycred_types ) ) {
			$query = array(
				'key'       => $field,
				'value'     => array_map( 'absint', $value ),
				'compare'   => 'BETWEEN',
				'type'      => 'NUMERIC',
				'inclusive' => true,
			);

			UM()->member_directory()->custom_filters_in_query[ $field ] = $value;
		}

		return $query;
	}

	/**
	 * Admin options for directory filtering
	 *
	 * @param $fields
	 *
	 * @return array
	 */
	public function admin_directory_options_profile( $fields ) {
		if ( ! function_exists( 'mycred_get_users_badges' ) ) {
			return $fields;
		}

		$fields = array_merge(
			array_slice( $fields, 0, 3 ),
			array(
				array(
					'id'    => '_um_mycred_hide_badges',
					'type'  => 'checkbox',
					'label' => __( 'Hide myCRED badges', 'um-mycred' ),
					'value' => UM()->query()->get_meta_value( '_um_mycred_hide_badges', null, 'na' ),
				),
			),
			array_slice( $fields, 3, count( $fields ) - 1 )
		);

		return $fields;
	}

	/**
	 * Display badges in Member Directories
	 *
	 * @param $args
	 */
	public function badges_tmpl( $args ) {
		if ( ! function_exists( 'mycred_get_users_badges' ) ) {
			return;
		}

		$hide_badges = ! empty( $args['mycred_hide_badges'] ) ? $args['mycred_hide_badges'] : ! UM()->options()->get( 'mycred_show_badges_in_members' );
		if ( empty( $hide_badges ) ) {
			?>
			<# if ( typeof user.badges !== 'undefined' ) { #>
				<# if ( user.badges !== '' ) { #>
					<div class="um-header" style="border:none;margin:initial;padding:initial;min-height:initial;">{{{user.badges}}}</div>
				<# } #>
			<# } #>
			<?php
		}
	}

	/**
	 * Extends AJAX member directory data
	 *
	 * @param $data_array
	 * @param $user_id
	 *
	 * @return mixed
	 */
	public function extend_ajax_members_data( $data_array, $user_id ) {
		$data_array['badges'] = apply_shortcodes( '[ultimatemember_user_badges user_id="' . $user_id . '"]' );
		return $data_array;
	}

	/**
	 * @param Member_Directory_Meta $query
	 * @param array $directory_data
	 * @param string $sortby
	 *
	 * @uses mycred_get_types
	 */
	public function add_sortby_custom( $query, $directory_data, $sortby ) {
		$mycred_types = mycred_get_types();

		$order = ( 0 === strpos( $sortby, 'most_' ) ) ? 'DESC' : 'ASC';
		$order = esc_sql( $order );

		$sortby = preg_replace( '/^(most_|least_)(.*?)/s', "$3", $sortby );

		if ( empty( $mycred_types ) || ! array_key_exists( $sortby, $mycred_types ) ) {
			return;
		}

		$this->maybe_add_join( $sortby, $query );

		$indexes = array_flip( array_keys( $this->joined ) );
		$index   = $indexes[ $sortby ];
		$index   = esc_sql( $index );

		$query->sql_order = " ORDER BY CAST( ummmycred{$index}.um_value AS SIGNED ) {$order}, u.user_registered DESC";
	}

	/**
	 * @param bool $skip
	 * @param Member_Directory_Meta $query
	 * @param $field
	 * @param $value
	 * @param $filter_type
	 * @param bool $is_default
	 *
	 * @uses mycred_get_types
	 *
	 * @return bool
	 */
	public function mycred_filter_query_meta( $skip, $query, $field, $value, $filter_type, $is_default ) {
		global $wpdb;

		$mycred_types = mycred_get_types();

		if ( ! empty( $mycred_types ) && array_key_exists( $field, $mycred_types ) ) {
			$skip = true;

			$min = min( $value );
			$max = max( $value );

			$this->maybe_add_join( $field, $query );

			$indexes = array_flip( array_keys( $this->joined ) );
			$index   = $indexes[ $field ];
			$index   = esc_sql( $index );

			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $index is static variable
			$query->where_clauses[] = $wpdb->prepare( "CAST( ummmycred{$index}.um_value AS SIGNED ) BETWEEN %d AND %d", $min, $max );

			if ( ! $is_default ) {
				$query->custom_filters_in_query[ $field ] = $value;
			}
		}

		return $skip;
	}

	private function maybe_add_join( $field, $query ) {
		global $wpdb;

		if ( empty( $this->joined[ $field ] ) ) {
			$this->joined[ $field ] = true;

			$indexes = array_flip( array_keys( $this->joined ) );
			$index   = $indexes[ $field ];
			$index   = esc_sql( $index );

			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $index is static variable
			$query->joins[] = $wpdb->prepare( "LEFT JOIN {$wpdb->prefix}um_metadata ummmycred{$index} ON ( ummmycred{$index}.user_id = u.ID AND ummmycred{$index}.um_key = %s )", $field );
		} else {
			$indexes = array_flip( array_keys( $this->joined ) );
			$index   = $indexes[ $field ];
			$index   = esc_sql( $index );
		}

		return $index;
	}
}
