<?php
namespace um_ext\um_mycred\core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Account
 *
 * @package um_ext\um_mycred\core
 */
class Account {


	/**
	 * Account constructor.
	 */
	public function __construct() {
		add_filter( 'um_account_page_default_tabs_hook', array( &$this, 'account_tab' ), 100 );
		add_filter( 'um_account_content_hook_points', array( &$this, 'points_tab_content' ) );

		add_action( 'um_submit_account_points_tab_errors_hook', array( &$this, 'transfer_errors' ), 10, 1 );
		add_filter( 'um_custom_error_message_handler', array( &$this, 'custom_error' ), 30, 2 );
		add_filter( 'um_custom_success_message_handler', array( &$this, 'custom_success' ), 10, 2 );

		add_action( 'mycred_update_user_balance', array( &$this, 'reset_cache' ), 9999, 4 );
	}

	/**
	 * Add tab to account page
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function account_tab( $tabs ) {
		$mycred_point_types = UM()->options()->get( 'mycred_point_types' );

		if ( empty( $mycred_point_types ) ) {
			return $tabs;
		}

		if ( count( $mycred_point_types ) > 1 ) {
			$label = __( 'My points', 'um-mycred' );
		} else {
			$mycred = mycred( $mycred_point_types[0] );
			$label  = $mycred->template_tags_general( __( 'My %_plural%', 'um-mycred' ) );
		}

		$tabs[1000]['points']['icon']         = 'um-faicon-trophy';
		$tabs[1000]['points']['title']        = $label;
		$tabs[1000]['points']['submit_title'] = $label;
		$tabs[1000]['points']['show_button']  = false;

		if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
			$tabs[1000]['points']['custom'] = true;
		}

		return $tabs;
	}

	/**
	 * Content to account tab
	 *
	 * @return string
	 */
	public function points_tab_content() {
		wp_enqueue_script( 'um_mycred' );
		wp_enqueue_style( 'um_mycred' );

		$mycred_referral_link = apply_filters( 'um_mycred_enable_referral_link', true );
		$user_id              = get_current_user_id();

		$t_args = compact( 'mycred_referral_link', 'user_id' );

		return UM()->get_template( 'account-points.php', um_mycred_plugin, $t_args );
	}

	/**
	 * Errors/Success for transferring points
	 *
	 * @param $args
	 */
	public function transfer_errors( $args ) {
		// phpcs:disable WordPress.Security.NonceVerification
		if ( ! empty( $_POST['mycred_transfer_uid'] ) && ! empty( $_POST['mycred_transfer_amount'] ) && ! empty( $_POST['mycred_transfer_type'] ) ) {

			$user       = sanitize_text_field( $_POST['mycred_transfer_uid'] );
			$amount     = sanitize_text_field( $_POST['mycred_transfer_amount'] );
			$point_type = sanitize_key( $_POST['mycred_transfer_type'] );

			// check if the current user can make points transfer
			if ( ! um_user( 'can_transfer_mycred' ) ) {
				$r = add_query_arg(
					array(
						'err'    => 'mycred_unauthorized',
						'points' => $point_type,
					),
					UM()->account()->tab_link( 'points' )
				);
				wp_safe_redirect( $r );
				exit;
			}

			//convert amount here to support , and . as decimal separators
			$dot_pos   = strrpos( $amount, '.' );
			$comma_pos = strrpos( $amount, ',' );
			$sep       = ( ( $dot_pos > $comma_pos ) && $dot_pos ) ? $dot_pos : ( ( ( $comma_pos > $dot_pos ) && $comma_pos ) ? $comma_pos : false );

			$string_array = str_split( $amount );

			if ( empty( $string_array[ $sep ] ) ) {
				$r = add_query_arg( 'err', 'mycred_invalid_amount', UM()->account()->tab_link( 'points' ) );
				wp_safe_redirect( $r );
				exit;
			}

			if ( $sep ) {
				$sep = $string_array[ $sep ];

				if ( ',' === $sep ) {
					$amount = str_replace( array( '.', ',' ), array( '', '.' ), $amount );
				} elseif ( '.' === $sep ) {
					$amount = str_replace( ',', '', $amount );
				}
			}

			$amount = (float) $amount;

			if ( ! is_numeric( $amount ) ) {
				$r = add_query_arg( 'err', 'mycred_invalid_amount', UM()->account()->tab_link( 'points' ) );
				wp_safe_redirect( $r );
				exit;
			}

			if ( $amount > UM()->myCRED()->points()->get_points_clean( get_current_user_id(), $point_type ) ) {
				$r = add_query_arg( 'err', 'mycred_not_enough_balance', UM()->account()->tab_link( 'points' ) );
				wp_safe_redirect( $r );
				exit;
			}

			if ( is_numeric( $user ) ) {
				if ( get_current_user_id() === $user ) {
					$r = add_query_arg(
						array(
							'err'    => 'mycred_myself',
							'points' => $point_type,
						),
						UM()->account()->tab_link( 'points' )
					);
					wp_safe_redirect( $r );
					exit;
				}

				if ( ! UM()->common()->users()::user_exists( $user ) ) {
					$r = add_query_arg( 'err', 'mycred_invalid_user', UM()->account()->tab_link( 'points' ) );
					wp_safe_redirect( $r );
					exit;
				}

				$user_id = $user;
			} else {
				if ( ! username_exists( $user ) && ! email_exists( $user ) ) {
					$r = add_query_arg( 'err', 'mycred_invalid_user', UM()->account()->tab_link( 'points' ) );
					wp_safe_redirect( $r );
					exit;
				}

				if ( is_email( $user ) ) {
					$user_id = email_exists( $user );
				} else {
					$user_id = username_exists( $user );
				}
			}

			// check if user can receive points
			um_fetch_user( $user_id );
			if ( um_user( 'cannot_receive_mycred' ) ) {
				$r = add_query_arg(
					array(
						'err'    => 'mycred_cant_receive',
						'points' => $point_type,
					),
					UM()->account()->tab_link( 'points' )
				);
				wp_safe_redirect( $r );
				exit;
			}

			UM()->myCRED()->points()->transfer( get_current_user_id(), $user_id, $amount, $point_type );
			$r = add_query_arg(
				array(
					'updated' => 'mycred_transfer_done',
					'points'  => $point_type,
				),
				UM()->account()->tab_link( 'points' )
			);
			wp_safe_redirect( $r );
			exit;

		}
		// phpcs:enable WordPress.Security.NonceVerification
	}

	/**
	 * Custom Error Message on upgrade account page
	 *
	 * @param string $msg
	 * @param string $err
	 *
	 * @return string
	 */
	public function custom_error( $msg, $err ) {
		if ( 'mycred_invalid_amount' === $err ) {
			$msg = __( 'Invalid amount.', 'um-mycred' );
		} elseif ( 'mycred_invalid_user' === $err ) {
			$msg = __( 'The user does not exist.', 'um-mycred' );
		} elseif ( 'mycred_not_enough_balance' === $err ) {
			$msg = __( 'You do not have enough balance.', 'um-mycred' );
		} elseif ( in_array( $err, array( 'mycred_cant_receive', 'mycred_myself', 'mycred_unauthorized' ), true ) ) {
			$point_type_default = defined( 'MYCRED_DEFAULT_TYPE_KEY' ) ? MYCRED_DEFAULT_TYPE_KEY : 'mycred_default';
			$mycred             = mycred( $point_type_default );

			if ( ! empty( $_GET['points'] ) ) {
				$mycred_point_types = UM()->options()->get( 'mycred_point_types' );

				if ( in_array( $point_type = sanitize_key( $_GET['points'] ), $mycred_point_types, true ) ) {
					$mycred = mycred( $point_type );
				}
			}

			if ( 'mycred_cant_receive' === $err ) {
				$msg = $mycred->template_tags_general( __( 'That user can not receive <span class="um-points-wrap">%_plural%</span>.', 'um-mycred' ) );
			} elseif ( 'mycred_myself' === $err ) {
				$msg = $mycred->template_tags_general( __( 'You can not transfer <span class="um-points-wrap">%_plural%</span> to yourself.', 'um-mycred' ) );
			} elseif ( 'mycred_unauthorized' === $err ) {
				$msg = $mycred->template_tags_general( __( 'You are not allowed to transfer <span class="um-points-wrap">%_plural%</span>.', 'um-mycred' ) );
			}
		}

		return $msg;
	}

	/**
	 * Custom Success Message on upgrade account page
	 *
	 * @param string $msg
	 * @param string $success
	 *
	 * @return string
	 */
	public function custom_success( $msg, $success ) {
		if ( 'mycred_transfer_done' === $success ) {
			$point_type_default = defined( 'MYCRED_DEFAULT_TYPE_KEY' ) ? MYCRED_DEFAULT_TYPE_KEY : 'mycred_default';
			$mycred             = mycred( $point_type_default );

			if ( ! empty( $_GET['points'] ) ) {
				$mycred_point_types = UM()->options()->get( 'mycred_point_types' );

				if ( in_array( $point_type = sanitize_key( $_GET['points'] ), $mycred_point_types, true ) ) {
					$mycred = mycred( $point_type );
				}
			}

			$msg = $mycred->template_tags_general( __( '%plural% transferred successfully', 'um-mycred' ) );
		}

		return $msg;
	}

	/**
	 * Reset user cached balance
	 *
	 * @param int $user_id
	 * @param $current_balance
	 * @param $amount
	 * @param $type
	 */
	public function reset_cache( $user_id, $current_balance, $amount, $type ) {
		delete_option( "um_cache_userdata_{$user_id}" );
	}
}
