<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * More profile privacy options.
 *
 * @param array $options
 *
 * @return array
 */
function um_followers_profile_privacy_options( $options ) {
	$options = array_merge(
		$options,
		array(
			'followed' => __( 'Only people I follow can view my profile', 'um-followers' ),
			'follower' => __( 'Followers', 'um-followers' ),
		)
	);

	return $options;
}
add_filter( 'um_profile_privacy_options', 'um_followers_profile_privacy_options', 100, 1 );

/**
 * Show field only for followers.
 *
 * @param bool  $can_view
 * @param array $data
 *
 * @return bool
 */
function um_followers_can_view_field( $can_view, $data ) {
	if ( isset( $data['public'] ) && ( '-5' === $data['public'] || '-6' === $data['public'] ) ) {
		if ( ! is_user_logged_in() ) {
			$can_view = false;
		} else {
			if ( ! um_is_user_himself() && ! UM()->Followers_API()->api()->followed( um_get_requested_user(), get_current_user_id() ) ) {
				$can_view = apply_filters( 'um_followers_not_follower_maybe_other', false, $data );
			}
		}
	}

	return $can_view;
}
add_filter( 'um_can_view_field_custom', 'um_followers_can_view_field', 10, 2 );

/**
 * Show field 'only for friends and followers',
 * case if not friend maybe follower.
 *
 * @param bool  $can_view
 * @param array $data
 *
 * @return bool
 */
function um_followers_not_friend_maybe_other( $can_view, $data ) {
	if ( isset( $data['public'] ) && '-6' === $data['public'] ) {
		if ( UM()->Followers_API()->api()->followed( um_get_requested_user(), get_current_user_id() ) ) {
			$can_view = true;
		}
	}

	return $can_view;
}
add_filter( 'um_friends_not_friend_maybe_other', 'um_followers_not_friend_maybe_other', 10, 2 );

/**
 * More messaging privacy options.
 *
 * @param array $options
 *
 * @return array
 */
function um_followers_messaging_privacy_options( $options ) {
	$options['followed'] = __( 'Only people I follow', 'um-followers' );
	$options['follower'] = __( 'Followers', 'um-followers' );
	return $options;
}
add_filter( 'um_messaging_privacy_options', 'um_followers_messaging_privacy_options', 10, 1 );
add_filter( 'um_user_notes_privacy_options_dropdown', 'um_followers_messaging_privacy_options', 10, 1 );
add_filter( 'um_user_photos_privacy_options_dropdown', 'um_followers_messaging_privacy_options', 10, 1 );

/**
 * @param bool $can_view
 * @param string $privacy
 * @param int $user_id
 *
 * @return bool
 */
function um_user_notes_followed_privacy( $can_view, $privacy, $user_id ) {
	if ( 'followed' !== $privacy ) {
		return $can_view;
	}

	if ( ! is_user_logged_in() ) {
		return false;
	}

	if ( ! UM()->Followers_API()->api()->followed( $user_id, um_profile_id() ) ) {
		return false;
	}

	return $can_view;
}
add_filter( 'um_user_notes_custom_privacy', 'um_user_notes_followed_privacy', 10, 3 );

/**
 * @param bool $can_view
 * @param string $privacy
 * @param int $user_id
 *
 * @return bool
 */
function um_user_notes_followers_privacy( $can_view, $privacy, $user_id ) {
	if ( 'follower' !== $privacy ) {
		return $can_view;
	}

	if ( ! is_user_logged_in() ) {
		return false;
	}

	if ( ! UM()->Followers_API()->api()->followed( um_profile_id(), $user_id ) ) {
		return false;
	}

	return $can_view;
}
add_filter( 'um_user_notes_custom_privacy', 'um_user_notes_followers_privacy', 10, 3 );

/**
 * @param bool $can_view
 * @param string $privacy
 * @param int $user_id
 * @param int $profile_id
 *
 * @return bool
 */
function um_user_photos_followed_privacy( $can_view, $privacy, $user_id, $profile_id ) {
	if ( 'followed' === $privacy && ! UM()->Followers_API()->api()->followed( $user_id, $profile_id ) ) {
		return false;
	}

	return $can_view;
}
add_filter( 'um_user_photos_custom_privacy', 'um_user_photos_followed_privacy', 10, 4 );

/**
 * @param bool $can_view
 * @param string $privacy
 * @param int $user_id
 * @param int $profile_id
 *
 * @return bool
 */
function um_user_photos_followers_privacy( $can_view, $privacy, $user_id, $profile_id ) {
	if ( 'followers' === $privacy && ! UM()->Followers_API()->api()->followed( $profile_id, $user_id ) ) {
		return false;
	}

	return $can_view;
}
add_filter( 'um_user_photos_custom_privacy', 'um_user_photos_followers_privacy', 10, 4 );

/**
 * @param array    $query
 * @param null|int $user_id
 * @param null|int $profile_id
 *
 * @return array
 */
function um_user_notes_add_followed_to_meta_query( $query, $user_id, $profile_id ) {
	if ( empty( $user_id ) ) {
		return $query;
	}

	if ( absint( $user_id ) === absint( $profile_id ) || UM()->Followers_API()->api()->followed( $user_id, $profile_id ) ) {
		$query[] = array(
			array(
				'key'     => '_privacy',
				'value'   => 'followed',
				'compare' => '=',
			),
		);
	}

	return $query;
}
add_filter( 'um_user_notes_change_meta_query', 'um_user_notes_add_followed_to_meta_query', 10, 3 );

/**
 * @param array    $query
 * @param null|int $user_id
 * @param null|int $profile_id
 *
 * @return array
 */
function um_user_notes_add_followers_to_meta_query( $query, $user_id, $profile_id ) {
	if ( empty( $user_id ) ) {
		return $query;
	}

	if ( absint( $user_id ) === absint( $profile_id ) || UM()->Followers_API()->api()->followed( $profile_id, $user_id ) ) {
		$query[] = array(
			array(
				'key'     => '_privacy',
				'value'   => 'follower',
				'compare' => '=',
			),
		);
	}

	return $query;
}
add_filter( 'um_user_notes_change_meta_query', 'um_user_notes_add_followers_to_meta_query', 10, 3 );

/**
 * @param array    $query
 * @param null|int $user_id
 * @param null|int $profile_id
 *
 * @return array
 */
function um_user_photos_add_followed_to_meta_query( $query, $user_id, $profile_id ) {
	if ( empty( $user_id ) ) {
		return $query;
	}

	if ( absint( $user_id ) === absint( $profile_id ) || UM()->Followers_API()->api()->followed( $user_id, $profile_id ) ) {
		$query[] = array(
			array(
				'key'     => '_privacy',
				'value'   => 'followed',
				'compare' => '=',
			),
		);
	}

	return $query;
}
add_filter( 'um_user_photos_change_meta_query', 'um_user_photos_add_followed_to_meta_query', 10, 3 );

/**
 * @param array    $query
 * @param null|int $user_id
 * @param null|int $profile_id
 *
 * @return array
 */
function um_user_photos_add_followers_to_meta_query( $query, $user_id, $profile_id ) {
	if ( empty( $user_id ) ) {
		return $query;
	}

	if ( absint( $user_id ) === absint( $profile_id ) || UM()->Followers_API()->api()->followed( $profile_id, $user_id ) ) {
		$query[] = array(
			array(
				'key'     => '_privacy',
				'value'   => 'follower',
				'compare' => '=',
			),
		);
	}

	return $query;
}
add_filter( 'um_user_photos_change_meta_query', 'um_user_photos_add_followers_to_meta_query', 10, 3 );

function um_user_photos_remove_followers_as_visible( $privacy ) {
	$privacy = array_flip( $privacy );
	unset( $privacy['followed'], $privacy['follower'] );
	return array_keys( $privacy );
}
add_filter( 'um_user_photos_accessible_by_author_privacy', 'um_user_photos_remove_followers_as_visible' );
add_filter( 'um_user_notes_accessible_by_author_privacy', 'um_user_photos_remove_followers_as_visible' );

/**
 * Filters an array of albums current user can access.
 *
 * @param array $public_ids An array of public albums.
 *
 * @return array An array of albums current user can access.
 */
function um_user_photo_albums_public_for_followed( $public_ids ) {
	if ( is_user_logged_in() ) {

		// Get followers.
		$followers = UM()->Followers_API()->api()->followers( get_current_user_id() );
		if ( $followers && is_array( $followers ) ) {

			// Prepare followers array.
			$author__in = array_unique( $followers );

			// Get albums with the "Only people I follow" privacy.
			$args = array(
				'author__in'  => $author__in,
				'fields'      => 'ids',
				'post_type'   => 'um_user_photos',
				'post_status' => 'publish',
				'meta_query'  => array(
					'privacy' => array(
						'relation' => 'OR',
						array(
							'key'     => '_privacy',
							'value'   => 'followed',
							'compare' => '=',
						),
					),
				),
			);

			$ids        = get_posts( $args );
			$public_ids = array_merge( $public_ids, $ids );
		}
	}
	return $public_ids;
}
add_filter( 'um_user_photo_albums_public_ids', 'um_user_photo_albums_public_for_followed' );


/**
 * Filters an array of albums current user can access.
 *
 * @param array $public_ids An array of public albums.
 *
 * @return array An array of albums current user can access.
 */
function um_user_photo_albums_public_for_followers( $public_ids ) {
	if ( is_user_logged_in() ) {

		// Get followed.
		$followings = UM()->Followers_API()->api()->following( get_current_user_id() );
		if ( $followings && is_array( $followings ) ) {

			// Prepare followed array.
			$author__in = array_unique( $followings );

			// Get albums with the "Followers" privacy.
			$args = array(
				'author__in'  => $author__in,
				'fields'      => 'ids',
				'post_type'   => 'um_user_photos',
				'post_status' => 'publish',
				'meta_query'  => array(
					'privacy' => array(
						'relation' => 'OR',
						array(
							'key'     => '_privacy',
							'value'   => 'follower',
							'compare' => '=',
						),
					),
				),
			);

			$ids        = get_posts( $args );
			$public_ids = array_merge( $public_ids, $ids );
		}
	}
	return $public_ids;
}
add_filter( 'um_user_photo_albums_public_ids', 'um_user_photo_albums_public_for_followers' );


/**
 * Add a hidden tab.
 *
 * @param array $tabs
 *
 * @return array
 */
function um_followers_add_tabs( $tabs ) {
	if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
		$tabs['followers'] = array(
			'name'     => __( 'Followers', 'um-followers' ),
			'notifier' => 0,
		);
		$tabs['following'] = array(
			'name'     => __( 'Following', 'um-followers' ),
			'notifier' => 0,
		);
	} else {
		$tabs['followers'] = array(
			'hidden' => true,
		);

		$tabs['following'] = array(
			'hidden' => true,
		);
	}

	return $tabs;
}
add_filter( 'um_profile_tabs', 'um_followers_add_tabs', 2000 );

/**
 * Add tabs based on user
 *
 * @param array $tabs
 *
 * @return array
 */
function um_followers_user_add_tab( $tabs ) {
	if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
		$user_id = um_user( 'ID' );
		if ( ! $user_id ) {
			return $tabs;
		}

		if ( ! empty( $tabs['followers'] ) ) {
			$tabs['followers']['notifier'] = UM()->Followers_API()->api()->count_followers_plain( $user_id );
		}

		if ( ! empty( $tabs['following'] ) ) {
			$tabs['following']['notifier'] = UM()->Followers_API()->api()->count_following_plain( $user_id );
		}
	} else {
		return $tabs;
	}

	return $tabs;
}
add_filter( 'um_user_profile_tabs', 'um_followers_user_add_tab', 1000 );

/**
 * Check if user can view user profile.
 *
 * @param string $can_view
 * @param int    $user_id
 *
 * @return string
 */
function um_followers_can_view_main( $can_view, $user_id ) {
	if ( ! is_user_logged_in() || get_current_user_id() != $user_id ) {
		$is_private_case_old = UM()->user()->is_private_case( $user_id, __( 'Followers', 'um-followers' ) );
		$is_private_case     = UM()->user()->is_private_case( $user_id, 'follower' );
		if ( $is_private_case || $is_private_case_old ) {
			$can_view = __( 'You must follow this user to view profile', 'um-followers' );
		}
		$is_private_case_old = UM()->user()->is_private_case( $user_id, __( 'Only people I follow can view my profile', 'um-followers' ) );
		$is_private_case     = UM()->user()->is_private_case( $user_id, 'followed' );
		if ( $is_private_case || $is_private_case_old ) {
			$can_view = __( 'You cannot view this profile because the user has not followed you', 'um-followers' );
		}
	}

	return $can_view;
}
add_filter( 'um_profile_can_view_main', 'um_followers_can_view_main', 10, 2 );

/**
 * Test case to hide profile
 *
 * @param bool   $default
 * @param string $option
 * @param int    $user_id
 *
 * @return bool
 */
function um_followers_private_filter_hook( $default, $option, $user_id ) {
	// user selected this option in privacy
	if ( 'follower' === $option || $option == __( 'Followers', 'um-followers' ) ) {
		if ( ! UM()->Followers_API()->api()->followed( $user_id, get_current_user_id() ) ) {
			return true;
		}
	}

	if ( 'followed' === $option || $option == __( 'Only people I follow can view my profile', 'um-followers' ) ) {
		if ( ! UM()->Followers_API()->api()->followed( get_current_user_id(), $user_id ) ) {
			return true;
		}
	}

	return $default;
}
add_filter( 'um_is_private_filter_hook', 'um_followers_private_filter_hook', 100, 3 );

/**
 * @param string $content
 * @param int $user_id
 * @param int $post_id
 *
 * @return mixed
 */
function um_followers_activity_mention_integration( $content, $user_id, $post_id ) {
	if ( ! UM()->options()->get( 'activity_followers_mention' ) ) {
		return $content;
	}

	$mention           = array();
	$mentioned_in_post = get_post_meta( $post_id, '_mentioned', true );
	$mentioned_in_post = $mentioned_in_post ? $mentioned_in_post : array();

	$following = UM()->Followers_API()->api()->following( $user_id );
	if ( ! empty( $following ) ) {
		$following_names = array();
		foreach ( $following as $user_id1 ) {
			um_fetch_user( $user_id1 );
			$display_name = um_user( 'display_name' );
			if ( empty( $display_name ) ) {
				continue;
			}
			$following_names[ $user_id1 ] = $display_name;
		}

		uasort(
			$following_names,
			static function( $a, $b ) {
				return strlen( $b ) - strlen( $a );
			}
		);

		foreach ( $following_names as $user_id1 => $name ) {
			preg_match( '/(^|\s)(@' . $name . ')($|\s)/um', $content, $matches );

			if ( empty( $matches[2] ) ) {
				continue;
			}

			$user_mentioned_in_post = false;
			if ( ! empty( $mentioned_in_post ) && in_array( $user_id1, $mentioned_in_post, true ) ) {
				$user_mentioned_in_post = true;
			}

			um_fetch_user( $user_id1 );
			$content = preg_replace( '/(^|\s)@(' . $name . ')($|\s)/um', '$1<a href="' . esc_url( um_user_profile_url() ) . '" class="um-link um-user-tag">$2</a>$3', $content, -1, $replacements );

			if ( false === $user_mentioned_in_post && ! empty( $replacements ) ) {
				do_action( 'um_following_new_mention', $user_id, $user_id1, $post_id );

				$mention[] = $user_id1;
			}
		}
	}

	$followers = UM()->Followers_API()->api()->followers( $user_id );
	if ( ! empty( $followers ) ) {
		$followers_names = array();
		foreach ( $followers as $user_id1 ) {
			um_fetch_user( $user_id1 );
			$display_name = um_user( 'display_name' );
			if ( empty( $display_name ) ) {
				continue;
			}
			$followers_names[ $user_id1 ] = $display_name;
		}

		uasort(
			$followers_names,
			static function( $a, $b ) {
				return strlen( $b ) - strlen( $a );
			}
		);

		foreach ( $followers_names as $user_id1 => $name ) {
			preg_match( '/(^|\s)(@' . $name . ')($|\s)/um', $content, $matches );

			if ( empty( $matches[2] ) ) {
				continue;
			}

			$user_mentioned_in_post = false;
			if ( ! empty( $mentioned_in_post ) && in_array( $user_id1, $mentioned_in_post, true ) ) {
				$user_mentioned_in_post = true;
			}

			um_fetch_user( $user_id1 );

			$content = preg_replace( '/(^|\s)@(' . $name . ')($|\s)/um', '$1<a href="' . esc_url( um_user_profile_url() ) . '" class="um-link um-user-tag">$2</a>$3', $content, -1, $replacements );

			if ( false === $user_mentioned_in_post && ! empty( $replacements ) ) {
				do_action( 'um_followers_new_mention', $user_id, $user_id1, $post_id );

				$mention[] = $user_id1;
			}
		}
	}

	if ( ! empty( $mention ) ) {
		$mention = array_merge( $mentioned_in_post, $mention );
		update_post_meta( $post_id, '_mentioned', $mention );
	}

	return $content;
}
add_filter( 'um_activity_mention_integration', 'um_followers_activity_mention_integration', 20, 3 );

/**
 * Lock activity wall for followers only.
 *
 * @param bool $can_view
 * @param int  $profile_id
 *
 * @return bool|string
 */
function um_wall_can_view_follower( $can_view, $profile_id ) {
	if ( $profile_id == get_current_user_id() ) {
		return true;
	}

	$privacy = get_user_meta( $profile_id, 'wall_privacy', true );

	if ( $privacy == 3 ) {
		if ( ! UM()->Followers_API()->api()->followed( $profile_id, get_current_user_id() ) ) {
			return __( 'You must follow this user to view their social activity', 'um-followers' );
		}
	} elseif ( $privacy == 4 ) {
		if ( ! UM()->Followers_API()->api()->followed( get_current_user_id(), $profile_id ) ) {
			return __( 'This user must follow you before you can view their wall', 'um-followers' );
		}
	}

	return $can_view;
}
add_filter( 'um_wall_can_view', 'um_wall_can_view_follower', 11, 2 );

function um_wall_can_view_follower_mysql( $private_walls ) {
	global $wpdb;

	if ( is_user_logged_in() ) {
		$following_users = UM()->Followers_API()->api()->following( get_current_user_id() );
		if ( ! empty( $following_users ) ) {
			$private_walls2 = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT user_id
					FROM {$wpdb->usermeta}
					WHERE meta_key='wall_privacy' AND
						  meta_value='3' AND
						  user_id != %d AND
						  user_id NOT IN('" . implode( "','", $following_users ) . "')",
					get_current_user_id()
				)
			);

			if ( ! empty( $private_walls2 ) ) {
				$private_walls = array_merge( $private_walls, $private_walls2 );
				$private_walls = array_unique( $private_walls );
			}
		}

		$followers_users = UM()->Followers_API()->api()->followers( get_current_user_id() );
		if ( ! empty( $followers_users ) ) {
			$private_walls3 = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT user_id
					FROM {$wpdb->usermeta}
					WHERE meta_key='wall_privacy' AND
						  meta_value='4' AND
						  user_id != %d AND
						  user_id NOT IN('" . implode( "','", $followers_users ) . "')",
					get_current_user_id()
				)
			);

			if ( ! empty( $private_walls3 ) ) {
				$private_walls = array_merge( $private_walls, $private_walls3 );
				$private_walls = array_unique( $private_walls );
			}
		}
	} else {
		$private_walls2 = $wpdb->get_col(
			"SELECT DISTINCT user_id
			FROM {$wpdb->usermeta}
			WHERE meta_key='wall_privacy' AND
			      (meta_value='3' OR meta_value='4')"
		);

		if ( ! empty( $private_walls2 ) ) {
			$private_walls = array_merge( $private_walls, $private_walls2 );
			$private_walls = array_unique( $private_walls );
		}
	}

	return $private_walls;
}
add_filter( 'um_activity_wall_private_walls', 'um_wall_can_view_follower_mysql' );

/**
 * @param string|array $classes
 * @return string|array
 */
function um_followers_profile_navbar_classes( $classes ) {
	if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
		$classes[] = 'um-has-followers-bar';
	} else {
		$classes .= ' um-followers-bar';
	}
	return $classes;
}
add_filter( 'um_profile_navbar_classes', 'um_followers_profile_navbar_classes', 10, 1 );

/**
 * @param bool   $restrict
 * @param string $who_can_pm
 * @param int    $recipient
 *
 * @return bool
 */
function um_followers_can_message_restrict( $restrict, $who_can_pm, $recipient ) {
	// only people I follow
	if ( 'followed' === $who_can_pm ) {
		if ( ! UM()->Followers_API()->api()->followed( get_current_user_id(), $recipient ) ) {
			return true;
		}
	}

	// followers can message
	if ( 'follower' === $who_can_pm ) {
		if ( ! UM()->Followers_API()->api()->followed( $recipient, get_current_user_id() ) ) {
			return true;
		}
	}

	return $restrict;
}
add_filter( 'um_messaging_can_message_restrict', 'um_followers_can_message_restrict', 10, 3 );

/**
 * Add options for profile tabs' privacy
 *
 * @param array $options
 *
 * @return array
 */
function um_followers_profile_tabs_privacy_options( $options ) {
	$options[6] = __( 'Only followers', 'um-followers' );

	// check if there is 'only friends' option
	if ( isset( $options[7] ) ) {
		$options[8] = __( 'Only friends and followers', 'um-followers' );
	}

	return $options;
}
add_filter( 'um_profile_tabs_privacy_list', 'um_followers_profile_tabs_privacy_options', 10, 1 );

/**
 * Show profile tab only for friends
 *
 * @param bool $can_view
 * @param int $privacy
 * @param string $tab
 * @param array $tab_data
 * @param int $user_id
 *
 * @return bool
 */
function um_followers_can_view_profile_tab( $can_view, $privacy, $tab, $tab_data, $user_id ) {
	if ( ! in_array( $privacy, array( 6, 8 ) ) ) {
		return $can_view;
	}

	if ( ! is_user_logged_in() ) {
		$can_view = false;
	} else {
		if ( get_current_user_id() === $user_id ) {
			$can_view = false;
		} else {
			if ( ! UM()->Followers_API()->api()->followed( $user_id, get_current_user_id() ) ) {
				$can_view = apply_filters( 'um_followers_profile_tab_not_follower_maybe_other', false, $privacy, $user_id );
			}
		}
	}

	return $can_view;
}
add_filter( 'um_profile_menu_can_view_tab', 'um_followers_can_view_profile_tab', 10, 5 );

/**
 * Show profile tab 'only for friends and followers',
 * case if not friend maybe follower
 *
 * @param bool $can_view
 * @param int $privacy
 * @param int $user_id
 *
 * @return bool
 */
function um_followers_profile_tab_not_friend_maybe_other( $can_view, $privacy, $user_id ) {
	if ( $privacy == 8 ) {
		if ( UM()->Followers_API()->api()->followed( $user_id, get_current_user_id() ) ) {
			$can_view = true;
		}
	}

	return $can_view;
}
add_filter( 'um_friends_profile_tab_not_friend_maybe_other', 'um_followers_profile_tab_not_friend_maybe_other', 10, 3 );
