<?php
namespace um_ext\um_followers\core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Followers_Main_API
 * @package um_ext\um_followers\core
 */
class Followers_Main_API {

	/**
	 * DB table name.
	 *
	 * @var string
	 */
	public $table_name = '';

	/**
	 * Followers_Main_API constructor.
	 */
	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'um_followers';
	}

	/**
	 * @param array $args
	 *
	 * @return array
	 */
	public function rest_get_following( $args ) {
		$response = array();
		$error    = array();

		if ( empty( $args['id'] ) ) {
			$error['error'] = esc_html__( 'You must provide a user ID', 'um-followers' );
			return $error;
		}

		$results = $this->following( $args['id'] );
		if ( ! $results ) {
			$error['error'] = esc_html__( 'No users were found', 'um-followers' );
			return $error;
		}
		$response['following']['count'] = $this->count_following_plain( $args['id'] );
		foreach ( $results as $k => $user_id ) {
			$user = get_userdata( $user_id );

			$response['following']['users'][ $k ]['ID']           = $user_id;
			$response['following']['users'][ $k ]['username']     = $user->user_login;
			$response['following']['users'][ $k ]['display_name'] = $user->display_name;
		}

		return $response;
	}

	/**
	 * @param array $args
	 *
	 * @return array
	 */
	public function rest_get_followers( $args ) {
		$response = array();
		$error    = array();

		if ( empty( $args['id'] ) ) {
			$error['error'] = esc_html__( 'You must provide a user ID', 'um-followers' );
			return $error;
		}

		$results = $this->followers( $args['id'] );
		if ( ! $results ) {
			$error['error'] = esc_html__( 'No users were found', 'um-followers' );
			return $error;
		}
		$response['followers']['count'] = $this->count_followers_plain( $args['id'] );
		foreach ( $results as $k => $user_id ) {
			$user = get_userdata( $user_id );

			$response['followers']['users'][ $k ]['ID']           = $user_id;
			$response['followers']['users'][ $k ]['username']     = $user->user_login;
			$response['followers']['users'][ $k ]['display_name'] = $user->display_name;
		}

		return $response;
	}

	/**
	 * Checks if user enabled email notification
	 *
	 * @param int $user_id
	 *
	 * @return bool|int
	 */
	public function enabled_email( $user_id ) {
		$_enable_new_follow = true;
		if ( get_user_meta( $user_id, '_enable_new_follow', true ) == 'yes' ) {
			$_enable_new_follow = 1;
		} else if ( get_user_meta( $user_id, '_enable_new_follow', true ) == 'no' ) {
			$_enable_new_follow = 0;
		}
		return $_enable_new_follow;
	}

	/**
	 * Show the followers list URL
	 *
	 * @param int $user_id
	 *
	 * @return bool|string
	 */
	public function followers_link( $user_id ) {
		$nav_link = um_user_profile_url( $user_id );
		$nav_link = add_query_arg( 'profiletab', 'followers', $nav_link );
		return $nav_link;
	}

	/**
	 * Show the following list URL
	 *
	 * @param int $user_id
	 *
	 * @return bool|string
	 */
	public function following_link( $user_id ) {
		$nav_link = um_user_profile_url( $user_id );
		$nav_link = add_query_arg( 'profiletab', 'following', $nav_link );
		return $nav_link;
	}

	/**
	 * Show the follow button for two users.
	 *
	 * @param int $user_id1
	 * @param int $user_id2
	 *
	 * @return string
	 */
	public function follow_button( $user_id1, $user_id2 ) {
		$hide_follow_button    = apply_filters( 'um_followers_hide_button', false );
		$allow_admin_to_follow = UM()->options()->get( 'followers_allow_admin_to_follow' );
		if ( $hide_follow_button || ( current_user_can( 'manage_options' ) && empty( $allow_admin_to_follow ) ) ) {
			return '';
		}

		if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
			$t_args = array(
				'redirect'   => '',
				'can_follow' => false,
				'followed'   => false,
				'user_id1'   => absint( $user_id1 ),
				'user_id2'   => absint( $user_id2 ),
			);

			if ( ! is_user_logged_in() ) {
				$redirect = um_get_core_page( 'login' );

				if ( UM()->is_request( 'ajax' ) ) {
					if ( isset( $_REQUEST['post_refferer'] ) ) {
						$link = get_permalink( absint( $_REQUEST['post_refferer'] ) );
					}

					if ( empty( $link ) ) {
						$link = um_get_core_page( 'members' );
					}
				} else {
					$link = UM()->permalinks()->get_current_url();
				}

				$redirect           = add_query_arg( 'redirect_to', $link, $redirect );
				$t_args['redirect'] = apply_filters( 'um_followers_button_redirect_url', $redirect );
			} else {

				$t_args['can_follow'] = $this->can_follow( $user_id1, $user_id2 );
				if ( $t_args['can_follow'] ) {
					$t_args['followed'] = $this->followed( $user_id1, $user_id2 );
				}
			}

			wp_enqueue_script( 'um_followers' );
			wp_enqueue_style( 'um_followers' );

			return UM()->get_template( 'v3/button.php', um_followers_plugin, $t_args );
		}

		wp_enqueue_script( 'um_followers' );
		wp_enqueue_style( 'um_followers' );

		if ( ! is_user_logged_in() ) {
			$redirect = um_get_core_page( 'login' );

			if ( UM()->is_request( 'ajax' ) ) {
				if ( isset( $_REQUEST['post_refferer'] ) ) {
					$link = get_permalink( absint( $_REQUEST['post_refferer'] ) );
				}

				if ( empty( $link ) ) {
					$link = um_get_core_page( 'members' );
				}
			} else {
				$link = UM()->permalinks()->get_current_url();
			}

			$redirect = add_query_arg( 'redirect_to', $link, $redirect );
			$redirect = apply_filters( 'um_followers_button_redirect_url', $redirect );
			return '<a href="' . esc_url( $redirect ) . '" class="um-login-to-follow-btn um-button um-alt">' . esc_html__( 'Follow', 'um-followers' ) . '</a>';
		}

		$res = '';
		if ( $this->can_follow( $user_id1, $user_id2 ) ) {
			if ( ! $this->followed( $user_id1, $user_id2 ) ) {
				$res = '<a href="javascript:void(0);" class="um-follow-btn um-button um-alt" data-user_id="' . esc_attr( $user_id1 ) . '">' . esc_html__( 'Follow', 'um-followers' ) . '</a>';
			} else {
				$res = '<a href="javascript:void(0);" class="um-unfollow-btn um-button" data-user_id="' . esc_attr( $user_id1 ) . '" data-following="' . esc_attr__( 'Following', 'um-followers' ) . '" data-unfollow="' . esc_attr__( 'Unfollow', 'um-followers' ) . '">' . esc_html__( 'Following', 'um-followers' ) . '</a>';
			}
		}
		return $res;
	}

	/**
	 * If user can follow.
	 *
	 * @param int $user_id1
	 * @param int $user_id2
	 *
	 * @return bool
	 */
	public function can_follow( $user_id1, $user_id2 ) {
		if ( ! is_user_logged_in() ) {
			return true;
		}

		$roles1 = UM()->roles()->get_all_user_roles( $user_id1 );

		$role2      = UM()->roles()->get_priority_user_role( $user_id2 );
		$role_data2 = UM()->roles()->role_data( $role2 );
		/** This filter is documented in ultimate-member/includes/core/class-roles-capabilities.php */
		$role_data2 = apply_filters( 'um_user_permissions_filter', $role_data2, $user_id2 );

		if ( ! $role_data2['can_follow'] ) {
			return false;
		}

		if ( ! empty( $role_data2['can_follow_roles'] ) &&
			( empty( $roles1 ) || count( array_intersect( $roles1, maybe_unserialize( $role_data2['can_follow_roles'] ) ) ) <= 0 ) ) {
			return false;
		}

		if ( $user_id1 != $user_id2 && is_user_logged_in() ) {
			return true;
		}

		return false;
	}

	/**
	 * Get the count of followers.
	 *
	 * @param int $user_id
	 *
	 * @return int
	 */
	public function count_followers_plain( $user_id = 0 ) {
		global $wpdb;

		$count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*)
				FROM {$wpdb->prefix}um_followers
				WHERE user_id1 = %d AND
					  user_id2 IN ( SELECT ID FROM {$wpdb->users} )",
				$user_id
			)
		);

		return isset( $count ) ? (int) $count : 0;
	}

	/**
	 * Get the count of followers in nice format.
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function count_followers( $user_id = 0 ) {
		wp_enqueue_script( 'um_followers' );
		wp_enqueue_style( 'um_followers' );

		$count = $this->count_followers_plain( $user_id );
		return '<span class="um-ajax-count-followers">' . number_format( $count ) . '</span>';
	}

	/**
	 * Get the count of following.
	 *
	 * @param int $user_id
	 *
	 * @return int
	 */
	public function count_following_plain( $user_id = 0 ) {
		global $wpdb;

		$count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*)
				FROM {$wpdb->prefix}um_followers
				WHERE user_id2 = %d AND
					  user_id1 IN ( SELECT ID FROM {$wpdb->users} )",
				$user_id
			)
		);

		return isset( $count ) ? (int) $count : 0;
	}

	/**
	 * Get the count of following in nice format.
	 *
	 * @param int $user_id
	 *
	 * @return string
	 */
	public function count_following( $user_id = 0 ) {
		wp_enqueue_script( 'um_followers' );
		wp_enqueue_style( 'um_followers' );

		$count = $this->count_following_plain( $user_id );
		return '<span class="um-ajax-count-following">' . number_format( $count ) . '</span>';
	}

	/**
	 * Add a follow action.
	 *
	 * @param int $user_id1
	 * @param int $user_id2
	 * @return bool|false|int
	 */
	public function add( $user_id1, $user_id2 ) {
		global $wpdb;

		// if already followed do not add
		if ( $this->followed( $user_id1, $user_id2 ) ) {
			return false;
		}

		$result = $wpdb->insert(
			$this->table_name,
			array(
				'time'     => current_time( 'mysql' ),
				'user_id1' => $user_id1,
				'user_id2' => $user_id2,
			),
			array(
				'%s',
				'%d',
				'%d',
			)
		);

		return $result;
	}

	/**
	 * Removes a follow connection.
	 *
	 * @param int $user_id1
	 * @param int $user_id2
	 *
	 * @return bool
	 */
	public function remove( $user_id1, $user_id2 ) {
		global $wpdb;

		// If user is not followed do not do anything
		if ( ! $this->followed( $user_id1, $user_id2 ) ) {
			return false;
		}

		$wpdb->delete(
			$this->table_name,
			array(
				'user_id1' => $user_id1,
				'user_id2' => $user_id2,
			)
		);

		return true;
	}

	/**
	 * Checks if user is follower of another user.
	 *
	 * @param int $user_id1
	 * @param int $user_id2
	 *
	 * @return bool
	 */
	public function followed( $user_id1, $user_id2 ) {
		global $wpdb;

		$results = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT user_id1
				FROM {$wpdb->prefix}um_followers
				WHERE user_id1 = %d AND
					  user_id2 = %d AND
					  user_id1 IN ( SELECT ID FROM {$wpdb->users} ) AND
					  user_id2 IN ( SELECT ID FROM {$wpdb->users} )
				LIMIT 1",
				$user_id1,
				$user_id2
			)
		);

		return ! empty( $results );
	}

	/**
	 * Get followers as array.
	 *
	 * @param int $user_id1
	 * @param array $args
	 * @return array|bool
	 */
	public function followers( $user_id1, $args = array() ) {
		global $wpdb;

		if ( ! empty( $args ) && $args['max'] > 0 ) {
			$results = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT user_id2
					FROM {$wpdb->prefix}um_followers
					WHERE user_id1 = %d AND
						  user_id2 IN ( SELECT ID FROM {$wpdb->users} )
					ORDER BY time DESC
					LIMIT %d",
					$user_id1,
					$args['max']
				)
			);
		} else {
			$results = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT user_id2
					FROM {$wpdb->prefix}um_followers
					WHERE user_id1 = %d AND
						  user_id2 IN ( SELECT ID FROM {$wpdb->users} )
					ORDER BY time DESC",
					$user_id1
				)
			);
		}

		return ! empty( $results ) ? array_map( 'absint', $results ) : false;
	}

	/**
	 * Get following as array.
	 *
	 * @param int $user_id2
	 * @param array $args
	 * @return array|bool
	 */
	public function following( $user_id2, $args = array() ) {
		global $wpdb;

		if ( ! empty( $args ) && $args['max'] > 0 ) {
			$results = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT user_id1
					FROM {$wpdb->prefix}um_followers
					WHERE user_id2 = %d AND
						  user_id1 IN ( SELECT ID FROM {$wpdb->users} )
					ORDER BY time DESC
					LIMIT %d",
					$user_id2,
					$args['max']
				)
			);
		} else {
			$results = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT user_id1
					FROM {$wpdb->prefix}um_followers
					WHERE user_id2 = %d AND
						  user_id1 IN ( SELECT ID FROM {$wpdb->users} )
					ORDER BY time DESC",
					$user_id2
				)
			);
		}

		return ! empty( $results ) ? array_map( 'absint', $results ) : false;
	}

	/**
	 * AJAX handler on click Follow button.
	 */
	public function ajax_followers_follow() {
		// phpcs:disable WordPress.Security.NonceVerification
		if ( empty( $_POST['user_id'] ) ) {
			wp_send_json_error( __( 'Wrong user ID.', 'um-followers' ) );
		}
		$user_id = absint( $_POST['user_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		if ( defined( 'UM_DEV_MODE' ) && UM_DEV_MODE && UM()->options()->get( 'enable_new_ui' ) ) {
			if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'um_followers_follow' . $user_id ) ) {
				wp_send_json_error( __( 'Wrong nonce', 'um-followers' ) );
			}
		} else {
			UM()->check_ajax_nonce();
		}

		$user_id2 = get_current_user_id();

		if ( ! is_numeric( $user_id ) ) {
			wp_send_json_error( __( 'Wrong user ID.', 'um-followers' ) );
		}

		if ( ! $this->can_follow( $user_id, $user_id2 ) ) {
			wp_send_json_error( __( 'You cannot follow this user.', 'um-followers' ) );
		}

		if ( $this->followed( $user_id, $user_id2 ) ) {
			wp_send_json_error( __( 'You have already followed this user.', 'um-followers' ) );
		}

		$this->add( $user_id, $user_id2 );

		$output        = array();
		$output['btn'] = $this->follow_button( $user_id, $user_id2 ); // following user id , current user id

		$output['stats'] = array(
			array(
				'user_id'   => absint( $user_id ),
				'followers' => $this->count_followers_plain( $user_id ),
			),
			array(
				'user_id'   => absint( $user_id2 ),
				'following' => $this->count_following_plain( $user_id2 ),
			),
		);

		do_action( 'um_followers_after_user_follow', $user_id, $user_id2 );

		wp_send_json_success( $output );
	}

	/**
	 * Ajax handler on click UnFollow button.
	 */
	public function ajax_followers_unfollow() {
		UM()->check_ajax_nonce();

		// phpcs:disable WordPress.Security.NonceVerification
		if ( empty( $_POST['user_id'] ) ) {
			wp_send_json_error( __( 'Wrong user ID.', 'um-followers' ) );
		}
		$user_id = absint( $_POST['user_id'] );
		// phpcs:enable WordPress.Security.NonceVerification

		$user_id2 = get_current_user_id();

		if ( ! is_numeric( $user_id ) ) {
			wp_send_json_error( __( 'Wrong user ID.', 'um-followers' ) );
		}

		if ( ! $this->can_follow( $user_id, $user_id2 ) ) {
			wp_send_json_error( __( 'You cannot follow this user.', 'um-followers' ) );
		}

		if ( ! $this->followed( $user_id, $user_id2 ) ) {
			wp_send_json_error( __( 'You have already unfollowed this user.', 'um-followers' ) );
		}

		$this->remove( $user_id, $user_id2 );

		$output        = array();
		$output['btn'] = $this->follow_button( $user_id, $user_id2 );

		$output['stats'] = array(
			array(
				'user_id'   => absint( $user_id ),
				'followers' => $this->count_followers_plain( $user_id ),
			),
			array(
				'user_id'   => absint( $user_id2 ),
				'following' => $this->count_following_plain( $user_id2 ),
			),
		);

		do_action( 'um_followers_after_user_unfollow', $user_id, $user_id2 );

		wp_send_json_success( $output );
	}

	public function ajax_get_users() {
		UM()->check_ajax_nonce();
		// phpcs:disable WordPress.Security.NonceVerification
		if ( empty( $_POST['type'] ) || ! in_array( $_POST['type'], array( 'followers', 'following' ), true ) ) {
			wp_send_json_error( __( 'Invalid data type.', 'um-followers' ) );
		}
		$type = sanitize_key( $_POST['type'] );

		if ( empty( $_POST['user_id'] ) || ! is_numeric( $_POST['user_id'] ) ) {
			wp_send_json_error( __( 'Invalid user ID.', 'um-followers' ) );
		}
		$user_id = absint( $_POST['user_id'] );

		if ( ! um_can_view_profile( $user_id ) ) {
			wp_send_json_error( __( 'You aren\'t able to see this information', 'um-followers' ) );
		}

		$args = array();
		if ( ! empty( $_POST['max'] ) && is_numeric( $_POST['max'] ) ) {
			$args['max'] = absint( $_POST['max'] );
		}

		$offset = 0;
		if ( ! empty( $_POST['offset'] ) && is_numeric( $_POST['offset'] ) ) {
			$offset = absint( $_POST['offset'] );
		}
		// phpcs:enable WordPress.Security.NonceVerification

		global $wpdb;

		if ( 'followers' === $type ) {
			$all_count = $this->count_followers_plain( $user_id );
			if ( $offset > 0 ) {
				$limit = $all_count - $offset;

				$users = $wpdb->get_col(
					$wpdb->prepare(
						"SELECT user_id2
						FROM {$wpdb->prefix}um_followers
						WHERE user_id1 = %d AND
							  user_id2 IN ( SELECT ID FROM {$wpdb->users} )
						ORDER BY time DESC
						LIMIT %d, %d",
						$user_id,
						$offset,
						$limit
					)
				);
			} else {
				$users = $this->followers( $user_id, $args );
			}
		} else {
			$all_count = $this->count_following_plain( $user_id );
			if ( $offset > 0 ) {
				$limit = $all_count - $offset;

				$users = $wpdb->get_col(
					$wpdb->prepare(
						"SELECT user_id1
						FROM {$wpdb->prefix}um_followers
						WHERE user_id2 = %d AND
							  user_id1 IN ( SELECT ID FROM {$wpdb->users} )
						ORDER BY time DESC
						LIMIT %d, %d",
						$user_id,
						$args['max'],
						$limit
					)
				);
			} else {
				$users = $this->following( $user_id, $args );
			}
		}

		$response = array( 'users' => array() );
		if ( $users ) {
			$users_data = array();
			foreach ( $users as $user_id ) {
				um_fetch_user( $user_id );
				$users_data[] = array(
					'url'    => um_user_profile_url(),
					'avatar' => get_avatar( $user_id, 40 ),
					'name'   => esc_attr( um_user( 'display_name' ) ),
				);
			}

			$response['users'] = $users_data;

			if ( 0 === $offset ) {
				$response['all'] = $all_count - count( $users );
			}
		}

		wp_send_json_success( $response );
	}
}
