<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WC_Email_Customer_New_Order extends WC_Email {

	/**
	 * Strings to find in subjects/headings.
	 *
	 * @var array
	 */
	public $find;

	/**
	 * Strings to replace in subjects/headings.
	 *
	 * @var array
	 */
	public $replace;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id             = 'customer_new_order';
		$this->title          = __( 'New order (Customer)', 'woocommerce' );
		$this->description    = __( 'New order emails can be sent to customers containing their order information and a notification that their enrollment is not yet complete.', 'woocommerce' );

		$this->template_base = untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/../templates/';
		$this->template_html  = 'emails/customer-new-order.php';
		$this->template_plain = 'emails/plain/customer-new-order.php';

		$this->subject        = __( 'Your new order', 'woocommerce' );
		$this->heading        = __( 'Your new order', 'woocommerce' );


		// Call parent constructor
		parent::__construct();

		add_action('woocommerce_checkout_order_processed_notification', array( $this, 'trigger' ));

		$this->customer_email = true;
	}

	/**
	 * Trigger the sending of this email.
	 *
	 * @param int $order_id The order ID.
	 * @param WC_Order $order Order object.
	 */
	public function trigger($order_id) {
		$order = wc_get_order( $order_id );

		if ( is_a( $order, 'WC_Order' ) ) {
			$this->object                  = $order;
			$this->recipient               = $this->object->get_billing_email();

			$this->find['order-date']      = '{order_date}';
			$this->find['order-number']    = '{order_number}';

			$this->replace['order-date']   = wc_format_datetime( $this->object->get_date_created() );
			$this->replace['order-number'] = $this->object->get_order_number();
		}
		if ( ! $this->get_recipient() ) {
			return;
		}

        $recipient = $this->get_recipient();
        $participant = get_post_meta($order->get_id(), 'billing_participant_is', true );

        if ($participant == 'au-pair' ) {
            $family_email = get_post_meta($order->get_id(), 'specific_email_au_pair_family', true);
            if(is_email($family_email)) {
                $recipient .= ',' . $family_email;
            }
        } elseif ($participant == 'children') {
            $family_email = get_post_meta($order->get_id(), 'specific_email_family', true);
            if (is_email($family_email)) {
                $recipient .= ',' . $family_email;
            }
        }

		$this->send( $recipient, $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}

	/**
	 * Get content html.
	 *
	 * @access public
	 * @return string
	 */
    public function get_content_html() {
        return wc_get_template_html( $this->template_html, array(
            'order'         => $this->object,
            'email_heading' => $this->get_heading(),
            'sent_to_admin' => false,
            'plain_text'    => false,
            'email'			=> $this,
        ), '', $this->template_base );
    }

    /**
     * Get content plain.
     *
     * @access public
     * @return string
     */
    public function get_content_plain() {
        return wc_get_template_html( $this->template_plain, array(
            'order'         => $this->object,
            'email_heading' => $this->get_heading(),
            'sent_to_admin' => false,
            'plain_text'    => true,
            'email'			=> $this,
        ), '', $this->template_base );
    }

	/**
	 * Initialise settings form fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'         => array(
				'title'       => __( 'Enable/Disable', 'woocommerce' ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable this email notification', 'woocommerce' ),
				'default'     => 'yes',
			),
			'subject' => array(
				'title'         => __( 'Email subject', 'woocommerce' ),
				'type'          => 'text',
				/* translators: %s: default subject */
				'description'   => sprintf( __( 'Defaults to %s', 'woocommerce' ), '<code>' . $this->subject . '</code>' ),
				'placeholder'   => '',
				'default'       => '',
				'desc_tip'      => true,
			),
			'heading' => array(
				'title'         => __( 'Email heading', 'woocommerce' ),
				'type'          => 'text',
				/* translators: %s: default heading */
				'description'   => sprintf( __( 'Defaults to %s', 'woocommerce' ), '<code>' . $this->heading . '</code>' ),
				'placeholder'   => '',
				'default'       => '',
				'desc_tip'      => true,
			),
			'email_type' => array(
				'title'         => __( 'Email type', 'woocommerce' ),
				'type'          => 'select',
				'description'   => __( 'Choose which format of email to send.', 'woocommerce' ),
				'default'       => 'html',
				'class'         => 'email_type wc-enhanced-select',
				'options'       => $this->get_email_type_options(),
				'desc_tip'      => true,
			),
		);
	}
}
