<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class WC_Email_Customer_Course_Changed extends WC_Email
{

    /**
     * Strings to find in subjects/headings.
     *
     * @var array
     */
    public $find;

    /**
     * Strings to replace in subjects/headings.
     *
     * @var array
     */
    public $replace;

    public $diff;

    public $labels = [
        'start' => 'Start Date',
        'end' => 'End Date'
    ];

    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->id = 'customer_course_changed';
        $this->title = __('Course Changed', 'woocommerce');
        $this->description = __('Notify the customer when a course in one of their orders changed.', 'woocommerce');

        $this->template_base = untrailingslashit(plugin_dir_path(__FILE__)) . '/../templates/';
        $this->template_html = 'emails/customer-course-changed.php';
        $this->template_plain = 'emails/plain/customer-course-changed.php';

        $this->subject = __('Course Updated', 'woocommerce');
        $this->heading = __('Course Updated', 'woocommerce');

        // Call parent constructor
        parent::__construct();

        add_action('woocommerce_update_product', array($this, 'trigger'), 10, 2);
        add_action('wp_insert_post_data', array($this, 'store_product_before'), 10, 2);

        $this->customer_email = true;
    }


    public function store_product_before($clean_data, $santized_data)
    {
        $post_id = $santized_data['ID'];

        if ($clean_data['post_type'] === 'product') {
            $product = wc_get_product($post_id);

            if ($product->get_attribute('pa_kind') !== 'Course') {
                return;
            }

            /** @var array $meta_to_monitor */
            $meta_to_monitor = $this->get_option('notify_on_attributes');

            if (empty($meta_to_monitor)) {
                $this->diff = [];
                return;
            }

            // fetch existing meta
            $meta_before = array_reduce($meta_to_monitor, function ($carry, $meta_key) use ($post_id) {
                $meta = get_post_meta($post_id, $meta_key, false);
                if (!empty($meta)) {
                    $label = $this->labels[$meta_key] ?? $meta_key;
                    $carry[$label] = array_pop($meta);
                }
                return $carry;
            }, []);

            // fetch product attributes before
            $attributes_before = [];
            foreach ($product->get_attributes() as $attribute) {
                /** @var WC_Product_Attribute $attribute */
                if (in_array($attribute->get_name(), $meta_to_monitor)) {
                    $label = wc_attribute_label($attribute->get_name());
                    $attributes_before[$label] = $attribute->get_options();
                }
            }

            // get meta after
            array_walk($santized_data, function ($value, $key) use ($meta_to_monitor, &$meta_after) {
                if (in_array($key, $meta_to_monitor)) {
                    $label = $this->labels[$key] ?? $key;
                    $meta_after[$label] = $value;
                }
            }, ARRAY_FILTER_USE_KEY);


            // get product attributes after
            $attribute_data = array_combine($santized_data['attribute_names'], $santized_data['attribute_values']);
            array_walk($attribute_data, function ($value, $attribute_name) use ($meta_to_monitor, &$attributes_after) {
                if (in_array($attribute_name, $meta_to_monitor)) {
                    $label = wc_attribute_label($attribute_name);
                    $attributes_after[$label] = array_map('intval', $value);
                }
            });

            // calculate diffs nested
            $this->diff = \Rogervila\ArrayDiffMultidimensional::compare(
                [
                    'meta' => $meta_after,
                    'attributes' => $attributes_after
                ],
                [
                    'meta' => $meta_before,
                    'attributes' => $attributes_before
                ]
            );
            $this->diff = array_reduce($this->diff, 'array_merge', []);
        }

        return $clean_data;
    }

    /**
     * Trigger the sending of this email.
     */
    public function trigger($product_id, WC_Product $product)
    {
        if ($product instanceof WC_Product && !empty($this->diff)) {
            if ($product->get_attribute('pa_kind') !== 'Course') {
                return;
            }
            global $wpdb;

            $customers = $wpdb->get_results("
                SELECT os.order_id, cl.first_name, cl.email
                FROM {$wpdb->prefix}woocommerce_order_itemmeta as woim
                JOIN {$wpdb->prefix}woocommerce_order_items as woi
                    ON      woi.order_item_id = woim.order_item_id
                JOIN {$wpdb->prefix}posts as p
                    ON      woi.order_id = p.ID
                JOIN {$wpdb->prefix}wc_order_stats as os
                    ON      woi.order_id = os.order_id                        
                JOIN {$wpdb->prefix}wc_customer_lookup as cl
                    ON os.customer_id = cl.customer_id
                WHERE   woim.meta_key = '_product_id'
                    AND     woim.meta_value = ({$product_id})
                    AND     p.post_status NOT IN ( 'wc-cancelled', 'wc-failed', 'wc-quit-course' )
                ",
                ARRAY_A
            );

            add_action('wp_mail', function ($email) {
                echo $email['message'];
                die();
            }, 10);

            add_filter('woocommerce_email_subject_' . $this->id, function () use ($product) {
                $product_name = $product->get_name();
                return "Taalthuis - Course \"$product_name\" Updated";
            });


            foreach ($customers as $customer) {
                // setup email variables
                $order = wc_get_order($customer['order_id']);

                $this->object = [
                    'order' => $order,
                    'customer' => [
                        'first_name' => $customer['first_name'],
                    ],
                    'product' => $product,
                    'changes' => $this->diff
                ];

                // set the recipient
                $this->recipient = $order->get_billing_email();
                if (!$this->get_recipient()) {
                    return;
                }

                // send the email
                $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
            }
        }
    }

    /**
     * Get content html.
     *
     * @access public
     * @return string
     */
    public function get_content_html()
    {
        return wc_get_template_html($this->template_html, array(
            'objects' => $this->object,
            'email_heading' => $this->get_heading(),
            'sent_to_admin' => false,
            'plain_text' => false,
            'email' => $this,
        ), '', $this->template_base);
    }

    /**
     * Get content plain.
     *
     * @access public
     * @return string
     */
    public function get_content_plain()
    {
        return wc_get_template_html($this->template_plain, array(
            'order' => $this->object,
            'email_heading' => $this->get_heading(),
            'sent_to_admin' => false,
            'plain_text' => true,
            'email' => $this,
        ), '', $this->template_base);
    }

    /**
     * Initialise settings form fields.
     */
    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'woocommerce'),
                'type' => 'checkbox',
                'label' => __('Enable this email notification', 'woocommerce'),
                'default' => 'yes',
            ),
            'subject' => array(
                'title' => __('Email subject', 'woocommerce'),
                'type' => 'text',
                /* translators: %s: default subject */
                'description' => sprintf(__('Defaults to %s', 'woocommerce'), '<code>' . $this->subject . '</code>'),
                'placeholder' => '',
                'default' => '',
                'desc_tip' => true,
            ),
            'heading' => array(
                'title' => __('Email heading', 'woocommerce'),
                'type' => 'text',
                /* translators: %s: default heading */
                'description' => sprintf(__('Defaults to %s', 'woocommerce'), '<code>' . $this->heading . '</code>'),
                'placeholder' => '',
                'default' => '',
                'desc_tip' => true,
            ),
            'email_type' => array(
                'title' => __('Email type', 'woocommerce'),
                'type' => 'select',
                'description' => __('Choose which format of email to send.', 'woocommerce'),
                'default' => 'html',
                'class' => 'email_type wc-enhanced-select',
                'options' => $this->get_email_type_options(),
                'desc_tip' => true,
            ),
            'notify_on_attributes' => array(
                'title' => __('Notify for attributes', 'woocommerce'),
                'type' => 'multiselect',
                'description' => __('Select for which attributes the notification should be sent.', 'woocommerce'),
                'class' => 'select2',
                'options' => $this->labels,
                'desc_tip' => true,
            ),
        );

        if (function_exists('get_current_screen')
            && get_current_screen()->base === 'woocommerce_page_wc-settings'
            && isset($_GET['section'])
            && $_GET['section'] === 'wc_email_' . $this->id) {
            add_action('admin_enqueue_scripts', function () {
                $suffix = \Automattic\Jetpack\Constants::is_true('SCRIPT_DEBUG') ? '' : '.min';
                $path = 'assets/js/select2/select2.full' . $suffix . '.js';
                $url = apply_filters('woocommerce_get_asset_url', plugins_url($path, WC_PLUGIN_FILE), $path);
                wp_enqueue_script('select2', $url, ['jquery'], '4.0.3');

                wc_enqueue_js("jQuery('.select2').select2();");
            });
        }
    }
}
