<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class WooCommerceUsersVisibility
 */
class WooCommerceUsersVisibility extends Singleton {

	const OPTION_NAME = 'woocommerce-users';

	/**
	 * WooCommerceUsersVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'woocommerce_users_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'woocommerce_users_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'WooCommerce Users', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$meets_plugin_requirements  = $this->meets_plugin_requirements( self::OPTION_NAME );
		$meets_version_requirements = $this->meets_version_requirements( self::OPTION_NAME );
		$current_widget             = Plugin::instance()->get_pro_option( self::OPTION_NAME );

		if ( ! $meets_version_requirements ) {
			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['version'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-woocommerce_users_enabled-',
					'style_transfer' => false,
				]
			);
		} elseif ( ! $meets_plugin_requirements ) {
			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['plugins'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-woocommerce_users_enabled-',
					'style_transfer' => false,
				]
			);
		} else {
			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_enabled',
				[
					'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SWITCHER,
					'default'        => '',
					'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
					'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
					'return_value'   => 'yes',
					'prefix_class'   => 'stax-woocommerce_users_enabled-',
					'style_transfer' => false,
				]
			);

			$options = [
				'any'               => __( 'User has purchased any product', 'visibility-logic-elementor-pro' ),
				'product'           => __( 'User has purchased specific product', 'visibility-logic-elementor-pro' ),
				'product_variation' => __( 'User has purchased specific product variation', 'visibility-logic-elementor-pro' ),
			];

			if ( $this->has_woo_subscriptions_plugin() ) {
				$options['any_subscription']      = __( 'User has any active subscription', 'visibility-logic-elementor-pro' );
				$options['specific_subscription'] = __( 'User has specific active subscription', 'visibility-logic-elementor-pro' );
			}

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_selection',
				[
					'label'          => __( 'Condition', 'visibility-logic-elementor-pro' ),
					'label_block'    => true,
					'type'           => Controls_Manager::SELECT,
					'options'        => $options,
					'default'        => 'any',
					'condition'      => [
						self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			if ( $this->has_woo_subscriptions_plugin() ) {
				$element->add_control(
					self::SECTION_PREFIX . 'woocommerce_users_subscription_condition',
					[
						'label'          => __( 'Products Condition', 'visibility-logic-elementor-pro' ),
						'label_block'    => true,
						'type'           => Controls_Manager::SELECT,
						'options'        => [
							'one' => __( 'Has at least one', 'visibility-logic-elementor-pro' ),
							'all' => __( 'Has all', 'visibility-logic-elementor-pro' ),
						],
						'default'        => 'one',
						'condition'      => [
							self::SECTION_PREFIX . 'woocommerce_users_selection' => 'specific_subscription',
							self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
						],
						'style_transfer' => false,
					]
				);

				$element->add_control(
					self::SECTION_PREFIX . 'woocommerce_users_subscription',
					[
						'label'          => __( 'Subscription Products', 'visibility-logic-elementor-pro' ),
						'type'           => 'stax_query',
						'query_type'     => 'subscriptions',
						'label_block'    => true,
						'multiple'       => true,
						'condition'      => [
							self::SECTION_PREFIX . 'woocommerce_users_selection' => 'specific_subscription',
							self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
						],
						'style_transfer' => false,
					]
				);
			}

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_product_condition',
				[
					'label'          => __( 'Products Condition', 'visibility-logic-elementor-pro' ),
					'label_block'    => true,
					'type'           => Controls_Manager::SELECT,
					'options'        => [
						'one' => __( 'At least one', 'visibility-logic-elementor-pro' ),
						'all' => __( 'All', 'visibility-logic-elementor-pro' ),
					],
					'default'        => 'one',
					'condition'      => [
						self::SECTION_PREFIX . 'woocommerce_users_selection' => [ 'product', 'product_variation' ],
						self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_product',
				[
					'label'          => __( 'Product', 'visibility-logic-elementor-pro' ),
					'type'           => 'stax_query',
					'query_type'     => 'products',
					'label_block'    => true,
					'multiple'       => true,
					'condition'      => [
						self::SECTION_PREFIX . 'woocommerce_users_selection' => 'product',
						self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'woocommerce_users_product_variation',
				[
					'label'          => __( 'Product Variation', 'visibility-logic-elementor-pro' ),
					'type'           => 'stax_query',
					'query_type'     => 'products_variations',
					'label_block'    => true,
					'multiple'       => true,
					'condition'      => [
						self::SECTION_PREFIX . 'woocommerce_users_selection' => 'product_variation',
						self::SECTION_PREFIX . 'woocommerce_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			if ( ! $this->has_woo_subscriptions_plugin() ) {
				$element->add_control(
					self::SECTION_PREFIX . 'woocommerce_users_subscriptions_requirement',
					[
						'type'            => Controls_Manager::RAW_HTML,
						'raw'             => sprintf(
							esc_html__( 'Want to check for a user\'s subscription? %1$s has to be installed and activated!', 'visibility-logic-elementor-pro' ),
							'<a href="https://staxwp.com/go/woocommerce-subscriptions" target="_blank">WooCommerce Subscriptions</a>',
						),
						'content_classes' => 'stx-plugin-requirement',
						'style_transfer'  => false,
					]
				);
			}
		}
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'woocommerce_users_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'woocommerce_users_enabled' ] ||
			! $this->meets_plugin_requirements( self::OPTION_NAME ) ||
			! $this->meets_version_requirements( self::OPTION_NAME ) ) {
			return $options;
		}

		switch ( $settings[ self::SECTION_PREFIX . 'woocommerce_users_selection' ] ) {
			case 'any':
				$options['woocommerce_users'] = $this->user_has_bought_anything();
				break;
			case 'product':
				$options['woocommerce_users'] = $this->user_has_bought_product( $settings[ self::SECTION_PREFIX . 'woocommerce_users_product' ], $settings[ self::SECTION_PREFIX . 'woocommerce_users_product_condition' ] );
				break;
			case 'product_variation':
				$options['woocommerce_users'] = $this->user_has_bought_product( $settings[ self::SECTION_PREFIX . 'woocommerce_users_product_variation' ], $settings[ self::SECTION_PREFIX . 'woocommerce_users_product_condition' ] );
				break;
			case 'any_subscription':
				if ( $this->has_woo_subscriptions_plugin() ) {
					$options['woocommerce_users'] = $this->user_has_any_active_subscription();
				}
				break;
			case 'specific_subscription':
				if ( $this->has_woo_subscriptions_plugin() ) {
					$options['woocommerce_users'] = $this->user_has_active_subscription( $settings[ self::SECTION_PREFIX . 'woocommerce_users_subscription' ], $settings[ self::SECTION_PREFIX . 'woocommerce_users_subscription_condition' ] );
				}
				break;
			default:
				$options['woocommerce_users'] = false;
		}

		return $options;
	}

	/**
	 * Check if user has bought anything
	 *
	 * @return void
	 */
	public function user_has_bought_anything() {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		global $wpdb;

		$paid_order_statuses = array_map( 'esc_sql', wc_get_is_paid_statuses() );

		$count = $wpdb->get_var(
			$wpdb->prepare(
				"
            SELECT COUNT(p.ID) FROM {$wpdb->prefix}posts AS p
            INNER JOIN {$wpdb->prefix}postmeta AS pm ON p.ID = pm.post_id
            WHERE p.post_status IN ( 'wc-" . implode( "','wc-", $paid_order_statuses ) . "' )
            AND p.post_type LIKE 'shop_order'
            AND pm.meta_key = '%s'
            AND pm.meta_value = %s
            LIMIT 1
        ",
				'_customer_user',
				(int) get_current_user_id()
			)
		);

		return $count > 0;
	}

	/**
	 * Check if user has bought product
	 *
	 * @param array  $product_ids
	 * @param string $condition
	 * @return boolean
	 */
	public function user_has_bought_product( $product_ids = [], $condition = 'one' ) {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		if ( empty( $product_ids ) ) {
			return false;
		}

		if ( ! in_array( $condition, [ 'one', 'all' ] ) ) {
			return false;
		}

		$has_bought_product = 'all' === $condition;
		$current_user       = wp_get_current_user();
		$user_email         = $current_user->user_email;

		foreach ( $product_ids as $key => $product_id ) {
			switch ( $condition ) {
				case 'one':
					if ( wc_customer_bought_product( $user_email, $current_user->ID, $product_id ) ) {
						$has_bought_product = true;
					}
					break;
				case 'all':
					if ( ! wc_customer_bought_product( $user_email, $current_user->ID, $product_id ) ) {
						$has_bought_product = false;
					}
					break;
				default:
			}
		}

		return $has_bought_product;
	}

	/**
	 * Check if user has any subscription
	 *
	 * @return boolean
	 */
	public function user_has_any_active_subscription() {
		if ( ! is_user_logged_in() || ! $this->has_woo_subscriptions_plugin() ) {
			return false;
		}

		return wcs_user_has_subscription( get_current_user_id(), '', 'active' );
	}

	/**
	 * Check if user has specific subscription
	 *
	 * @param array  $product_ids
	 * @param string $condition
	 * @return boolean
	 */
	public function user_has_active_subscription( $product_ids = [], $condition = 'one' ) {
		if ( ! is_user_logged_in() || ! $this->has_woo_subscriptions_plugin() ) {
			return false;
		}

		if ( empty( $product_ids ) ) {
			return false;
		}

		if ( ! in_array( $condition, [ 'one', 'all' ] ) ) {
			return false;
		}

		$has_sub = 'all' === $condition;
		$user_id = get_current_user_id();

		foreach ( $product_ids as $key => $product_id ) {
			switch ( $condition ) {
				case 'one':
					if ( wcs_user_has_subscription( $user_id, (int) $product_id, 'active' ) ) {
						$has_sub = true;
					}
					break;
				case 'all':
					if ( ! wcs_user_has_subscription( $user_id, (int) $product_id, 'active' ) ) {
						$has_sub = false;
					}
					break;
				default:
			}
		}

		return $has_sub;
	}

	/**
	 * Check if Woo Subscriptions plugin is active
	 *
	 * @return boolean
	 */
	public function has_woo_subscriptions_plugin() {
		return \Stax\VisibilityLogic\Resources::instance()->is_plugin_active( 'woocommerce-subscriptions/woocommerce-subscriptions.php' );
	}
}

WooCommerceUsersVisibility::instance();
