<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class PostVisibility
 */
class PostVisibility extends Singleton {

	const OPTION_NAME = 'post-page';

	/**
	 * PostVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'post_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'post_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'Post/Page', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$element->add_control(
			self::SECTION_PREFIX . 'post_enabled',
			[
				'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'default'        => '',
				'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
				'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
				'return_value'   => 'yes',
				'prefix_class'   => 'stax-post_enabled-',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'custom_post_type',
			[
				'label'          => __( 'Custom Post Type', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SELECT2,
				'options'        => Helpers::get_post_types(),
				'multiple'       => true,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'specific_post',
			[
				'type'           => 'stax_query',
				'label'          => __( 'Post/Page', 'visibility-logic-elementor-pro' ),
				'query_type'     => 'posts',
				'label_block'    => true,
				'multiple'       => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$taxonomies = Helpers::get_taxonomies();

		$element->add_control(
			self::SECTION_PREFIX . 'taxonomy',
			[
				'label'          => __( 'Taxonomy', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SELECT2,
				'options'        => $taxonomies,
				'multiple'       => false,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		foreach ( $taxonomies as $taxonomy_key => $taxonomy ) {
			if ( $taxonomy_key ) {
				$element->add_control(
					self::SECTION_PREFIX . 'taxonomy_term_' . $taxonomy_key,
					[
						'label'          => __( 'Terms', 'visibility-logic-elementor-pro' ),
						'type'           => Controls_Manager::SELECT2,
						'options'        => Helpers::get_taxonomy_terms( $taxonomy_key ),
						'multiple'       => true,
						'label_block'    => true,
						'condition'      => [
							self::SECTION_PREFIX . 'post_enabled' => 'yes',
							self::SECTION_PREFIX . 'taxonomy' => $taxonomy_key,
						],
						'style_transfer' => false,
					]
				);
			}
		}

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta',
			[
				'label'          => __( 'Meta', 'visibility-logic-elementor-pro' ),
				'type'           => 'stax_query',
				'query_type'     => 'fields',
				'object_type'    => 'post',
				'description'    => __( 'If you want to limit visualization by meta field value.', 'visibility-logic-elementor-pro' ),
				'multiple'       => true,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_operator',
			[
				'label'          => __( 'Meta Condition Type', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::HIDDEN,
				'default'        => 'yes',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_status',
			[
				'type'           => Controls_Manager::SELECT2,
				'label'          => __( 'Meta Condition', 'visibility-logic-elementor-pro' ),
				'description'    => __( 'Select the condition that must be met by the meta', 'visibility-logic-elementor-pro' ),
				'options'        => [
					'any'                     => __( 'Any Value', 'visibility-logic-elementor-pro' ),
					'empty'                   => __( 'Empty', 'visibility-logic-elementor-pro' ),
					'not_empty'               => __( 'Not empty', 'visibility-logic-elementor-pro' ),
					'specific_value'          => __( 'Is equal to', 'visibility-logic-elementor-pro' ),
					'specific_value_multiple' => __( 'Is equal to one of', 'visibility-logic-elementor-pro' ),
					'not_specific_value'      => __( 'Not equal to', 'visibility-logic-elementor-pro' ),
					'contain'                 => __( 'Contains', 'visibility-logic-elementor-pro' ),
					'not_contain'             => __( 'Does not contain', 'visibility-logic-elementor-pro' ),
					'is_between'              => __( 'Between', 'visibility-logic-elementor-pro' ),
					'less_than'               => __( 'Less than', 'visibility-logic-elementor-pro' ),
					'greater_than'            => __( 'Greater than', 'visibility-logic-elementor-pro' ),
					'is_array'                => __( 'Is array', 'visibility-logic-elementor-pro' ),
					'is_array_and_contains'   => __( 'Is array and contains', 'visibility-logic-elementor-pro' ),
				],
				'default'        => 'any',
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_value',
			[
				'label'          => __( 'Meta Condition Value', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXTAREA,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
					self::SECTION_PREFIX . 'post_meta_status' => [
						'specific_value',
						'specific_value_multiple',
						'not_specific_value',
						'contain',
						'not_contain',
						'is_between',
						'less_than',
						'greater_than',
						'is_array_and_contains',
					],
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_value_2',
			[
				'label'          => __( 'Meta Condition Value 2', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXTAREA,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
					self::SECTION_PREFIX . 'post_meta_status' => 'is_between',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_notice_array',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'Type in comma separated strings.', 'visibility-logic-elementor-pro' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
					self::SECTION_PREFIX . 'post_meta_status' => [
						'specific_value_multiple',
						'is_array_and_contains',
					],
				],
				'style_transfer'  => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_notice_numeric',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'The value of the selected meta and also the the values of the conditions must be numeric.', 'visibility-logic-elementor-pro' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
					self::SECTION_PREFIX . 'post_meta_status' => [
						'is_between',
						'less_than',
						'greater_than',
					],
				],
				'style_transfer'  => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_meta_notice_any',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'Setting the condition to "Any Value" will check if the meta has any value.', 'visibility-logic-elementor-pro' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_meta!'   => '',
					self::SECTION_PREFIX . 'post_meta_status' => [
						'any',
					],
				],
				'style_transfer'  => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_parent',
			[
				'label'          => __( 'Is Parent', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post with children.', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_root',
			[
				'label'          => __( 'Is Root', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post of first level (without parent).', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_leaf',
			[
				'label'          => __( 'Is Leaf', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post of last level (without children).', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_node',
			[
				'label'          => __( 'Is Node', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post of intermedial level (with parent and child).', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_node_level',
			[
				'label'          => __( 'Node level', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::NUMBER,
				'min'            => 1,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_node!'   => '',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_sibling',
			[
				'label'          => __( 'Has Siblings', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post with siblings.', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_friend',
			[
				'label'          => __( 'Has Term Buddies', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'description'    => __( 'Triggered for post grouped in taxonomies with other posts.', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_friend_terms',
			[
				'label'          => __( 'Terms where find Buddies', 'visibility-logic-elementor-pro' ),
				'type'           => 'stax_query',
				'query_type'     => 'terms',
				'description'    => __( 'Specific a Term for current post has friends.', 'visibility-logic-elementor-pro' ),
				'multiple'       => true,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_friend!' => '',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_with_param',
			[
				'label'          => __( 'With Parameter', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXT,
				'description'    => __( 'Write here the name of the parameter passed in GET or POST method', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_param_status',
			[
				'label'          => __( 'Parameter Status', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::CHOOSE,
				'options'        => [
					'not'   => [
						'title' => __( 'Not isset', 'visibility-logic-elementor-pro' ),
						'icon'  => 'fa fa-circle-o',
					],
					'isset' => [
						'title' => __( 'Isset', 'visibility-logic-elementor-pro' ),
						'icon'  => 'fa fa-dot-circle-o',
					],
					'value' => [
						'title' => __( 'Definited value', 'visibility-logic-elementor-pro' ),
						'icon'  => 'fa fa-circle',
					],
				],
				'default'        => 'isset',
				'toggle'         => false,
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_with_param!' => '',
				],
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'post_param_value',
			[
				'label'          => __( 'Parameter Value', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXT,
				'description'    => __( 'The specific value of the parameter', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'post_enabled' => 'yes',
					self::SECTION_PREFIX . 'post_with_param!' => '',
					self::SECTION_PREFIX . 'post_param_status' => 'value',
				],
				'style_transfer' => false,
			]
		);
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'post_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'post_enabled' ] ) {
			return $options;
		}

		// Post Type.
		if ( ! empty( $settings[ self::SECTION_PREFIX . 'custom_post_type' ] ) ) {
			$options['post_custom_type'] = false;
			$post_type                   = get_post_type();

			if ( in_array( $post_type, $settings[ self::SECTION_PREFIX . 'custom_post_type' ] ) ) {
				$options['post_custom_type'] = true;
			}
		}

		// Specific Posts.
		if ( ! empty( $settings[ self::SECTION_PREFIX . 'specific_post' ] ) ) {
			$options['post_specific_id'] = false;
			$post_id                     = 0;
			$current_query_object        = get_queried_object();

			if ( $current_query_object instanceof \WP_Post ) {
				$post_id = $current_query_object->ID;
			}

			if ( in_array( $post_id, $settings[ self::SECTION_PREFIX . 'specific_post' ] ) ) {
				$options['post_specific_id'] = true;
			}
		}

		// Taxonomy.
		if ( $settings[ self::SECTION_PREFIX . 'taxonomy' ] ) {
			$options['post_taxonomy'] = false;
			$post_taxonomies          = get_post_taxonomies();

			if ( ! in_array( $settings[ self::SECTION_PREFIX . 'taxonomy' ], $post_taxonomies ) ) {
				$options['post_taxonomy'] = false;
			} else {
				$tax_term_key = self::SECTION_PREFIX . 'taxonomy_term_' . $settings[ self::SECTION_PREFIX . 'taxonomy' ];
				$terms        = get_the_terms( get_the_ID(), $settings[ self::SECTION_PREFIX . 'taxonomy' ] );
				$terms        = wp_list_pluck( $terms, 'term_id' );

				if ( ! empty( $settings[ $tax_term_key ] ) && is_array( $settings[ $tax_term_key ] ) ) {
					if ( array_intersect( $terms, $settings[ $tax_term_key ] ) ) {
						$options['post_taxonomy'] = true;
					}
				} else {
					if ( ! empty( $terms ) ) {
						$options['post_taxonomy'] = false;
					}
				}
			}
		}

		// Meta.
		if ( ! empty( $settings[ self::SECTION_PREFIX . 'post_meta' ] ) ) {
			$options['post_meta'] = false;
			$selected_metas       = $settings[ self::SECTION_PREFIX . 'post_meta' ];

			if ( is_array( $selected_metas ) ) {
				$meta_check_type  = $settings[ self::SECTION_PREFIX . 'post_meta_status' ];
				$meta_check_value = $settings[ self::SECTION_PREFIX . 'post_meta_value' ];

				$meta_is_consistent = true;

				foreach ( $selected_metas as $key => $meta ) {
					$meta_value = get_post_meta( get_the_ID(), $meta, true );

					switch ( $meta_check_type ) {
						case 'not_empty':
						case 'any':
							if ( empty( $meta_value ) ) {
								$meta_is_consistent = false;
							}
							break;
						case 'empty':
							if ( ! empty( $meta_value ) ) {
								$meta_is_consistent = false;
							}
							break;
						case 'specific_value':
							if ( $meta_value !== $meta_check_value ) {
								$meta_is_consistent = false;
							}
							break;
						case 'specific_value_multiple':
							$values = explode( ',', $meta_check_value );

							$value_found = false;

							foreach ( $values as $val ) {
								if ( $val === $meta_value ) {
									$value_found = true;
								}
							}

							$meta_is_consistent = $value_found;
							break;
						case 'not_specific_value':
							if ( $meta_value === $meta_check_value ) {
								$meta_is_consistent = false;
							}
							break;
						case 'contain':
							if ( strpos( $meta_value, $meta_check_value ) === false ) {
								$meta_is_consistent = false;
							}
							break;
						case 'not_contain':
							if ( strpos( $meta_value, $meta_check_value ) !== false ) {
								$meta_is_consistent = false;
							}
							break;
						case 'is_between':
							$meta_check_value_2 = $settings[ self::SECTION_PREFIX . 'post_meta_value_2' ];

							if ( ! is_numeric( $meta_value ) || ! is_numeric( $meta_check_value ) || ! is_numeric( $meta_check_value_2 ) ) {
								$meta_is_consistent = false;
							}

							if ( (int) $meta_check_value > (int) $meta_value || (int) $meta_value > (int) $meta_check_value_2 ) {
								$meta_is_consistent = false;
							}
							break;
						case 'less_than':
							if ( ! is_numeric( $meta_value ) || ! is_numeric( $meta_check_value ) ) {
								$meta_is_consistent = false;
							}

							if ( (int) $meta_value > (int) $meta_check_value ) {
								$meta_is_consistent = false;
							}
							break;
						case 'greater_than':
							if ( ! is_numeric( $meta_value ) || ! is_numeric( $meta_check_value ) ) {
								$meta_is_consistent = false;
							}

							if ( $meta_value < $meta_check_value ) {
								$meta_is_consistent = false;
							}
							break;
						case 'is_array':
							if ( ! is_array( $meta_value ) ) {
								$meta_is_consistent = false;
							}
							break;
						case 'is_array_and_contains':
							if ( ! is_array( $meta_value ) ) {
								$meta_is_consistent = false;
							}

							$values = explode( ',', $meta_check_value );

							if ( empty( array_intersect( $meta_value, $values ) ) ) {
								$meta_is_consistent = false;
							}
							break;
						default:
					}
				}

				$options['post_meta'] = $meta_is_consistent;
			}
		}

		// Parent Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_parent' ] ) {
			$options['post_is_parent'] = false;

			$children = get_children(
				[
					'post_parent' => get_the_ID(),
					'post_type'   => get_post_type(),
					'numberposts' => -1,
					'post_status' => 'publish',
				]
			);

			$options['post_is_parent'] = ! empty( $children );
		}

		// Root Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_root' ] ) {
			$options['post_is_root'] = false;

			if ( ! wp_get_post_parent_id( get_the_ID() ) ) {
				$options['post_is_root'] = true;
			}
		}

		// Leaf Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_leaf' ] ) {
			$options['post_is_leaf'] = false;

			$children = get_children(
				[
					'post_parent' => get_the_ID(),
					'post_type'   => get_post_type(),
					'numberposts' => -1,
					'post_status' => 'publish',
				]
			);

			$options['post_is_leaf'] = empty( $children );
		}

		// Node Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_node' ] ) {
			$options['post_is_node'] = false;

			if ( wp_get_post_parent_id( get_the_ID() ) ) {
				$children = get_children(
					[
						'post_parent' => get_the_ID(),
						'post_type'   => get_post_type(),
						'numberposts' => -1,
						'post_status' => 'publish',
					]
				);

				if ( ! empty( $children ) ) {
					$parents    = get_post_ancestors( get_the_ID() );
					$node_level = count( $parents + 1 );
					if ( ! $settings[ self::SECTION_PREFIX . 'post_node_level' ] || $node_level === $settings[ self::SECTION_PREFIX . 'post_node_level' ] ) {
						$options['post_is_node'] = true;
					}
				}
			}
		}

		// Siblings Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_sibling' ] ) {
			$options['post_has_siblings'] = false;

			$parent_post_id = wp_get_post_parent_id( get_the_ID() );
			if ( $parent_post_id ) {
				$children = get_children(
					[
						'post_parent' => $parent_post_id,
						'post_type'   => get_post_type(),
						'numberposts' => -1,
						'post_status' => 'publish',
					]
				);

				$options['post_has_siblings'] = ! empty( $children ) && count( $children ) > 1;
			}
		}

		// Terms Buddy Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_friend' ] ) {
			$options['post_has_term_friends'] = false;
			$post_id                          = get_the_ID();
			$post_ids                         = [];

			if ( $settings[ self::SECTION_PREFIX . 'post_friend_terms' ] ) {
				$term  = get_term( $settings[ self::SECTION_PREFIX . 'post_friend_terms' ] );
				$terms = [ $term ];
			} else {
				$terms = wp_get_post_terms();
			}

			if ( ! empty( $terms ) ) {
				foreach ( $terms as $term ) {
					$term_posts = get_posts(
						[
							'posts_per_page' => -1,
							'post_type'      => get_post_type(),
							'tax_query'      => [
								[
									'taxonomy' => $term->taxonomy,
									'field'    => 'term_id',
									'terms'    => $term->term_id,
								],
							],
						]
					);

					if ( ! empty( $term_posts ) && count( $term_posts ) > 1 ) {
						$post_ids = wp_list_pluck( $term_posts, 'ID' );

						if ( in_array( $post_id, $post_ids ) ) {
							$options['post_has_term_friends'] = true;
							break;
						}
					}
				}
			}
		}

		// Parameter Post.
		if ( $settings[ self::SECTION_PREFIX . 'post_with_param' ] ) {
			$options['post_has_custom_param'] = false;

			if ( 'not' === $settings[ self::SECTION_PREFIX . 'post_param_status' ] ) {
				if ( ! isset( $_REQUEST[ $settings[ self::SECTION_PREFIX . 'post_with_param' ] ] ) ) {
					$options['post_has_custom_param'] = true;
				}
			} elseif ( 'isset' === $settings[ self::SECTION_PREFIX . 'post_param_status' ] ) {
				if ( isset( $_REQUEST[ $settings[ self::SECTION_PREFIX . 'post_with_param' ] ] ) ) {
					$options['post_has_custom_param'] = true;
				}
			} elseif ( 'value' === $settings[ self::SECTION_PREFIX . 'post_param_status' ] ) {
				if ( isset( $_REQUEST[ $settings[ self::SECTION_PREFIX . 'post_with_param' ] ] ) &&
				$_REQUEST[ $settings[ self::SECTION_PREFIX . 'post_with_param' ] ] === $settings[ self::SECTION_PREFIX . 'post_param_value' ] ) {
					$options['post_has_custom_param'] = true;
				}
			}
		}

		return $options;
	}

}

PostVisibility::instance();
