<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class IpReferrerVisibility
 */
class IpReferrerVisibility extends Singleton {

	const OPTION_NAME = 'ip-referrer';

	/**
	 * IpReferrerVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'ip_referrer_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 *
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'ip_referrer_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'IP & Referrer', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$element->add_control(
			self::SECTION_PREFIX . 'ip_referrer_enabled',
			[
				'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'default'        => '',
				'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
				'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
				'return_value'   => 'yes',
				'render_type'    => 'ui',
				'prefix_class'   => 'stax-ip_referrer_enabled-',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'remote_ip',
			[
				'label'          => __( 'Remote IP', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXT,
				'description'    => __( 'List of IPs who will be able to view this element.<br>Separate IPs by comma. (ex. "1.1.1.1, 8.8.8.8, 4.4.4.4")', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'ip_referrer_enabled' => 'yes',
				],
				'render_type'    => 'ui',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'referrer_list',
			[
				'label'          => __( 'Referral sites URLs or domains', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::TEXTAREA,
				'placeholder'    => 'facebook.com' . PHP_EOL . 'google.com' . PHP_EOL . 'example.com',
				'description'    => __( 'List of referral websites (one per line).', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'ip_referrer_enabled' => 'yes',
				],
				'render_type'    => 'ui',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'referrer_match_type',
			[
				'label'          => __( 'Referrer Match Type', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SELECT,
				'default'        => 'domain',
				'options'        => [
					'domain'   => esc_html__( 'Match Domain', 'visibility-logic-elementor-pro' ),
					'strict'   => esc_html__( 'Match Exact URL', 'visibility-logic-elementor-pro' ),
					'contains' => esc_html__( 'Contains', 'visibility-logic-elementor-pro' ),
				],
				'description'    => __( 'How to check the referrer with the defined URLs', 'visibility-logic-elementor-pro' ),
				'condition'      => [
					self::SECTION_PREFIX . 'ip_referrer_enabled' => 'yes',
				],
				'render_type'    => 'ui',
				'style_transfer' => false,
			]
		);
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'ip_referrer_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'ip_referrer_enabled' ] ) {
			return $options;
		}

		if ( $settings[ self::SECTION_PREFIX . 'remote_ip' ] ) {
			$ips = explode( ',', $settings[ self::SECTION_PREFIX . 'remote_ip' ] );
			$ips = array_map( 'trim', $ips );

			$options['ip_referrer'] = in_array( Helpers::instance()->get_visitor_ip(), $ips );
		}

		$sites = [];

		if ( $settings[ self::SECTION_PREFIX . 'referrer_list' ] ) {
			$sites = explode( PHP_EOL, $settings[ self::SECTION_PREFIX . 'referrer_list' ] );
		}

		if ( ! empty( $sites ) ) {
			$has_ref = false;

			if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
				$referrers = array_map( 'trim', $sites );
				$referrer  = sanitize_text_field( $_SERVER['HTTP_REFERER'] );

				foreach ( $referrers as $ref ) {
					if ( $settings[ self::SECTION_PREFIX . 'referrer_match_type' ] === 'domain' ) {

						$referrer = parse_url( $_SERVER['HTTP_REFERER'], PHP_URL_HOST );
						$ref      = str_replace( [ 'https://', 'http://' ], '', $ref );

						if ( $ref === $referrer || $ref === str_replace( 'www.', '', $referrer ) ) {
							$has_ref = true;
						}
					} elseif ( $settings[ self::SECTION_PREFIX . 'referrer_match_type' ] === 'strict' ) {
						$ref = str_replace( [ 'https://', 'http://' ], '', $ref );
						$ref = rtrim( $ref, '/' );

						$referrer = str_replace( [ 'https://', 'http://' ], '', $referrer );
						$referrer = rtrim( $referrer, '/' );

						$has_ref = $ref === $referrer;
					} elseif ( $settings[ self::SECTION_PREFIX . 'referrer_match_type' ] === 'contains' ) {

						$has_ref = strpos( $referrer, $ref ) !== false;
					}
				}
			}

			$options['host_referrer'] = $has_ref;
		}

		return $options;
	}

}

IpReferrerVisibility::instance();
