<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class GeoLocationVisibility
 */
class GeoLocationVisibility extends Singleton {

	const OPTION_NAME = 'geo-location';

	/**
	 * GeoLocationVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'geo_location_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 *
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'geo_location_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'Geo Location', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$visi_settings              = get_option( 'stax_visibility_settings', [] );
		$meets_version_requirements = $this->meets_version_requirements( self::OPTION_NAME );
		$current_widget             = Plugin::instance()->get_pro_option( self::OPTION_NAME );

		if ( ! $meets_version_requirements ) {
			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['version'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-geo_location_enabled-',
					'style_transfer' => false,
				]
			);
		} elseif ( ! isset( $visi_settings['maxmind_account_id'] ) || ! $visi_settings['maxmind_account_id'] ||
			! isset( $visi_settings['maxmind_license_key'] ) || ! $visi_settings['maxmind_license_key'] ) {
			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => sprintf(
						esc_html__( 'This option requires %1$s account ID and license key setup!', 'visibility-logic-elementor-pro' ),
						'<a href="' . admin_url( 'options-general.php?page=' . STAX_VISIBILITY_SLUG_PREFIX . 'settings' ) . '" target="_blank">Maxmind API</a>',
					),
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-geo_location_enabled-',
					'style_transfer' => false,
				]
			);
		} else {
			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_enabled',
				[
					'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SWITCHER,
					'default'        => '',
					'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
					'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
					'return_value'   => 'yes',
					'render_type'    => 'ui',
					'prefix_class'   => 'stax-geo_location_enabled-',
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_type',
				[
					'label'          => __( 'Location Type', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SELECT,
					'default'        => 'country',
					'options'        => [
						'country' => esc_html__( 'Country', 'visibility-logic-elementor-pro' ),
						'city'    => esc_html__( 'City', 'visibility-logic-elementor-pro' ),
					],
					'condition'      => [
						self::SECTION_PREFIX . 'geo_location_enabled' => 'yes',
					],
					'render_type'    => 'ui',
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_countries',
				[
					'label'          => __( 'Countries', 'visibility-logic-elementor' ),
					'type'           => 'stax_query',
					'query_type'     => 'geo_location_country',
					'label_block'    => true,
					'multiple'       => true,
					'condition'      => [
						self::SECTION_PREFIX . 'geo_location_enabled' => 'yes',
						self::SECTION_PREFIX . 'geo_location_type' => 'country',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'geo_location_cities',
				[
					'label'          => __( 'Cities', 'visibility-logic-elementor' ),
					'type'           => 'stax_query',
					'query_type'     => 'geo_location_city',
					'label_block'    => true,
					'multiple'       => true,
					'condition'      => [
						self::SECTION_PREFIX . 'geo_location_enabled' => 'yes',
						self::SECTION_PREFIX . 'geo_location_type' => 'city',
					],
					'style_transfer' => false,
				]
			);
		}
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'geo_location_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'geo_location_enabled' ] ||
			! $this->meets_version_requirements( self::OPTION_NAME ) ) {
			return $options;
		}

		$visi_settings = get_option( 'stax_visibility_settings', [] );

		if ( ! isset( $visi_settings['maxmind_account_id'] ) || ! $visi_settings['maxmind_account_id'] ||
			! isset( $visi_settings['maxmind_license_key'] ) || ! $visi_settings['maxmind_license_key'] ) {
			return $options;
		}

		$visitor_ip = Helpers::instance()->get_visitor_ip();
		$geo_type   = $settings[ self::SECTION_PREFIX . 'geo_location_type' ];

		$saved_country = get_transient( "visibility.{$geo_type}.{$visitor_ip}.country" );
		$saved_city    = get_transient( "visibility.{$geo_type}.{$visitor_ip}.city" );

		if ( 'country' === $geo_type && $saved_country ) {
			$countries = [];

			foreach ( $settings[ self::SECTION_PREFIX . 'geo_location_countries' ] as $country ) {
				$countries[] = strtolower( $country );
			}

			$options['geo-location'] = in_array( strtolower( $saved_country ), $countries );

			return $options;
		}

		if ( 'city' === $geo_type && $saved_country && $saved_city ) {
			$raw_cities = $settings[ self::SECTION_PREFIX . 'geo_location_cities' ];
			$cities     = [];
			$countries  = [];

			foreach ( $raw_cities as $city ) {
				$city_data   = explode( '|', $city );
				$cities[]    = strtolower( $city_data[0] );
				$countries[] = strtolower( $city_data[1] );
			}

			$options['geo-location'] = in_array( strtolower( $saved_city ), $cities ) && in_array( strtolower( $saved_country ), $countries );

			return $options;
		}

		$country_name = '';
		$city_name    = '';

		try {
			$reader = new \GeoIp2\WebService\Client( $visi_settings['maxmind_account_id'], $visi_settings['maxmind_license_key'], [ 'en' ], [ 'host' => 'geolite.info' ] );

			$record       = $reader->country( $visitor_ip );
			$country_name = $record->country->isoCode;

			if ( 'city' === $geo_type ) {
				$record    = $reader->city( $visitor_ip );
				$city_name = $record->city->name;
			}
		} catch ( \GeoIp2\Exception\AddressNotFoundException $e ) {
			if ( current_user_can( 'manage_options' ) ) {
				echo 'Admin notice: IP address not found.';
			}
		} catch ( \MaxMind\Exception\PermissionRequiredException $e ) {
			if ( current_user_can( 'manage_options' ) ) {
				echo 'Admin notice: Cannot access this MaxMind server.';
			}
		} catch ( \Exception $e ) {
			if ( current_user_can( 'manage_options' ) ) {
				echo 'Admin notice: Cannot authenticate with MaxMind. Please check Account ID and API Key.';
			}
		}

		if ( 'country' === $geo_type && ! is_string( $country_name ) ) {
			$options['geo-location'] = false;
			return $options;
		}

		if ( 'city' === $geo_type && ! is_string( $city_name ) ) {
			$options['geo-location'] = false;
			return $options;
		}

		set_transient( "visibility.{$geo_type}.{$visitor_ip}.country", $country_name, 60*60*24 );
		set_transient( "visibility.{$geo_type}.{$visitor_ip}.city", $city_name, 60*60*24 );

		if ( 'country' === $geo_type ) {
			$countries = [];

			foreach ( $settings[ self::SECTION_PREFIX . 'geo_location_countries' ] as $country ) {
				$countries[] = strtolower( $country );
			}

			$options['geo-location'] = in_array( strtolower( $country_name ), $countries );
		} elseif ( 'city' === $geo_type ) {
			$raw_cities = $settings[ self::SECTION_PREFIX . 'geo_location_cities' ];
			$cities     = [];
			$countries  = [];

			foreach ( $raw_cities as $city ) {
				$city_data   = explode( '|', $city );
				$cities[]    = strtolower( $city_data[0] );
				$countries[] = strtolower( $city_data[1] );
			}

			$options['geo-location'] = in_array( strtolower( $city_name ), $cities ) && in_array( strtolower( $country_name ), $countries );
		}

		return $options;
	}
}

GeoLocationVisibility::instance();
