<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class EDDUsersVisibility
 */
class EDDUsersVisibility extends Singleton {

	const OPTION_NAME = 'edd-users';

	/**
	 * EDDUsersVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'edd_users_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 *
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'edd_users_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'EDD Users', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$meets_plugin_requirements  = $this->meets_plugin_requirements( self::OPTION_NAME );
		$meets_version_requirements = $this->meets_version_requirements( self::OPTION_NAME );
		$current_widget             = Plugin::instance()->get_pro_option( self::OPTION_NAME );

		if ( ! $meets_version_requirements ) {
			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['version'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-edd_users_enabled-',
					'style_transfer' => false,
				]
			);
		} elseif ( ! $meets_plugin_requirements ) {
			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['plugins'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-edd_users_enabled-',
					'style_transfer' => false,
				]
			);
		} else {
			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_enabled',
				[
					'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SWITCHER,
					'default'        => '',
					'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
					'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
					'return_value'   => 'yes',
					'prefix_class'   => 'stax-edd_users_enabled-',
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_selection',
				[
					'label'          => __( 'Condition', 'visibility-logic-elementor-pro' ),
					'label_block'    => true,
					'type'           => Controls_Manager::SELECT,
					'options'        => [
						'any'     => __( 'User has purchased any product', 'visibility-logic-elementor-pro' ),
						'product' => __( 'User has purchased specific product', 'visibility-logic-elementor-pro' ),
					],
					'default'        => 'any',
					'condition'      => [
						self::SECTION_PREFIX . 'edd_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_product_condition',
				[
					'label'          => __( 'Products Condition', 'visibility-logic-elementor-pro' ),
					'label_block'    => true,
					'type'           => Controls_Manager::SELECT,
					'options'        => [
						'one' => __( 'At least one', 'visibility-logic-elementor-pro' ),
						'all' => __( 'All', 'visibility-logic-elementor-pro' ),
					],
					'default'        => 'one',
					'condition'      => [
						self::SECTION_PREFIX . 'edd_users_selection' => 'product',
						self::SECTION_PREFIX . 'edd_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'edd_users_product',
				[
					'label'          => __( 'Products', 'visibility-logic-elementor-pro' ),
					'type'           => 'stax_query',
					'query_type'     => 'edd_products',
					'label_block'    => true,
					'multiple'       => true,
					'condition'      => [
						self::SECTION_PREFIX . 'edd_users_selection' => 'product',
						self::SECTION_PREFIX . 'edd_users_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);
		}
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'edd_users_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'edd_users_enabled' ] ||
			! $this->meets_plugin_requirements( self::OPTION_NAME ) ||
			! $this->meets_version_requirements( self::OPTION_NAME ) ) {
			return $options;
		}

		switch ( $settings[ self::SECTION_PREFIX . 'edd_users_selection' ] ) {
			case 'any':
				$options['edd_users'] = edd_has_purchases( get_current_user_id() );
				break;
			case 'product':
				$options['edd_users'] = $this->user_has_bought_product( $settings[ self::SECTION_PREFIX . 'edd_users_product' ], $settings[ self::SECTION_PREFIX . 'edd_users_product_condition' ] );
				break;
			default:
				$options['edd_users'] = false;
		}

		return $options;
	}

	/**
	 * Check if user has bought product
	 *
	 * @param array  $product_ids
	 * @param string $condition
	 * @return boolean
	 */
	public function user_has_bought_product( $product_ids = [], $condition = 'one' ) {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		if ( empty( $product_ids ) ) {
			return false;
		}

		if ( ! in_array( $condition, [ 'one', 'all' ] ) ) {
			return false;
		}

		$has_bought_product = 'all' === $condition;
		$user_id            = get_current_user_id();

		foreach ( $product_ids as $key => $product_id ) {
			switch ( $condition ) {
				case 'one':
					if ( edd_has_user_purchased( $user_id, $product_id ) ) {
						$has_bought_product = true;
					}
					break;
				case 'all':
					if ( ! edd_has_user_purchased( $user_id, $product_id ) ) {
						$has_bought_product = false;
					}
					break;
				default:
			}
		}

		return $has_bought_product;
	}

}

EDDUsersVisibility::instance();
