<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Modules\DynamicTags\Module;

/**
 * Class DynamicConditionsVisibility
 */
class DynamicConditionsVisibility extends Singleton {

	const OPTION_NAME = 'dynamic-conditions';

	private $element_settings = [];

	private $date_instance;

	/**
	 * DynamicConditionsVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'dynamic_conditions_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );

		$this->date_instance = new Date();
	}

	/**
	 * Register section
	 *
	 * @param $element
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'dynamic_conditions_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'Dynamic Conditions', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$meets_plugin_requirements = $this->meets_plugin_requirements( self::OPTION_NAME );

		if ( ! $meets_plugin_requirements ) {
			$current_widget = Plugin::instance()->get_pro_option( self::OPTION_NAME );

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_requirement',
				[
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => $current_widget['require']['text']['plugins'],
					'content_classes' => 'stx-plugin-requirement',
					'style_transfer'  => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_requirement_enabled',
				[
					'label'          => __( 'Error', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HIDDEN,
					'default'        => 'error',
					'prefix_class'   => 'stax-dynamic_conditions_enabled-',
					'style_transfer' => false,
				]
			);
		} else {
			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_enabled',
				[
					'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SWITCHER,
					'default'        => '',
					'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
					'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
					'return_value'   => 'yes',
					'prefix_class'   => 'stax-dynamic_conditions_enabled-',
					'style_transfer' => false,
				]
			);

			$value_condition = [
				'equal',
				'not_equal',
				'contains',
				'not_contains',
				'less',
				'greater',
				'between',
				'in_array',
				'in_array_contains',
			];

			$render_type = 'ui';

			$categories = [
				Module::BASE_GROUP,
				Module::TEXT_CATEGORY,
				Module::URL_CATEGORY,
				Module::GALLERY_CATEGORY,
				Module::IMAGE_CATEGORY,
				Module::MEDIA_CATEGORY,
				Module::POST_META_CATEGORY,
			];

			$categories_text_only = [
				Module::BASE_GROUP,
				Module::TEXT_CATEGORY,
				Module::URL_CATEGORY,
				Module::POST_META_CATEGORY,
			];

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_tag',
				[
					'label'          => __( 'Dynamic Tag', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::MEDIA,
					'dynamic'        => [
						'active'     => true,
						'categories' => $categories,
					],
					'render_type'    => $render_type,
					'placeholder'    => __( 'Select condition field', 'visibility-logic-elementor-pro' ),
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_condition',
				[
					'label'          => __( 'Condition', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SELECT2,
					'multiple'       => false,
					'label_block'    => true,
					'options'        => [
						'equal'             => __( 'Is equal to', 'visibility-logic-elementor-pro' ),
						'not_equal'         => __( 'Is not equal to', 'visibility-logic-elementor-pro' ),
						'contains'          => __( 'Contains', 'visibility-logic-elementor-pro' ),
						'not_contains'      => __( 'Does not contain', 'visibility-logic-elementor-pro' ),
						'empty'             => __( 'Is empty', 'visibility-logic-elementor-pro' ),
						'not_empty'         => __( 'Is not empty', 'visibility-logic-elementor-pro' ),
						'between'           => __( 'Between', 'visibility-logic-elementor-pro' ),
						'less'              => __( 'Less than', 'visibility-logic-elementor-pro' ),
						'greater'           => __( 'Greater than', 'visibility-logic-elementor-pro' ),
						'in_array'          => __( 'In array', 'visibility-logic-elementor-pro' ),
						'in_array_contains' => __( 'In array contains', 'visibility-logic-elementor-pro' ),
					],
					'description'    => __( 'Select your condition for this widget visibility.', 'visibility-logic-elementor-pro' ),
					'prefix_class'   => 'dc-has-condition dc-condition-',
					'render_type'    => 'template',
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_type',
				[
					'label'          => __( 'Compare Type', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::SELECT,
					'multiple'       => false,
					'label_block'    => true,
					'options'        => [
						'default'   => __( 'Text', 'visibility-logic-elementor-pro' ),
						'date'      => __( 'Date', 'visibility-logic-elementor-pro' ),
						'days'      => __( 'Weekdays', 'visibility-logic-elementor-pro' ),
						'months'    => __( 'Months', 'visibility-logic-elementor-pro' ),
						'strtotime' => __( 'String to time', 'visibility-logic-elementor-pro' ),
					],
					'default'        => 'default',
					'render_type'    => $render_type,
					'description'    => __( 'Select what do you want to compare', 'visibility-logic-elementor-pro' ),
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => $value_condition,
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_value',
				[
					'type'           => Controls_Manager::TEXTAREA,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'dynamic'        => [
						'active'     => true,
						'categories' => $categories_text_only,
					],
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => $value_condition,
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => [ 'default', 'strtotime' ],
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_value2',
				[
					'type'           => Controls_Manager::TEXTAREA,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'description'    => __( 'Add a second condition value, if between is selected', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'dynamic'        => [
						'active'     => true,
						'categories' => $categories_text_only,
					],
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'between' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => [ 'default', 'strtotime' ],
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_date_value',
				[
					'type'           => Controls_Manager::DATE_TIME,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => $value_condition,
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'date',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_date_value2',
				[
					'type'           => Controls_Manager::DATE_TIME,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'description'    => __( 'Add a second condition value, if between is selected', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'between' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'date',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_day_array_value',
				[
					'type'           => Controls_Manager::SELECT2,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getDaysTranslated(),
					'multiple'       => true,
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'in_array' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'days',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);
			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_day_value',
				[
					'type'           => Controls_Manager::SELECT,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getDaysTranslated(),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => array_diff( $value_condition, [ 'in_array' ] ),
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'days',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_day_value2',
				[
					'type'           => Controls_Manager::SELECT,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'description'    => __( 'Add a second condition value, if between is selected', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getDaysTranslated(),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'between' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'days',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_month_array_value',
				[
					'type'           => Controls_Manager::SELECT2,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getMonthsTranslated(),
					'multiple'       => true,
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'in_array' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'months',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_month_value',
				[
					'type'           => Controls_Manager::SELECT,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ),
					'description'    => __( 'Add your conditional value to compare here.', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getMonthsTranslated(),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => array_diff( $value_condition, [ 'in_array' ] ),
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'months',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_month_value2',
				[
					'type'           => Controls_Manager::SELECT,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'description'    => __( 'Add a second condition value, if between is selected', 'visibility-logic-elementor-pro' ),
					'options'        => Date::getMonthsTranslated(),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'between' ],
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'months',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_in_array_description',
				[
					'type'           => Controls_Manager::RAW_HTML,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'show_label'     => false,
					'raw'            => __( 'Use comma-separated values, to check if dynamic-value is equal with one of each item.', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'in_array' ],
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_in_array_contains_description',
				[
					'type'           => Controls_Manager::RAW_HTML,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'show_label'     => false,
					'raw'            => __( 'Use comma-separated values, to check if dynamic-value contains one of each item.', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => [ 'in_array_contains' ],
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$languageArray = explode( '_', get_locale() );
			$language      = array_shift( $languageArray );
			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_date_description',
				[
					'type'           => Controls_Manager::RAW_HTML,
					'label'          => __( 'Conditional value', 'visibility-logic-elementor-pro' ) . ' 2',
					'show_label'     => false,
					'raw'            => '<div class="elementor-control-field-description">'
						. '<a href="https://php.net/manual/' . $language . '/function.strtotime.php" target="_blank">'
						. __( 'Supported Date and Time Formats', 'visibility-logic-elementor-pro' ) . '</a></div>',
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_condition' => $value_condition,
						self::SECTION_PREFIX . 'dynamic_conditions_type'      => 'strtotime',
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_headline_expert',
				[
					'label'          => __( 'Expert', 'visibility-logic-elementor-pro' ),
					'type'           => Controls_Manager::HEADING,
					'separator'      => 'before',
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_parse_shortcodes',
				[
					'type'           => Controls_Manager::SWITCHER,
					'label'          => __( 'Parse shortcodes', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);

			$element->add_control(
				self::SECTION_PREFIX . 'dynamic_conditions_prevent_date_parsing',
				[
					'type'           => Controls_Manager::SWITCHER,
					'label'          => __( 'Prevent date parsing', 'visibility-logic-elementor-pro' ),
					'render_type'    => $render_type,
					'condition'      => [
						self::SECTION_PREFIX . 'dynamic_conditions_enabled' => 'yes',
					],
					'style_transfer' => false,
				]
			);
		}
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();



		if ( ! isset( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'dynamic_conditions_enabled' ] ||
			! $this->meets_plugin_requirements( self::OPTION_NAME ) ) {
			return $options;
		}

		$parsed_settings = $this->get_element_settings( $item );


		if ( ! $this->has_condition( $parsed_settings ) ) {
			return $options;
		}

		$options['dynamic_conditions'] = $this->loop_values( $parsed_settings );

		return $options;
	}

	/**
	 * Loop widget-values and check the condition
	 *
	 * @param $settings
	 * @return bool|mixed
	 */
	private function loop_values( $settings ) {
		$condition               = false;
		$dynamic_tag_value_array = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_tag' );

		if ( ! is_array( $dynamic_tag_value_array ) ) {
			$dynamic_tag_value_array = [ $dynamic_tag_value_array ];
		}

		$compare_type = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_type', 'default' );
		$check_values = $this->get_check_value( $compare_type, $settings );
		$check_value  = $check_values[0];
		$check_value2 = $check_values[1];

		foreach ( $dynamic_tag_value_array as $dynamic_tag_value ) {
			if ( is_array( $dynamic_tag_value ) ) {
				if ( ! empty( $dynamic_tag_value['id'] ) ) {
					$dynamic_tag_value = wp_get_attachment_url( $dynamic_tag_value['id'] );
				} else {
					continue;
				}
			}

			if ( ! empty( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_parse_shortcodes' ] ) ) {
				$dynamic_tag_value = do_shortcode( $dynamic_tag_value );
			}

			$this->parse_dynamic_tag_value( $dynamic_tag_value, $compare_type );

			$compare_values = $this->compare_values( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_condition' ], $dynamic_tag_value, $check_value, $check_value2 );
			$condition      = $compare_values[0];
			$break          = $compare_values[1];
			$break_false    = $compare_values[2];

			if ( $break && $condition ) {
				break;
			}

			if ( $break_false && ! $condition ) {
				break;
			}
		}

		return $condition;
	}

	/**
	 * Gets settings with english locale (needed for date)
	 *
	 * @param \Elementor\Element_Base $item
	 * @return array
	 */
	private function get_element_settings( $item ) {
		$id                = $item->get_id();
		$settings          = $item->get_settings_for_display();
		$original_settings = $settings;

		$fields = '__dynamic__
            dynamic_conditions_tag
            dynamic_conditions_condition
            dynamic_conditions_type
            dynamic_conditions_day_value
            dynamic_conditions_day_value2
            dynamic_conditions_day_array_value
            dynamic_conditions_month_value
            dynamic_conditions_month_value2
            dynamic_conditions_month_array_value
            dynamic_conditions_date_value
            dynamic_conditions_date_value2
            dynamic_conditions_value
            dynamic_conditions_value2
            dynamic_conditions_parse_shortcodes';

		$field_array = explode( "\n", $fields );

		$prevent_date_parsing = $settings[ self::SECTION_PREFIX . 'dynamic_conditions_prevent_date_parsing' ];
		$this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_prevent_date_parsing' ] = $prevent_date_parsing;

		if ( empty( $prevent_date_parsing ) ) {
			$currentLocale = setlocale( LC_ALL, 0 );
			setlocale( LC_ALL, 'en_GB' );
			add_filter( 'date_i18n', [ $this->date_instance, 'filterDateI18n' ], 10, 4 );
			add_filter( 'get_the_date', [ $this->date_instance, 'filterPostDate' ], 10, 3 );
			add_filter( 'get_the_modified_date', [ $this->date_instance, 'filterPostDate' ], 10, 3 );
		}

		foreach ( $field_array as $field ) {
			$field = trim( $field );
			if ( '__dynamic__' === $field ) {
				$this->element_settings[ $id ][ $field ] = isset( $original_settings[ $field ] ) ? $original_settings[ $field ] : '';
			} else {
				$this->element_settings[ $id ][ self::SECTION_PREFIX . $field ] = isset( $original_settings[ self::SECTION_PREFIX . $field ] ) ? $original_settings[ self::SECTION_PREFIX . $field ] : '';
			}
		}

		if ( empty( $prevent_date_parsing ) ) {
			remove_filter( 'date_i18n', [ $this->date_instance, 'filterDateI18n' ], 10 );
			remove_filter( 'get_the_date', [ $this->date_instance, 'filterPostDate' ], 10 );
			remove_filter( 'get_the_modified_date', [ $this->date_instance, 'filterPostDate' ], 10 );

			Date::setLocale( $currentLocale );
		}

		$tag_data = $this->get_dynamic_tag_data( $id );
		$this->convert_acf_date( $id, $tag_data );

		$this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_data' ] = [
			'id'           => $id,
			'type'         => $item->get_type(),
			'name'         => $item->get_name(),
			'selected_tag' => $tag_data['selected_tag'],
			'tag_data'     => $tag_data['tag_data'],
			'tag_key'      => $tag_data['tag_key'],
		];

		return $this->element_settings[ $id ];
	}

	/**
	 * Returns data of dynamic tag
	 *
	 * @param $id
	 * @return array
	 */
	private function get_dynamic_tag_data( $id ) {
		$dynamic_empty = empty( $this->element_settings[ $id ]['__dynamic__'] )
			|| empty( $this->element_settings[ $id ]['__dynamic__'][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ] );
		$static_empty  = empty( $this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ] )
			|| empty( $this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ]['url'] );

		if ( $dynamic_empty && $static_empty ) {
			return [
				'selected_tag' => null,
				'tag_data'     => null,
				'tag_key'      => null,
			];
		}

		$selected_tag = null;
		$tag_settings = null;
		$tag_data     = [];
		$tag_key      = null;

		if ( $dynamic_empty ) {
			$this->element_settings[ $id ]['__dynamic__'] = [
				self::SECTION_PREFIX . 'dynamic_conditions_tag' => $this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ],
			];

			$selected_tag = 'static';
		}

		$tag  = $this->element_settings[ $id ]['__dynamic__'][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ];
		$data = Parser::parse_shortcodes( $tag, true );

		foreach ( $data['attrs'] as $attr ) {
			if ( isset( $attr['name'] ) ) {
				$selected_tag = $attr['name'];
			}
		}

		if ( strpos( $selected_tag, 'acf-' ) === false ) {
			if ( $data ) {
				$settings_encoded = '';

				foreach ( $data['attrs'] as $attr ) {
					if ( isset( $attr['settings'] ) ) {
						$settings_encoded = $attr['settings'];
					}
				}

				if ( $settings_encoded ) {
					$tag_settings = json_decode( urldecode( $settings_encoded ), true );

					if ( ! empty( $tag_settings['key'] ) ) {
						$tag_key  = $tag_settings['key'];
						$tag_data = get_field_object( explode( ':', $tag_settings['key'] )[0] );
					}
				}
			}
		}

		return [
			'selected_tag' => $selected_tag,
			'tag_data'     => $tag_data,
			'tag_key'      => $tag_key,
		];
	}

	/**
	 * Convert acf date to timestamp
	 *
	 * @param $id
	 * @param array $data
	 */
	private function convert_acf_date( $id, array $data ) {
		if ( empty( $data ) ) {
			return;
		}

		if ( ! empty( $this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_prevent_date_parsing' ] ) ) {
			return;
		}

		$allowedTypes = [
			'date_time_picker',
			'date_picker',
		];

		$tag_data = $data['tag_data'];

		if ( empty( $data['tag_key'] ) || strpos( $data['selected_tag'], 'acf-' ) !== 0 ) {
			return;
		}

		if ( empty( $tag_data['type'] ) || ! in_array( trim( $tag_data['type'] ), $allowedTypes, true ) ) {
			return;
		}

		if ( empty( $tag_data['value'] ) || empty( $tag_data['return_format'] ) ) {
			return;
		}

		$time = \DateTime::createFromFormat( $tag_data['return_format'], Date::unTranslateDate( $tag_data['value'] ) );

		if ( empty( $time ) ) {
			return;
		}

		if ( $tag_data['type'] === 'date_picker' ) {
			$time->setTime( 0, 0, 0 );
		}

		$timestamp = $time->getTimestamp();

		$this->element_settings[ $id ][ self::SECTION_PREFIX . 'dynamic_conditions_tag' ] = $timestamp;
	}

	/**
	 * Checks if element has a condition
	 *
	 * @param $settings
	 * @return bool
	 */
	public function has_condition( $settings ) {

		if ( empty( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_data' ]['selected_tag'] ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Get value to compare
	 *
	 * @param $compareType
	 * @param $settings
	 * @return array
	 */
	private function get_check_value( $compareType, $settings ) {
		switch ( $compareType ) {
			case 'days':
				if ( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_condition' ] === 'in_array' ) {
					$check_value = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_day_array_value' );
					$check_value = $this->parse_shortcode( $check_value, $settings );

					if ( is_array( $check_value ) ) {
						$check_value = implode( ',', $check_value );
					}
				} else {
					$check_value = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_day_value' );
					$check_value = $this->parse_shortcode( $check_value );
				}
				$check_value2 = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_day_value2' );
				$check_value2 = $this->parse_shortcode( $check_value2, $settings );
				$check_value  = Date::unTranslateDate( $check_value );
				$check_value2 = Date::unTranslateDate( $check_value2 );
				break;
			case 'months':
				if ( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_condition' ] === 'in_array' ) {
					$check_value = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_month_array_value' );
					$check_value = $this->parse_shortcode( $check_value, $settings );

					if ( is_array( $check_value ) ) {
						$check_value = implode( ',', $check_value );
					}
				} else {
					$check_value = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_month_value' );
					$check_value = $this->parse_shortcode( $check_value, $settings );
				}
				$check_value2 = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_month_value2' );
				$check_value2 = $this->parse_shortcode( $check_value2, $settings );
				$check_value  = Date::unTranslateDate( $check_value );
				$check_value2 = Date::unTranslateDate( $check_value2 );
				break;
			case 'date':
				$check_value  = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_date_value' );
				$check_value2 = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_date_value2' );
				$check_value  = $this->parse_shortcode( $check_value, $settings );
				$check_value2 = $this->parse_shortcode( $check_value2, $settings );
				$check_value  = Date::stringToTime( $check_value );
				$check_value2 = Date::stringToTime( $check_value2 );
				break;
			case 'strtotime':
				$check_value  = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_value' );
				$check_value2 = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_value2' );
				$check_value  = $this->parse_shortcode( $check_value, $settings );
				$check_value2 = $this->parse_shortcode( $check_value2, $settings );
				$check_value  = Date::stringToTime( $check_value );
				$check_value2 = Date::stringToTime( $check_value2 );
				break;
			case 'default':
			default:
				$check_value  = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_value' );
				$check_value2 = $this->check_empty( $settings, self::SECTION_PREFIX . 'dynamic_conditions_value2' );
				$check_value  = $this->parse_shortcode( $check_value, $settings );
				$check_value2 = $this->parse_shortcode( $check_value2, $settings );
				break;
		}

		return [
			$check_value,
			$check_value2,
		];
	}

	/**
	 * Compare values
	 *
	 * @param $compare
	 * @param $dynamic_tag_value
	 * @param $check_value
	 * @param $check_value2
	 * @return array
	 */
	private function compare_values( $compare, $dynamic_tag_value, $check_value, $check_value2 ) {
		$break       = false;
		$break_false = false;
		$condition   = false;

		switch ( $compare ) {
			case 'equal':
				$condition = $check_value == $dynamic_tag_value;
				$break     = true;
				break;
			case 'not_equal':
				$condition   = $check_value != $dynamic_tag_value;
				$break_false = true;
				break;
			case 'contains':
				if ( empty( $check_value ) ) {
					break;
				}
				$condition = strpos( $dynamic_tag_value, $check_value ) !== false;
				$break     = true;
				break;
			case 'not_contains':
				if ( empty( $check_value ) ) {
					break;
				}

				$condition   = strpos( $dynamic_tag_value, $check_value ) === false;
				$break_false = true;
				break;
			case 'empty':
				$condition   = empty( $dynamic_tag_value );
				$break_false = true;
				break;
			case 'not_empty':
				$condition = ! empty( $dynamic_tag_value );
				$break     = true;
				break;
			case 'less':
				if ( is_numeric( $dynamic_tag_value ) ) {
					$condition = $dynamic_tag_value < $check_value;
				} else {
					$condition = strlen( $dynamic_tag_value ) < strlen( $check_value );
				}

				$break = true;
				break;
			case 'greater':
				if ( is_numeric( $dynamic_tag_value ) ) {
					$condition = $dynamic_tag_value > $check_value;
				} else {
					$condition = strlen( $dynamic_tag_value ) > strlen( $check_value );
				}

				$break = true;
				break;
			case 'between':
				$condition = $dynamic_tag_value >= $check_value && $dynamic_tag_value <= $check_value2;
				$break     = true;
				break;
			case 'in_array':
				$condition = in_array( $dynamic_tag_value, explode( ',', $check_value ) ) !== false;
				$break     = true;
				break;
			case 'in_array_contains':
				foreach ( explode( ',', $check_value ) as $toCheck ) {
					$condition = strpos( $dynamic_tag_value, $toCheck ) !== false;
					if ( $condition ) {
						break;
					}
				}

				$break = true;
				break;
		}

		return [
			$condition,
			$break,
			$break_false,
		];
	}

	/**
	 * Parse shortcode if active
	 *
	 * @param $value
	 * @param array $settings
	 * @return string
	 */
	private function parse_shortcode( $value, $settings = [] ) {
		if ( empty( $settings[ self::SECTION_PREFIX . 'dynamic_conditions_parse_shortcodes' ] ) ) {
			return $value;
		}

		return do_shortcode( $value );
	}

	/**
	 * Parse value of widget to timestamp, day or month
	 *
	 * @param $dynamic_tag_value
	 * @param $compareType
	 */
	private function parse_dynamic_tag_value( &$dynamic_tag_value, $compareType ) {
		switch ( $compareType ) {
			case 'days':
				$dynamic_tag_value = date( 'N', Date::stringToTime( $dynamic_tag_value ) );
				break;
			case 'months':
				$dynamic_tag_value = date( 'n', Date::stringToTime( $dynamic_tag_value ) );
				break;
			case 'strtotime':
			case 'date':
				$dynamic_tag_value = Date::stringToTime( $dynamic_tag_value );
				break;
		}
	}

	/**
	 * Checks if an array or entry in array is empty and return its value
	 *
	 * @param array $array
	 * @param null  $key
	 * @return array|mixed|null
	 */
	public function check_empty( $array = [], $key = null, $fallback = null ) {
		if ( empty( $key ) ) {
			return ! empty( $array ) ? $array : $fallback;
		}

		return ! empty( $array[ $key ] ) ? $array[ $key ] : $fallback;
	}

}

DynamicConditionsVisibility::instance();
