<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class ArchiveVisibility
 */
class ArchiveVisibility extends Singleton {

	const OPTION_NAME = 'archive';

	/**
	 * ArchiveVisibility constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'archive_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'archive_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'Archives', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$element->add_control(
			self::SECTION_PREFIX . 'archive_enabled',
			[
				'label'          => __( 'Enable', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SWITCHER,
				'default'        => '',
				'label_on'       => __( 'Yes', 'visibility-logic-elementor-pro' ),
				'label_off'      => __( 'No', 'visibility-logic-elementor-pro' ),
				'return_value'   => 'yes',
				'prefix_class'   => 'stax-archive_enabled-',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'archive_custom_post_type',
			[
				'label'          => __( 'Post Type Archive', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SELECT2,
				'options'        => Helpers::get_post_types(),
				'multiple'       => true,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'archive_enabled' => 'yes',
				],
				'description'    => esc_html( 'Checks if the current page is an archive for any of the selected Post Types.', 'visibility-logic-elementor-pro' ),
				'style_transfer' => false,
			]
		);

		$taxonomies = Helpers::get_taxonomies();

		$element->add_control(
			self::SECTION_PREFIX . 'archive_taxonomy',
			[
				'label'          => __( 'Taxonomy Archive', 'visibility-logic-elementor-pro' ),
				'type'           => Controls_Manager::SELECT2,
				'options'        => $taxonomies,
				'multiple'       => false,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'archive_enabled' => 'yes',
				],
				'description'    => esc_html( 'Checks if the current page is a Taxonomy archive.', 'visibility-logic-elementor-pro' ),
				'style_transfer' => false,
			]
		);

		foreach ( $taxonomies as $taxonomy_key => $taxonomy ) {
			if ( $taxonomy_key ) {
				$element->add_control(
					self::SECTION_PREFIX . 'archive_taxonomy_term_' . $taxonomy_key,
					[
						'label'          => __( 'Terms', 'visibility-logic-elementor-pro' ),
						'type'           => Controls_Manager::SELECT2,
						'options'        => Helpers::get_taxonomy_terms( $taxonomy_key ),
						'multiple'       => true,
						'label_block'    => true,
						'condition'      => [
							self::SECTION_PREFIX . 'archive_enabled' => 'yes',
							self::SECTION_PREFIX . 'archive_taxonomy' => $taxonomy_key,
						],
						'description'    => esc_html( 'Checks if the current page is a Taxonomy archive for the selected term.', 'visibility-logic-elementor-pro' ),
						'style_transfer' => false,
					]
				);
			}
		}

	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'archive_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'archive_enabled' ] ) {
			return $options;
		}

		// Post Type.
		if ( ! empty( $settings[ self::SECTION_PREFIX . 'archive_custom_post_type' ] ) ) {
			$options['archive_custom_post_type'] = false;

			if ( is_post_type_archive( $settings[ self::SECTION_PREFIX . 'archive_custom_post_type' ] ) ) {
				$options['archive_custom_post_type'] = true;
			}
		}

		// Taxonomy.
		if ( $settings[ self::SECTION_PREFIX . 'archive_taxonomy' ] ) {
			$options['archive_taxonomy'] = false;

			if ( ! is_tax( $settings[ self::SECTION_PREFIX . 'archive_taxonomy' ] ) ) {
				$options['archive_taxonomy'] = false;
			} else {
				$tax_term_key = self::SECTION_PREFIX . 'archive_taxonomy_term_' . $settings[ self::SECTION_PREFIX . 'archive_taxonomy' ];

				// Match any term from the chosen taxonomy
				if ( empty( $settings[ $tax_term_key ] ) ) {
					$options['archive_taxonomy'] = true;
				} elseif ( is_array( $settings[ $tax_term_key ] ) ) {
					if ( in_array( get_queried_object_id(), $settings[ $tax_term_key ], false ) ) {
						$options['archive_taxonomy'] = true;
					} else {
						$options['archive_taxonomy'] = false;
					}
				}
			}
		}

		return $options;
	}

}

ArchiveVisibility::instance();
