<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;

/**
 * Class AdvancedUserMeta
 */
class AdvancedUserMeta extends Singleton {

	const OPTION_NAME = 'advanced_user_meta';

	/**
	 * AdvancedUserMeta constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->register_elementor_settings( 'advanced_user_meta_section' );

		add_filter( 'stax/visibility/apply_conditions', [ $this, 'apply_conditions' ], 10, 3 );
	}

	/**
	 * Register section
	 *
	 * @param $element
	 * @return void
	 */
	public function register_section( $element ) {
		$element->start_controls_section(
			self::SECTION_PREFIX . 'advanced_user_meta_section',
			[
				'tab'       => self::VISIBILITY_TAB,
				'label'     => __( 'Advanced User Meta', 'visibility-logic-elementor-pro' ),
				'condition' => [
					self::SECTION_PREFIX . 'enabled' => 'yes',
				],
			]
		);

		$element->end_controls_section();
	}

	/**
	 * @param $element \Elementor\Widget_Base
	 * @param $section_id
	 * @param $args
	 */
	public function register_controls( $element, $args ) {
		$element->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_enabled',
			[
				'label'          => __( 'Enable', 'visibility-logic-elementor' ),
				'type'           => Controls_Manager::SWITCHER,
				'default'        => '',
				'label_on'       => __( 'Yes', 'visibility-logic-elementor' ),
				'label_off'      => __( 'No', 'visibility-logic-elementor' ),
				'return_value'   => 'yes',
				'prefix_class'   => 'stax-advanced_user_meta_enabled-',
				'style_transfer' => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_rule',
			[
				'type'           => Controls_Manager::SELECT2,
				'label'          => __( 'Rule', 'visibility-logic-elementor' ),
				'description'    => __( '"AND" = all conditions must be true, "OR" = at least one condition must be true', 'visibility-logic-elementor' ),
				'options'        => [
					'and' => __( 'AND', 'visibility-logic-elementor' ),
					'or'  => __( 'OR', 'visibility-logic-elementor' ),
				],
				'default'        => 'and',
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'advanced_user_meta_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_options',
			[
				'type'           => 'stax_query',
				'label'          => __( 'Meta Name', 'visibility-logic-elementor' ),
				'query_type'     => 'fields',
				'object_type'    => 'user',
				'placeholder'    => __( 'Meta key or Name', 'visibility-logic-elementor' ),
				'label_block'    => true,
				'multiple'       => true,
				'style_transfer' => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_status',
			[
				'type'           => Controls_Manager::SELECT2,
				'label'          => __( 'Meta Condition', 'visibility-logic-elementor' ),
				'description'    => __( 'Select the condition that must be met by the meta', 'visibility-logic-elementor' ),
				'options'        => [
					'none'                    => __( 'None', 'visibility-logic-elementor' ),
					'empty'                   => __( 'Empty', 'visibility-logic-elementor' ),
					'not_empty'               => __( 'Not empty', 'visibility-logic-elementor' ),
					'specific_value'          => __( 'Is equal to', 'visibility-logic-elementor' ),
					'specific_value_multiple' => __( 'Is equal to one of', 'visibility-logic-elementor' ),
					'not_specific_value'      => __( 'Not equal to', 'visibility-logic-elementor' ),
					'contain'                 => __( 'Contains', 'visibility-logic-elementor' ),
					'not_contain'             => __( 'Does not contain', 'visibility-logic-elementor' ),
					'is_between'              => __( 'Between', 'visibility-logic-elementor' ),
					'less_than'               => __( 'Less than', 'visibility-logic-elementor' ),
					'greater_than'            => __( 'Greater than', 'visibility-logic-elementor' ),
					'is_array'                => __( 'Is array', 'visibility-logic-elementor' ),
					'is_array_and_contains'   => __( 'Is array and contains', 'visibility-logic-elementor' ),
				],
				'default'        => 'none',
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
				],
				'style_transfer' => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_value',
			[
				'label'          => __( 'Condition Value', 'visibility-logic-elementor' ),
				'type'           => Controls_Manager::TEXTAREA,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
					self::SECTION_PREFIX . 'advanced_user_meta_status' => [
						'specific_value',
						'specific_value_multiple',
						'not_specific_value',
						'contain',
						'not_contain',
						'is_between',
						'less_than',
						'greater_than',
						'is_array_and_contains',
					],
				],
				'style_transfer' => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_value_2',
			[
				'label'          => __( 'Condition Value 2', 'visibility-logic-elementor' ),
				'type'           => Controls_Manager::TEXTAREA,
				'label_block'    => true,
				'condition'      => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
					self::SECTION_PREFIX . 'advanced_user_meta_status' => 'is_between',
				],
				'style_transfer' => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_notice_array',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'Type in comma separated strings.', 'visibility-logic-elementor' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
					self::SECTION_PREFIX . 'advanced_user_meta_status' => [
						'specific_value_multiple',
						'is_array_and_contains',
					],
				],
				'style_transfer'  => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_notice_numeric',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'The value of the selected meta and also the the values of the conditions must be numeric.', 'visibility-logic-elementor' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
					self::SECTION_PREFIX . 'advanced_user_meta_status' => [
						'is_between',
						'less_than',
						'greater_than',
					],
				],
				'style_transfer'  => false,
			]
		);

		$repeater->add_control(
			self::SECTION_PREFIX . 'advanced_user_meta_notice_none',
			[
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => __( 'Setting the condition to "None" will have no effect.', 'visibility-logic-elementor' ),
				'content_classes' => 'stax-generic-notice',
				'condition'       => [
					self::SECTION_PREFIX . 'advanced_user_meta_options!' => '',
					self::SECTION_PREFIX . 'advanced_user_meta_status' => [
						'none',
					],
				],
				'style_transfer'  => false,
			]
		);

		$element->add_control(
			self::SECTION_PREFIX . 'advanced_meta_conditions',
			[
				'label'          => __( 'Conditions', 'plugin-name' ),
				'type'           => \Elementor\Controls_Manager::REPEATER,
				'fields'         => $repeater->get_controls(),
				'default'        => [],
				'title_field'    => '{{{ ' . self::SECTION_PREFIX . 'advanced_user_meta_options' . ' }}}',
				'condition'      => [
					self::SECTION_PREFIX . 'advanced_user_meta_enabled' => 'yes',
				],
				'style_transfer' => false,
			]
		);
	}

	/**
	 * Apply conditions
	 *
	 * @param array                   $options
	 * @param array                   $settings
	 * @param \Elementor\Element_Base $item
	 *
	 * @return array
	 */
	public function apply_conditions( $options, $settings, $item ) {
		$settings = $item->get_settings_for_display();

		if ( ! isset( $settings[ self::SECTION_PREFIX . 'advanced_user_meta_enabled' ] ) ||
			! (bool) $settings[ self::SECTION_PREFIX . 'advanced_user_meta_enabled' ] ) {
			return $options;
		}

		$current_user     = wp_get_current_user();
		$meta_consistency = [];

		foreach ( $settings[ self::SECTION_PREFIX . 'advanced_meta_conditions' ] as $condition ) {
			$meta_check_type  = $condition[ self::SECTION_PREFIX . 'advanced_user_meta_status' ];
			$meta_check_value = $condition[ self::SECTION_PREFIX . 'advanced_user_meta_value' ];

			foreach ( $condition[ self::SECTION_PREFIX . 'advanced_user_meta_options' ] as $meta ) {
				$user_meta = get_user_meta( $current_user->ID, $meta, true );

				if ( isset( $current_user->{$meta} ) ) {
					$user_meta = $current_user->{$meta};
				} else {
					$user_meta = get_user_meta( $current_user->ID, $meta, true );
				}

				switch ( $meta_check_type ) {
					case 'empty':
						$meta_consistency[] = empty( $user_meta );
						break;
					case 'not_empty':
						$meta_consistency[] = ! empty( $user_meta );
						break;
					case 'specific_value':
						$meta_consistency[] = $user_meta == $meta_check_value;
						break;
					case 'specific_value_multiple':
						$values = explode( ',', $meta_check_value );

						$value_found = false;

						foreach ( $values as $item ) {
							if ( $item === $user_meta ) {
								$value_found = true;
								break;
							}
						}

						$meta_consistency[] = $value_found;
						break;
					case 'not_specific_value':
						$meta_consistency[] = $user_meta !== $meta_check_value;
						break;
					case 'contain':
						$meta_consistency[] = strpos( $user_meta, $meta_check_value ) !== false;
						break;
					case 'not_contain':
						$meta_consistency[] = strpos( $user_meta, $meta_check_value ) === false;
						break;
					case 'is_between':
						$meta_check_value_2 = $condition[ self::SECTION_PREFIX . 'advanced_user_meta_value_2' ];

						if ( ! is_numeric( $user_meta ) || ! is_numeric( $meta_check_value ) || ! is_numeric( $meta_check_value_2 ) ) {
							$meta_consistency[] = false;
						} else {
							$meta_consistency[] = (int) $meta_check_value < (int) $user_meta || (int) $user_meta < (int) $meta_check_value_2;
						}
						break;
					case 'less_than':
						if ( ! is_numeric( $user_meta ) || ! is_numeric( $meta_check_value ) ) {
							$meta_consistency[] = false;
						} else {
							$meta_consistency[] = (int) $user_meta < (int) $meta_check_value;
						}
						break;
					case 'greater_than':
						if ( ! is_numeric( $user_meta ) || ! is_numeric( $meta_check_value ) ) {
							$meta_consistency[] = false;
						} else {
							$meta_consistency[] = (int) $user_meta > (int) $meta_check_value;
						}
						break;
					case 'is_array':
						$meta_consistency[] = is_array( $user_meta );
						break;
					case 'is_array_and_contains':
						if ( ! is_array( $user_meta ) ) {
							$meta_consistency[] = false;
						} else {
							$values = explode( ',', $meta_check_value );

							$meta_consistency[] = ! empty( array_intersect( $user_meta, $values ) );
						}
						break;
					default:
				}
			}
		}

		if ( $settings[ self::SECTION_PREFIX . 'advanced_user_meta_rule' ] === 'and' ) {
			$rules = array_unique( $meta_consistency );

			if ( count( $rules ) !== 1 ) {
				$options['advanced_user_meta'] = false;
			} else {
				$options['advanced_user_meta'] = $rules[0];
			}
		} else {
			$any_true = false;

			foreach ( $meta_consistency as $item ) {
				if ( $item ) {
					$any_true = true;
					break;
				}
			}

			$options['advanced_user_meta'] = $any_true;
		}

		return $options;
	}

}

AdvancedUserMeta::instance();
