<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class FunctionCaller {

	use Wp_Trait;

	/**
	 * Get values for posts
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_posts( $request ) {
		$ids     = (array) $request['id'];
		$results = [];
		$is_ctp  = false;

		if ( ! empty( $ids ) ) {
			$first  = reset( $ids );
			$is_ctp = ! is_numeric( $first );
		}

		if ( $is_ctp ) {
			$post_types = self::get_post_types();
			if ( ! empty( $ids ) ) {
				foreach ( $ids as $aid ) {
					if ( isset( $post_types[ $aid ] ) ) {
						$results[ $aid ] = $post_types[ $aid ];
					}
				}
			}
		} else {
			$query = new \WP_Query(
				[
					'post_type'      => 'any',
					'post__in'       => $ids,
					'posts_per_page' => -1,
				]
			);

			foreach ( $query->posts as $post ) {
				$results[ $post->ID ] = $post->post_title;
			}
		}

		return $results;
	}

	/**
	 * Get values for terms
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_terms( $request ) {
		$ids     = (array) $request['id'];
		$results = [];
		$tid     = reset( $ids );

		if ( is_numeric( $tid ) ) {
			$query_params = [
				'include' => $ids,
			];
		} else {
			$query_params = [
				'slug' => $ids,
			];
		}

		$terms = get_terms( $query_params );

		foreach ( $terms as $term ) {
			$results[ $term->term_id ] = $term->name;
		}

		return $results;
	}

	/**
	 * Get posts
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_posts( $data ) {
		$results     = [];
		$object_type = ( ! empty( $data['object_type'] ) ) ? $data['object_type'] : 'any';

		if ( $object_type == 'type' ) {
			$list = self::get_post_types();
			if ( ! empty( $list ) ) {
				foreach ( $list as $akey => $alist ) {
					if ( strlen( $data['q'] ) > 2 ) {
						if ( strpos( $akey, $data['q'] ) === false && strpos( $alist, $data['q'] ) === false ) {
							continue;
						}
					}
					$results[] = [
						'id'   => $akey,
						'text' => $alist,
					];
				}
			}
		} else {
			$query_params = [
				'post_type'      => $object_type,
				's'              => $data['q'],
				'posts_per_page' => -1,
			];
			if ( 'attachment' === $query_params['post_type'] ) {
				$query_params['post_status'] = 'inherit';
			}
			$query = new \WP_Query( $query_params );
			foreach ( $query->posts as $post ) {
				$post_title = $post->post_title;
				if ( empty( $data['object_type'] ) || $object_type == 'any' ) {
					$post_title = '[' . $post->ID . '] ' . $post_title . ' (' . $post->post_type . ')';
				}
				if ( ! empty( $data['object_type'] ) && $object_type == 'elementor_library' ) {
					$etype      = get_post_meta( $post->ID, '_elementor_template_type', true );
					$post_title = '[' . $post->ID . '] ' . $post_title . ' (' . $post->post_type . ' > ' . $etype . ')';
				}

				$results[] = [
					'id'   => $post->ID,
					'text' => $post_title,
				];
			}
		}

		return $results;
	}

	/**
	 * Get post terms
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_terms( $data ) {
		$results    = [];
		$taxonomies = ( ! empty( $data['object_type'] ) ) ? $data['object_type'] : get_object_taxonomies( '' );

		$terms = get_terms(
			[
				'taxonomy'   => $taxonomies,
				'search'     => $data['q'],
				'hide_empty' => false,
			]
		);

		foreach ( $terms as $term ) {
			$term_name = $term->name;

			if ( empty( $data['object_type'] ) ) {
				$taxonomy  = get_taxonomy( $term->taxonomy );
				$term_name = $taxonomy->labels->singular_name . ': ' . $term_name;
			}

			$results[] = [
				'id'   => $term->term_id,
				'text' => $term_name,
			];
		}

		return $results;
	}

	/**
	 * Get values for products variations
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_products_variations( $request ) {
		$ids     = (array) $request['id'];
		$results = [];

		if ( empty( $ids ) ) {
			return $results;
		}

		$products = wc_get_products(
			[
				'post__in' => $ids,
				'status'   => 'publish',
				'orderby'  => 'menu_order',
				'order'    => 'asc',
				'limit'    => -1,
			]
		);

		foreach ( $products as $key => $product ) {
			if ( 'variable' === $product->get_type() ) {
				foreach ( $product->get_available_variations() as $variation ) {
					$results[ $variation['variation_id'] ] = strip_tags( wc_get_product( $variation['variation_id'] )->get_formatted_name() );
				}
			}
		}

		return $results;
	}

	/**
	 * Get products variations
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_products_variations( $data ) {
		$variations = [];

		$products = wc_get_products(
			[
				's'       => $data['q'],
				'status'  => 'publish',
				'orderby' => 'name',
				'order'   => 'asc',
				'limit'   => -1,
			]
		);

		foreach ( $products as $key => $product ) {
			if ( 'variable' === $product->get_type() ) {
				foreach ( $product->get_available_variations() as $variation ) {
					$variations[] = [
						'id'   => $variation['variation_id'],
						'text' => strip_tags( wc_get_product( $variation['variation_id'] )->get_formatted_name() ),
					];
				}
			}
		}

		return $variations;
	}

	/**
	 * Get values for products
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_products( $request ) {
		$ids = (array) $request['id'];

		if ( empty( $ids ) ) {
			return [];
		}

		return wp_list_pluck(
			get_posts(
				[
					'post__id'       => $ids,
					'status'         => 'publish',
					'post_type'      => 'product',
					'posts_per_page' => -1,
				]
			),
			'post_title',
			'ID'
		);
	}

	/**
	 * Get products
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_products( $data ) {
		$results = [];

		$products = get_posts(
			[
				's'              => $data['q'],
				'status'         => 'publish',
				'post_type'      => 'product',
				'posts_per_page' => -1,
			]
		);

		foreach ( $products as $key => $product ) {
			$results[] = [
				'id'   => $product->ID,
				'text' => $product->post_title,
			];
		}

		return $results;
	}

	/**
	 * Get values for subscriptions
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_subscriptions( $request ) {
		$ids = (array) $request['id'];

		if ( empty( $ids ) ) {
			return [];
		}

		return wp_list_pluck(
			get_posts(
				[
					'post__id'       => $ids,
					'post_type'      => 'product',
					'posts_per_page' => -1,
					'meta_query'     => [
						[
							'key'     => '_subscription_price',
							'compare' => 'EXISTS',
						],
					],
				]
			),
			'post_title',
			'ID'
		);
	}

	/**
	 * Get subscriptions
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_subscriptions( $data ) {
		$results = [];

		$subscriptions = get_posts(
			[
				's'              => $data['q'],
				'post_type'      => 'product',
				'posts_per_page' => -1,
				'meta_query'     => [
					[
						'key'     => '_subscription_price',
						'compare' => 'EXISTS',
					],
				],
			]
		);

		foreach ( $subscriptions as $key => $subscription ) {
			$results[] = [
				'id'   => $subscription->ID,
				'text' => $subscription->post_title,
			];
		}

		return $results;
	}

	/**
	 * Get values for geo location country
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_geo_location_country( $request ) {
		$ids = (array) $request['id'];

		if ( empty( $ids ) ) {
			return [];
		}

		$countries = @json_decode( file_get_contents( STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/data/countries.json' ), true );
		$results   = [];

		if ( ! is_array( $countries ) ) {
			return $results;
		}

		foreach ( $countries as $country ) {
			if ( in_array( $country['country_code'], $ids ) ) {
				$results[ $country['country_code'] ] = $country['country_name'];
			}
		}

		return $results;
	}

	/**
	 * Get geo location country
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_geo_location_country( $data ) {
		$results = [];

		$countries = @json_decode( file_get_contents( STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/data/countries.json' ), true );

		if ( ! is_array( $countries ) ) {
			return [];
		}

		foreach ( $countries as $country ) {
			if ( strpos( strtolower( $country['country_name'] ), strtolower( $data['q'] ) ) !== false ) {
				$results[] = [
					'id'   => $country['country_code'],
					'text' => $country['country_name'],
				];
			}
		}

		return $results;
	}

	/**
	 * Get values for geo location city
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_geo_location_city( $request ) {
		$ids = (array) $request['id'];

		if ( empty( $ids ) ) {
			return [];
		}

		$cities  = @json_decode( file_get_contents( STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/data/cities.json' ), true );
		$results = [];

		if ( ! is_array( $cities ) ) {
			return $results;
		}

		foreach ( $cities as $city ) {
			$hash = strtolower( $city['name'] ) . '|' . strtolower( $city['country'] );
			if ( in_array( $hash, $ids ) ) {
				$results[ $hash ] = $city['name'] . "({$city['country']})";
			}
		}

		return $results;
	}

	/**
	 * Get geo location city
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_geo_location_city( $data ) {
		$results = [];

		$cities = @json_decode( file_get_contents( STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/data/cities.json' ), true );

		if ( ! is_array( $cities ) ) {
			return [];
		}

		foreach ( $cities as $city ) {
			if ( strpos( strtolower( $city['name'] ), strtolower( $data['q'] ) ) !== false ) {
				$hash = strtolower( $city['name'] ) . '|' . strtolower( $city['country'] );

				$results[] = [
					'id'   => $hash,
					'text' => $city['name'] . "({$city['country']})",
				];
			}
		}

		return $results;
	}

	/**
	 * Get values for edd products
	 *
	 * @param [type] $request
	 * @return array
	 */
	public static function get_value_titles_for_edd_products( $request ) {
		$ids = (array) $request['id'];

		if ( empty( $ids ) ) {
			return [];
		}

		return wp_list_pluck(
			get_posts(
				[
					'post__id'       => $ids,
					'post_type'      => 'download',
					'posts_per_page' => -1,
				]
			),
			'post_title',
			'ID'
		);
	}

	/**
	 * Get edd products
	 *
	 * @param array $data
	 * @return array
	 */
	public static function get_edd_products( $data ) {
		$results = [];

		$products = get_posts(
			[
				's'              => $data['q'],
				'post_type'      => 'download',
				'posts_per_page' => -1,
			]
		);

		foreach ( $products as $key => $product ) {
			$results[] = [
				'id'   => $product->ID,
				'text' => $product->post_title,
			];
		}

		return $results;
	}

}
