<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Plugin
 */
class Plugin extends Singleton {

	const LICENSE_PRODUCT_ID    = 'visibility-logic';
	const LICENSE_SERVER_URL    = 'https://my.staxwp.com/';
	const LICENSE_DISPLAY_TITLE = 'Visibility Logic Pro';

	/**
	 * Minimum Visibility Logic free version
	 *
	 * @var string
	 */
	public $visibility_free_ver_required = '2.1.8';

	/**
	 * Plugin constructor
	 */
	public function __construct() {
		parent::__construct();

		$this->license_setup();

		add_action( 'plugins_loaded', [ $this, 'load_plugin' ], 12 );
	}

	/**
	 * Load plugin
	 *
	 * @return void
	 */
	public function load_plugin() {
		if ( ! did_action( 'stax/visibility/pre_init' ) ) {
			add_action( 'admin_notices', [ $this, 'fails_to_load' ] );

			return;
		}

		/* Required version */
		if ( ! version_compare( STAX_VISIBILITY_VERSION, $this->visibility_free_ver_required, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'fail_load_out_of_date' ] );

			return;
		}

		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'Helpers.php';
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/Date.php';
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/Parser.php';
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/Template.php';
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'admin/Settings.php';
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'admin/pages/OptionSettings.php';

		// Load active options.
		$disabled_options = get_option( '_stax_visibility_disabled_options', [] );

		foreach ( $this->get_pro_options() as $slug => $option ) {
			if ( ! isset( $disabled_options[ $slug ] ) && file_exists( $option['class'] ) ) {
				require_once $option['class'];
			}
		}

		if ( ! shortcode_exists( 'elementor-template' ) ) {
			add_shortcode( 'elementor-template', [ $this, 'register_shortcode' ] );
		}

		add_action( 'elementor/init', [ $this, 'load_elementor_modules' ] );
		add_action( 'elementor/dynamic_tags/register_tags', [ $this, 'register_dynamic_tags' ], 10, 1 );
		add_action( 'elementor/editor/after_enqueue_scripts', [ $this, 'register_context_menu' ] );
		add_filter( 'stax/visibility/options', [ $this, 'add_pro_options' ], 20 );

		// Support for Elementor Flex Container
		if ( version_compare( STAX_VISIBILITY_VERSION, '2.3.1', '<' ) ) {
			add_action(
				'elementor/element/container/section_layout/after_section_end',
				function( $element ) {
					$element->start_controls_section(
						self::SECTION_PREFIX . 'container_support_upgrade',
						[
							'tab'   => self::VISIBILITY_TAB,
							'label' => __( 'Update required', 'visibility-logic-elementor-pro' ),
						]
					);

					$element->add_control(
						self::SECTION_PREFIX . 'container_support_upgrade_notice',
						[
							'label'           => __( 'Important Notice', 'visibility-logic-elementor-pro' ),
							'type'            => \Elementor\Controls_Manager::RAW_HTML,
							'raw'             => __( 'Please update the free version of Visiblity Logic to version 2.3.1 or higher to enable support for Elementor Flex Container.', 'visibility-logic-elementor-pro' ),
							'content_classes' => 'stax-generic-notice',
						]
					);

					$element->end_controls_section();
				}
			);
		}
	}

	/**
	 * Load elementor modules
	 */
	public function load_elementor_modules() {
		// Load traits.
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/traits/Wp_Trait.php';

		// Load function caller.
		require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/FunctionCaller.php';
	}

	/**
	 * Register dynamic tags
	 *
	 * @param $dynamic_tags
	 * @return void
	 */
	public function register_dynamic_tags( $dynamic_tags ) {
		if ( \Stax\VisibilityLogic\Resources::instance()->is_plugin_installed( 'elementor-pro/elementor-pro.php' ) &&
			\Stax\VisibilityLogic\Resources::instance()->is_plugin_active( 'elementor-pro/elementor-pro.php' ) ) {
			require_once STAX_VISIBILITY_PRO_CORE_PATH . 'helpers/NumberPostsTag.php';

			$dynamic_tags->register_tag( NumberPostsTag::class );
		}
	}

	/**
	 * Register context menu
	 *
	 * @return void
	 */
	public function register_context_menu() {
		wp_enqueue_script(
			'elementor-stax-visibility-context-menu',
			STAX_VISIBILITY_PRO_BASE_URL . 'assets/js/context-menu.js',
			[],
			STAX_VISIBILITY_PRO_VERSION,
			false
		);
	}

	/**
	 * Get pro options
	 *
	 * @param boolean $require_check
	 * @return array
	 */
	public function get_pro_options( $require_check = true ) {
		$options = [
			'ip-referrer'        => [
				'name'  => __( 'IP & Referrer', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'IpReferrerVisibility.php',
			],
			'geo-location'       => [
				'name'    => __( 'Geo Location', 'visibility-logic-elementor-pro' ),
				'class'   => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'GeoLocationVisibility.php',
				'require' => [
					'free_version' => '2.3.1',
					'text'         => [
						'version' => sprintf( __( 'Geo Location option requires min version %1$s of Visibility Logic Free. Please update!', 'visibility-logic-elementor-pro' ), '2.3.1' ),
					],
				],
			],
			'advanced-user-meta' => [
				'name'  => __( 'Advanced User Meta', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'AdvancedUserMeta.php',
			],
			'post-page'          => [
				'name'  => __( 'Posts & Pages', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'PostVisibility.php',
			],
			'archive'            => [
				'name'  => __( 'Archive Pages', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'ArchiveVisibility.php',
			],
			'conditional-tags'   => [
				'name'  => __( 'Conditional Tags', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'ConditionalTagsVisibility.php',
			],
			'dynamic-conditions' => [
				'name'    => __( 'Dynamic Conditions', 'visibility-logic-elementor-pro' ),
				'class'   => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'DynamicConditionsVisibility.php',
				'require' => [
					'plugins' => [
						'elementor-pro/elementor-pro.php',
					],
					'text'    => [
						'plugins' => sprintf(
							esc_html__( 'This option requires %1$s installed and activated!', 'visibility-logic-elementor-pro' ),
							'<a href="https://staxwp.com/go/elementor-pro" target="_blank">Elementor Pro</a>'
						),
					],
				],

			],
			'woocommerce-users'  => [
				'name'    => __( 'WooCommerce Users', 'visibility-logic-elementor-pro' ),
				'class'   => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'WooCommerceUsersVisibility.php',
				'require' => [
					'free_version' => '2.2.6',
					'plugins'      => [
						'woocommerce/woocommerce.php',
					],
					'text'         => [
						'plugins' => sprintf(
							esc_html__( 'This option requires %1$s installed and activated!', 'visibility-logic-elementor-pro' ),
							'<a href="https://staxwp.com/go/woocommerce" target="_blank">WooCommerce</a>',
						),
						'version' => sprintf( __( 'WooCommerce Users requires min version %1$s of Visibility Logic Free. Please update!', 'visibility-logic-elementor-pro' ), '2.2.6' ),
					],
				],
			],
			'edd-users'          => [
				'name'    => __( 'EDD Users', 'visibility-logic-elementor-pro' ),
				'class'   => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'EDDUsersVisibility.php',
				'require' => [
					'free_version' => '2.2.6',
					'plugins'      => [
						'easy-digital-downloads/easy-digital-downloads.php',
					],
					'text'         => [
						'plugins' => sprintf(
							esc_html__( 'This option requires %1$s installed and activated!', 'visibility-logic-elementor-pro' ),
							'<a href="https://staxwp.com/go/easy-digital-downloads" target="_blank">Easy Digital Downloads</a>',
						),
						'version' => sprintf( __( 'EDD Users requires min version %1$s of Visibility Logic Free. Please update!', 'visibility-logic-elementor-pro' ), '2.2.6' ),
					],
				],
			],
			'fallback'           => [
				'name'  => __( 'Fallback', 'visibility-logic-elementor-pro' ),
				'class' => STAX_VISIBILITY_PRO_CORE_SETTINGS_PATH . 'FallbackVisibility.php',
			],
		];

		if ( ! $require_check ) {
			return $options;
		}

		foreach ( $options as &$option ) {
			if ( ! isset( $option['require'] ) ) {
				continue;
			}

			if ( ! isset( $option['require']['plugins'] ) || ! is_array( $option['require']['plugins'] ) || empty( $option['require']['plugins'] ) ) {
				unset( $option['require'] );

				continue;
			}

			if ( ! isset( $option['require']['text'] ) ) {
				unset( $option['require'] );

				continue;
			}

			$pass = true;

			foreach ( $option['require']['plugins'] as $plugin ) {
				if ( ! \Stax\VisibilityLogic\Resources::instance()->is_plugin_installed( $plugin ) ||
				! \Stax\VisibilityLogic\Resources::instance()->is_plugin_active( $plugin ) ) {
					$pass = false;
				}
			}

			if ( $pass ) {
				unset( $option['require'] );
			}
		}

		return $options;
	}

	/**
	 * Get option
	 *
	 * @param string $slug
	 * @return bool|array
	 */
	public function get_pro_option( $slug ) {
		$options = $this->get_pro_options( false );

		if ( isset( $options[ $slug ] ) ) {
			return $options[ $slug ];
		}

		return false;
	}

	/**
	 * Add pro options to admin list
	 *
	 * @param array $options
	 * @return array
	 */
	public function add_pro_options( $options ) {
		$pro_options = $this->get_pro_options();

		foreach ( $pro_options as $slug => $pro_option ) {
			unset( $pro_options[ $slug ]['class'] );
		}

		return array_merge( $options, $pro_options );
	}

	/**
	 * Register shortcode.
	 *
	 * @param array $atts The shortcode attributes.
	 *
	 * @return mixed Content.
	 * @since 1.0.0
	 * @access public
	 */
	public function register_shortcode( $atts ) {
		if ( empty( $atts['id'] ) ) {
			return '';
		}

		$content = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display( $atts['id'] );

		return $content;
	}

	/**
	 * License Setup
	 *
	 * @return void
	 * @since 1.1.16 Updated API manager library.
	 */
	public function license_setup() {

		// Load WC_AM_Client class if it exists.
		include_once STAX_VISIBILITY_PRO_CORE_PATH . 'ApiManager.php';

		// Instantiate WC_AM_Client class object if the WC_AM_Client class is loaded.
		if ( class_exists( ApiManager::class ) ) {

			/**
			 * This file is only an example that includes a plugin header, and this code used to instantiate the client object. The variable $wcam_lib
			 * can be used to access the public properties from the WC_AM_Client class, but $wcam_lib must have a unique name. To find data saved by
			 * the WC_AM_Client in the options table, search for wc_am_client_{product_id}, so in this example it would be wc_am_client_13.
			 *
			 * All data here is sent to the WooCommerce API Manager API, except for the $software_title, which is used as a title, and menu label, for
			 * the API Key activation form the client will see.
			 *
			 * ****
			 * NOTE
			 * ****
			 * If $product_id is empty, the customer can manually enter the product_id into a form field on the activation screen.
			 *
			 * @param string $file Must be __FILE__ from the root plugin file, or theme functions, file locations.
			 * @param int $product_id Must match the Product ID number (integer) in the product.
			 * @param string $software_version This product's current software version.
			 * @param string $plugin_or_theme 'plugin' or 'theme'
			 * @param string $api_url The URL to the site that is running the API Manager. Example: https://www.toddlahman.com/
			 * @param string $software_title The name, or title, of the product. The title is not sent to the API Manager APIs, but is used for menu titles.
			 *
			 * Example:
			 *
			 * $wcam_lib = new WC_AM_Client_25( $file, $product_id, $software_version, $plugin_or_theme, $api_url, $software_title );
			 */

			// Example of empty string product_id.

			global $wcam_lib_stax_visibility;
			$wcam_lib_stax_visibility = new ApiManager( STAX_VISIBILITY_PRO_FILE, self::LICENSE_PRODUCT_ID, STAX_VISIBILITY_PRO_VERSION, 'plugin', self::LICENSE_SERVER_URL, self::LICENSE_DISPLAY_TITLE, 'visibility-logic-elementor-pro' );
		}
	}

	/**
	 * Check if license is active
	 *
	 * @return boolean
	 */
	public function is_license_active() {
		global $wcam_lib_stax_visibility;
		return $wcam_lib_stax_visibility->get_api_key_status();
	}

	/**
	 * Fires admin notice when Visibility Logic is not installed and activated.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function fails_to_load() {
		$screen = get_current_screen();

		if ( isset( $screen->parent_file ) && 'plugins.php' === $screen->parent_file && 'update' === $screen->id ) {
			return;
		}

		$plugin = 'visibility-logic-elementor/conditional.php';
		if ( $this->is_plugin_installed( $plugin ) ) {
			if ( ! current_user_can( 'activate_plugins' ) ) {
				return;
			}

			$activation_url = wp_nonce_url( 'plugins.php?action=activate&amp;plugin=' . $plugin . '&amp;plugin_status=all&amp;paged=1&amp;s', 'activate-plugin_' . $plugin );
			/* translators: %s: html tags */
			$message  = '<p>' . sprintf( __( '%1$s Visibility Logic Pro %2$s plugin requires %1$s Visibility Logic %2$s plugin to be activated.', 'visibility-logic-elementor-pro' ), '<strong>', '</strong>' ) . '</p>';
			$message .= '<p>' . sprintf( '<a href="%s" class="button-primary">%s</a>', $activation_url, __( 'Activate Visibility Logic', 'visibility-logic-elementor-pro' ) ) . '</p>';
		} else {
			if ( ! current_user_can( 'install_plugins' ) ) {
				return;
			}

			$install_url = wp_nonce_url( self_admin_url( 'update.php?action=install-plugin&plugin=visibility-logic-elementor' ), 'install-plugin_visibility-logic-elementor' );
			/* translators: %s: html tags */
			$message  = '<p>' . sprintf( __( '%1$s Visibility Logic Pro %2$s plugin requires %1$s Visibility Logic %2$s plugin to be installed.', 'visibility-logic-elementor-pro' ), '<strong>', '</strong>' ) . '</p>';
			$message .= '<p>' . sprintf( '<a href="%s" class="button-primary">%s</a>', $install_url, __( 'Install Visibility Logic', 'visibility-logic-elementor-pro' ) ) . '</p>';
		}
		echo '<div class="error"><p>' . $message . '</p></div>';
	}

	/**
	 * Check if plugin is installed
	 *
	 * @return boolean
	 */
	public function is_plugin_installed( $plugin ) {
		$plugins = get_plugins();

		return isset( $plugins[ $plugin ] );
	}

	/**
	 * Version compare notice
	 *
	 * @return void
	 */
	public function fail_load_out_of_date() {
		$message = '<p>' . sprintf( __( 'Visibility Logic Pro requires Visibility Logic base plugin to be updated to at least version %1$s', 'visibility-logic-elementor-pro' ), $this->visibility_free_ver_required ) . '</p>';

		echo '<div class="error">' . $message . '</div>';
	}

}

Plugin::instance();
