<?php

namespace Stax\VisibilityLogicPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Helpers
 */
class Helpers {

	/**
	 * @var null
	 */
	protected static $instances;

	/**
	 * @return Helpers|null
	 */
	public static function instance() {
		$class = static::class;

		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Get visitor ip
	 *
	 * @return string
	 */
	public function get_visitor_ip() {
		if ( isset( $_SERVER['HTTP_CF_CONNECTING_IP'] ) ) {
			$_SERVER['REMOTE_ADDR']    = $_SERVER['HTTP_CF_CONNECTING_IP'];
			$_SERVER['HTTP_CLIENT_IP'] = $_SERVER['HTTP_CF_CONNECTING_IP'];
		}

		$client  = @$_SERVER['HTTP_CLIENT_IP'];
		$forward = @$_SERVER['HTTP_X_FORWARDED_FOR'];
		$remote  = $_SERVER['REMOTE_ADDR'];

		if ( filter_var( $client, FILTER_VALIDATE_IP ) ) {
			$ip = $client;
		} elseif ( filter_var( $forward, FILTER_VALIDATE_IP ) ) {
			$ip = $forward;
		} else {
			$ip = $remote;
		}

		if ( $ip === '127.0.0.1' ) {
			$ip = '8.8.8.8';
		}

		return $ip;
	}

	/**
	 * Get taxonomies
	 *
	 * @param boolean $dynamic
	 * @return array
	 */
	public static function get_taxonomies( $dynamic = false ) {
		$taxonomies = get_taxonomies( [], 'objects', 'and' );
		$result     = [];
		$result[''] = 'None';

		if ( $dynamic ) {
			$result['dynamic'] = 'Dynamic';
		}

		$result['category'] = 'Categories posts (category)';
		$result['post_tag'] = 'Tags posts (post_tag)';

		if ( $taxonomies ) {
			foreach ( $taxonomies as $taxonomy ) {
				$result[ $taxonomy->name ] = $taxonomy->label . ' (' . $taxonomy->name . ')';
			}
		}

		return $result;
	}

	/**
	 * Get taxonomy terms
	 *
	 * @param string  $taxonomy
	 * @param boolean $flat
	 * @return array
	 */
	public static function get_taxonomy_terms( $taxonomy = null, $flat = false ) {
		$result      = [];
		$flat_result = [];
		$result['']  = 'None';

		if ( $taxonomy ) {
			$terms = get_terms( $taxonomy );
			if ( ! empty( $terms ) ) {
				foreach ( $terms as $term ) {
					$result[ $term->term_id ] = $term->name . ' (' . $term->slug . ')';
				}
			}
		} else {
			$taxonomies = self::get_taxonomies();
			foreach ( $taxonomies as $tax_key => $tax_item ) {
				if ( $tax_key ) {
					$terms = get_terms( $tax_key );
					if ( ! empty( $terms ) ) {
						$tmp          = [];
						$tmp['label'] = $tax_item;

						foreach ( $terms as $term ) {
							$tmp['options'][ $term->term_id ] = $term->name . ' (' . $term->slug . ')';
							$flat_result[ $term->term_id ]    = $tax_item . ' > ' . $term->name . ' (' . $term->slug . ')';
						}

						$result[] = $tmp;
					}
				}
			}
		}

		if ( $flat ) {
			return $flat_result;
		}

		return $result;
	}

	/**
	 * Get post types
	 *
	 * @param boolean $exclude
	 * @return array
	 */
	public static function get_post_types( $exclude = true ) {
		$args = [
			'public' => true,
		];

		$post_types = get_post_types( $args );

		if ( $exclude ) {
			$post_types = array_diff( $post_types, [ 'attachment', 'elementor_library', 'oceanwp_library', 'page' ] );
		}

		foreach ( $post_types as $akey => $acpt ) {
			$cpt                 = get_post_type_object( $acpt );
			$post_types[ $akey ] = $cpt->label;
		}

		return $post_types;
	}

	/**
	 * Get all posts
	 *
	 * @param [type]  $myself
	 * @param boolean $group
	 * @param string  $order_by
	 * @return array
	 */
	public static function get_all_posts( $myself = null, $group = false, $order_by = 'title' ) {
		$args = [
			'public' => true,
		];

		$post_type_all = get_post_types( $args, 'names', 'and' );

		$type_excluded    = [ 'elementor_library', 'oceanwp_library', 'ae_global_templates' ];
		$types_registered = array_diff( $post_type_all, $type_excluded );

		$templates[0] = 'None';

		$exclude_io = [];

		if ( isset( $myself ) && $myself ) {
			$exclude_io = [ $myself ];
		}

		$get_templates = get_posts(
			[
				'post_type'    => $types_registered,
				'numberposts'  => -1,
				'post__not_in' => $exclude_io,
				'post_status'  => 'publish',
				'orderby'      => $order_by,
				'order'        => 'DESC',
			]
		);

		if ( ! empty( $get_templates ) ) {
			foreach ( $get_templates as $template ) {

				if ( $group ) {
					$templates[ $template->post_type ]['options'][ $template->ID ] = $template->post_title;
					$templates[ $template->post_type ]['label']                    = $template->post_type;
				} else {
					$templates[ $template->ID ] = $template->post_title;
				}
			}
		}

		return $templates;
	}


	/**
	 * Get elementor templates
	 *
	 * @return array
	 */
	public static function get_templates() {
		return \Elementor\Plugin::instance()->templates_manager->get_source( 'local' )->get_items(
			[
				'type' => [ 'section', 'archive', 'page', 'single' ],
			]
		);
	}

	/**
	 * Get all elementor templates
	 *
	 * @return array
	 */
	public static function get_all_template() {
		$templates    = [];
		$templates[0] = __( 'None', 'visibility-logic-elementor-pro' );

		$get_templates = self::get_templates();
		if ( ! empty( $get_templates ) ) {
			foreach ( $get_templates as $template ) {
				$templates[ $template['template_id'] ] = $template['title'] . ' (' . $template['type'] . ')';
			}
		}

		return $templates;
	}

}
