<?php

namespace WordpressModelsPlugin\DependencyInjection\FederatedPages;

use Symfony\Component\DependencyInjection\Attribute\Autowire;
use WordpressModels\Assets;
use WordpressModels\DependencyInjection\HookAttributes\Attributes\Action;
use WordpressModels\DependencyInjection\Pages\PageRegistry;
use WordpressModels\Page\AsyncPageData;

class FederatedPluginApp
{

    /**
     * Data to be passed to the federated plugin app as wp_localize_script data.
     *
     * @var array
     */
    private array $fpaData;

    public function __construct(#[Autowire(service: 'assets.wordpress-models-plugin')]
                                private Assets       $assets,
                                private PageRegistry $pageRegistry)
    {
        $this->fpaData = [
            'asyncDataNonce' => wp_create_nonce(AsyncPageData::NONCE_ID),
        ];
    }

    public function addData(string $key, mixed $value)
    {
        $this->fpaData[$key] = $value;
    }

    /**
     * Add an anchor to the footer to be used in React to render portals onto.
     *
     * @return void
     */
    #[Action('admin_footer', admin: true)]
    public function renderPortalAnchor(): void
    {
        echo '<div id="portal-anchor" class="fpa-page" style="position: absolute; top: -32px"></div>';
    }


    /**
     * Replace script enqueues on page load hooks with a custom hook.
     *
     * @return void
     */
    #[Action('admin_menu', 11, admin: true)]
    public function enqueueIprOnPageLoadHook(): void
    {
        $loadHooks = $this->pageRegistry->getLoadHooks();
        foreach ($this->pageRegistry->getPages() as $page) {
            if ($loadHook = $loadHooks[$page->getPageId()] ?? false) {
                remove_action("load-$loadHook", [$page, 'enqueueScripts']);
                // replace the load hook with our own, to prepare the page load
                add_action("load-$loadHook", fn() => $this->preparePageLoad($loadHook));
            }
        }
    }

    public function hideAdminNotices()
    {
        // capture notices in an output buffer, and thow them away
        add_action('admin_notices', fn() => ob_start(), PHP_INT_MIN);
        add_action('admin_notices', fn() => ob_end_clean(), PHP_INT_MAX);
    }

    /**
     * Set up the enqueues for the federated plugin app.
     *
     * @return void
     */
    public function preparePageLoad(string $loadHook): void
    {
        add_action('in_admin_header', [$this, 'hideAdminNotices']);

        add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
        add_action('admin_print_scripts', [$this, 'renderPreloadLinks']);

        /**
         * localize the page data
         * we place this on the load hook, so that the header is already rendered, and
         * we can inject data, such as admin notices
         * we register this action during load-$loadHook, so that {@see _wp_menu_output} doesn't append the
         * full url als query parameter
         */
        add_action($loadHook, [$this, 'localizePageData']);
    }

    /**
     * Enqueue the scripts for the federated plugin app.
     *
     * @return void
     */
    public function enqueueScripts(): void
    {
        $this->assets->enqueueCompiledScript('ext-react-router');
        $this->assets->enqueueCompiledScript('fpa');
    }

    public function localizePageData()
    {
        $this->addData('scriptEntries', $this->getAllPagesScripts());

        wp_localize_script(
            'fpa',
            'fpa',
            apply_filters('wpm_fpa_localize_data', $this->fpaData)
        );
    }

    /**
     * Render preload links for the script entries.
     *
     * This optimizes load times, as the preloads are loaded before the scripts are requested in the browser.
     * This prevents the browser from have to first load the main script, and then request the chunks.
     *
     * @return void
     */
    public function renderPreloadLinks()
    {
        foreach ($this->getAllPagesScripts() as $script) {
            echo "<link rel='modulepreload' href='$script'>";
        }
    }

    /**
     * Get all valid page scripts urls, indexed by page id.
     *
     * @return array
     */
    public function getAllPagesScripts(): array
    {
        $pages = $this->pageRegistry->getPages();
        $scripts = [];
        foreach ($pages as $page) {
            if (!$url = $page->getScriptUrl()) {
                continue;
            }
            $scripts[$page->getPageId()] = $url;
        }
        return $scripts;
    }

}
