<?php

namespace WordpressModels;

use Symfony\Component\Config\ConfigCache;
use Symfony\Component\Config\ConfigCacheInterface;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Container;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Dumper\PhpDumper;
use Symfony\Component\DependencyInjection\Loader\PhpFileLoader;
use WordpressModels\DependencyInjection\CliCommandAttributes\WpCliCommandAttributesCompilerPass;
use WordpressModels\DependencyInjection\CliCommandAttributes\WpCliCommandAttributesExtension;
use WordpressModels\DependencyInjection\HookAttributes\ContainerHookRegistry;
use WordpressModels\DependencyInjection\HookAttributes\HookAttributesCompilerPass;
use WordpressModels\DependencyInjection\HookAttributes\HookAttributesExtension;
use WordpressModels\DependencyInjection\Initializer\InitializerCompilerPass;
use WordpressModels\DependencyInjection\Initializer\InitializerExtension;
use WordpressModels\DependencyInjection\Initializer\ServiceInitializer;
use WordpressModels\DependencyInjection\RestRouteAttributes\RestRouteAttributesExtension;
use WordpressModels\DependencyInjection\RestRouteAttributes\RestRouteCompilerPass;

final class WordpressPluginContainers
{

    const CONTAINER_CACHE_DIR = WPM_PLUGIN_DIR . '/cache';
    const CONTAINER_FILE = self::CONTAINER_CACHE_DIR . '/container.php';
    const WPM_CONTAINER_CLASSNAME = 'WpmContainer';

    private Container $container;
    private ConfigCacheInterface $configCache;

    public function __construct()
    {
        $this->configCache = new ConfigCache(self::CONTAINER_FILE, WP_DEBUG);

        if (!$this->configCache->isFresh()) {
            $this->container = new ContainerBuilder();

            // register the services
            add_action('register_services', [$this, 'registerServices']);

            // compile the container
            add_action('plugins_loaded', [$this, 'compileContainer']);
        } else {
            // load the compiled container
            require_once self::CONTAINER_FILE;
            $class = self::WPM_CONTAINER_CLASSNAME;
            /** @var \WpmContainer container */
            $this->container = new $class();

            add_action('plugins_loaded', [$this, 'initializeServices']);
        }

        add_action('init_services', [$this, 'executeServiceInitialization']);
    }

    public function initializeServices(): void
    {
        // initialize the services
        do_action('init_services', $this->container);
    }

    public function getContainer(): Container
    {
        return $this->container;
    }

    /**
     * Run the container initialization.
     *
     * Firstly, for all active plugins, create a plugin container and register it to the global container.
     * Then, we allow plugins to register their own services to the global container.
     * Finally, we compile the global container to finalize it.
     *
     * @return void
     * @throws \Exception
     */
    public function compileContainer()
    {
        /** @var ContainerBuilder $container */
        $container = $this->container;

        do_action('register_services', $container);

        $compilerPasses = apply_filters('container_compiler_passes', [
            new InitializerCompilerPass(),
            new HookAttributesCompilerPass(),
            new WpCliCommandAttributesCompilerPass(),
            new RestRouteCompilerPass()
        ]);
        foreach ($compilerPasses as $compilerPass) {
            $container->addCompilerPass($compilerPass);
        }

        $extensions = apply_filters('container_extensions', [
            new InitializerExtension(),
            new HookAttributesExtension(),
            new WpCliCommandAttributesExtension(),
            new RestRouteAttributesExtension()
        ]);
        foreach ($extensions as $extension) {
            $this->container->registerExtension($extension);
            $this->container->loadFromExtension($extension->getAlias(), []);
        }

        // compile to finalize the global container
        $container->compile();

        $dumper = new PhpDumper($container);
        if (!is_dir(self::CONTAINER_CACHE_DIR)) {
            mkdir(self::CONTAINER_CACHE_DIR);
        }
        try {
            $this->configCache->write(
                $dumper->dump(['class' => self::WPM_CONTAINER_CLASSNAME]),
                $container->getResources());
        } catch (\RuntimeException $e) {
            throw new \Exception('Could not write to the cache directory');
        }

        // initialize the services
        $this->initializeServices();
    }

    /**
     * Initialize services before wordpress is fully loaded.
     *
     * @return void
     * @throws \Exception
     */
    public function executeServiceInitialization()
    {
        /** @var ContainerHookRegistry $service */
        $service = $this->container->get(ContainerHookRegistry::class);
        $service->registerHooks();

        /** @var ServiceInitializer $initializerService */
        $initializerService = $this->container->get(ServiceInitializer::class);
        $initializerService->initializeServices();
    }

    public function registerServices(ContainerBuilder $container)
    {

        // glob all must use and plugin config directories
        $mustUseConfigs = glob(WPMU_PLUGIN_DIR . '/*/config', GLOB_ONLYDIR);
        $pluginConfigs = glob(WP_PLUGIN_DIR . '/*/config', GLOB_ONLYDIR);

        $additionalConfigs = apply_filters('container_config_directories', []);

        // realpath to resolve any symlinks, and array_unique to remove duplicates
        // this array is in-order, so that the wordpress-models config is loaded first
        $allDirs = array_unique(array_map('realpath', [
            __DIR__ . '/../config',
            ...$mustUseConfigs,
            ...$pluginConfigs,
            ...$additionalConfigs
        ]));

        // load services from config files
        $loader = new PhpFileLoader($this->container, new FileLocator($allDirs));
        foreach ($allDirs as $configDir) {
            if (!file_exists($configDir . '/services.php')) {
                continue;
            }
            $loader->load($configDir . '/services.php');
        }

        $container->setDefinition('wpdb', new Definition(\wpdb::class))
            ->setFactory([self::class, "getWpdb"])
            ->setPublic(true);
    }

    public static function getWpdb()
    {
        global $wpdb;
        return $wpdb;
    }
}