<?php

namespace WordpressModels\DependencyInjection\HookAttributes;

use WordpressModels\DependencyInjection\HookAttributes\Attributes\Action;
use WordpressModels\DependencyInjection\Initializer\Attributes\ForceInitialize;

/**
 * Registry for hooks.
 *
 * At compile time, the hooks added to this registry. The {@see ContainerHookRegistry::registerHooks()} method
 * is called on 'plugins_loaded' to register the hooks.
 */
#[ForceInitialize]
class ContainerHookRegistry
{

    /**
     * @var array{
     *     type: string,
     *     hook: string,
     *     callback: string,
     *     priority: int,
     *     accepted_args: int,
     *     when: string|null,
     *     admin: bool,
     * }[]
     */
    private array $hooks = [];

    public function __construct()
    {
    }

    public function registerHooks()
    {
        foreach ($this->hooks as $action) {
            if (isset($action['when']) && (bool)$action['when']) {
                add_action($action['when'], function () use ($action) {
                    $this->registerHook($action);
                });
                continue;
            }

            $this->registerHook($action);
        }
    }

    #[Action('before_reset_container', 10)]
    public function deregisterHooks()
    {
        foreach ($this->hooks as $action) {
            call_user_func('remove_' . $action['type'],
                $action['hook'],
                $action['callback'],
                $action['priority']);
        }
    }

    public function addHook(string   $id,
                            string   $type,
                            callable $callback,
                            string   $hook,
                            int      $priority = 10,
                            int      $accepted_args = 1,
                            ?string  $when = null,
                            bool     $admin = false,
                            array    $injectServices = []
    )
    {
        $this->hooks[] = [
            'type' => $type,
            'hook' => $hook,
            'callback' => $callback,
            'priority' => $priority,
            'accepted_args' => $accepted_args,
            'when' => $when,
            'admin' => $admin,
            'injectServices' => $injectServices,
        ];
    }

    public function addAjaxHook(string   $action,
                                bool     $private,
                                bool     $public,
                                int      $priority,
                                ?string  $nonceId,
                                string   $nonceKey,
                                callable $handler)
    {
        if ($private) {
            $this->hooks[] = [
                'type' => 'action',
                'hook' => "wp_ajax_$action",
                'callback' => new AjaxActionWrapper($handler, $nonceId, $nonceKey),
                'priority' => $priority,
                'accepted_args' => 0
            ];
        }

        if ($public) {
            $this->hooks[] = [
                'type' => 'action',
                'hook' => "wp_ajax_nopriv_$action",
                'callback' => new AjaxActionWrapper($handler, $nonceId, $nonceKey),
                'priority' => $priority,
                'accepted_args' => 0
            ];
        }
    }

    /**
     * @param array $action
     * @return void
     */
    public function registerHook(array $action): void
    {
        $callback = $action['injectServices'] ? $this->wrapCallback($action['callback'], $action['injectServices']) : $action['callback'];

        call_user_func('add_' . $action['type'],
            $action['hook'],
            $callback,
            $action['priority'],
            $action['accepted_args']);
    }


    /**
     * Create a wrapper for the callback that injects the services.
     *
     * @param callable $callback -- original callback
     * @param array $injectServices -- in-order services to inject
     * @return \Closure
     */
    private function wrapCallback(callable $callback, array $injectServices)
    {
        return function (...$args) use ($callback, $injectServices) {
            $args = array_merge($args, $injectServices);
            return call_user_func($callback, ...$args);
        };
    }

}