<?php
/**
 * @license proprietary
 *
 * Modified by Beau Fiechter on 14-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\WordpressModels\DependencyInjection\Metabox;

use WRCE\Dependencies\Symfony\Component\DependencyInjection\ChildDefinition;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\ContainerBuilder;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\ContainerInterface;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Extension\Extension;
use WRCE\Dependencies\WordpressModels\DependencyInjection\Metabox\Attributes\AsMetabox;
use WRCE\Dependencies\WordpressModels\PostMetaBox;

class MetaboxExtension extends Extension
{

    private ContainerInterface $container;

    public function load(array $configs, ContainerBuilder $container)
    {
        $this->container = $container;
        $container->registerAttributeForAutoconfiguration(AsMetabox::class, self::tagMetabox(...));

        // add the registry
        $container->register(MetaboxRegistry::class)
            ->setPublic(true)
            ->setAutowired(true)
            ->setAutoconfigured(true);
    }

    /**
     * Add the 'metabox' tag to the definition.
     *
     * Sets the definition as public and lazy, and adds the 'metabox' tag to the definition.
     *
     * Also validates that the attributed class implements {@see PostMetaBoxInterface} and is not a subclass of {@see PostMetaBox}.
     *
     * Auto-injects the metabox ID and postType from the attributed values.
     *
     * @param ChildDefinition $definition
     * @param AsMetabox $metabox
     * @param \ReflectionClass $reflection
     * @return void
     */
    public static function tagMetabox(ChildDefinition $definition, AsMetabox $metabox, \ReflectionClass $reflection): void
    {
        if ($reflection->isSubclassOf(PostMetaBox::class)) {
            throw new \InvalidArgumentException(sprintf('"%s" is used as a base class. This class is intended for use without the Service Container and should not be tagged with #[%s], use "%s" instead',
                $reflection->getName(), AsMetabox::class, AbstractMetabox::class));
        }

        if (!$reflection->isSubclassOf(AbstractMetabox::class)) {
            throw new \InvalidArgumentException(sprintf('The class "%s" must be a subclass of "%s"',
                $reflection->getName(), AbstractMetabox::class));
        }

        // set the definition as public and lazy
        $definition->setPublic(true);
        $definition->setLazy(true);

        // inject id and postType with the attributed values via
        $autoInjectProps = [
            'id' => 'withId',
            'postType' => 'withPostType'
        ];
        foreach ($autoInjectProps as $param => $method) {
            $values = [$metabox->{$param}];
            // use immutable method calls (true) to avoid overwriting previous method calls
            $definition->addMethodCall($method, $values, true);
        }

        // add the tag
        $definition->addTag('metabox', [
            'id' => $metabox->id,
            'title' => $metabox->title,
            'screen' => $metabox->postType,
            'context' => $metabox->context,
            'priority' => $metabox->priority,
        ]);
    }
}