<?php
/**
 * @license MIT
 *
 * Modified by Beau Fiechter on 14-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace WRCE\Dependencies\Doctrine\DBAL\Driver\API\SQLSrv;

use WRCE\Dependencies\Doctrine\DBAL\Driver\API\ExceptionConverter as ExceptionConverterInterface;
use WRCE\Dependencies\Doctrine\DBAL\Driver\Exception;
use WRCE\Dependencies\Doctrine\DBAL\Exception\ConnectionException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\DatabaseObjectNotFoundException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\DriverException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\ForeignKeyConstraintViolationException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\InvalidFieldNameException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\NonUniqueFieldNameException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\NotNullConstraintViolationException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\SyntaxErrorException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\TableExistsException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\TableNotFoundException;
use WRCE\Dependencies\Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use WRCE\Dependencies\Doctrine\DBAL\Query;

/**
 * @internal
 *
 * @link https://docs.microsoft.com/en-us/sql/relational-databases/errors-events/database-engine-events-and-errors
 */
final class ExceptionConverter implements ExceptionConverterInterface
{
    public function convert(Exception $exception, ?Query $query): DriverException
    {
        switch ($exception->getCode()) {
            case 102:
                return new SyntaxErrorException($exception, $query);

            case 207:
                return new InvalidFieldNameException($exception, $query);

            case 208:
                return new TableNotFoundException($exception, $query);

            case 209:
                return new NonUniqueFieldNameException($exception, $query);

            case 515:
                return new NotNullConstraintViolationException($exception, $query);

            case 547:
            case 4712:
                return new ForeignKeyConstraintViolationException($exception, $query);

            case 2601:
            case 2627:
                return new UniqueConstraintViolationException($exception, $query);

            case 2714:
                return new TableExistsException($exception, $query);

            case 3701:
            case 15151:
                return new DatabaseObjectNotFoundException($exception, $query);

            case 11001:
            case 18456:
                return new ConnectionException($exception, $query);
        }

        return new DriverException($exception, $query);
    }
}
