<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * Modified by Beau Fiechter on 22-July-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\Symfony\Component\Serializer\DependencyInjection;

use WRCE\Dependencies\Symfony\Component\DependencyInjection\Argument\BoundArgument;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Compiler\PriorityTaggedServiceTrait;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\ContainerBuilder;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Exception\RuntimeException;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Reference;
use WRCE\Dependencies\Symfony\Component\Serializer\Debug\TraceableEncoder;
use WRCE\Dependencies\Symfony\Component\Serializer\Debug\TraceableNormalizer;

/**
 * Adds all services with the tags "serializer.encoder" and "serializer.normalizer" as
 * encoders and normalizers to the "serializer" service.
 *
 * @author Javier Lopez <f12loalf@gmail.com>
 * @author Robin Chalas <robin.chalas@gmail.com>
 */
class SerializerPass implements CompilerPassInterface
{
    use PriorityTaggedServiceTrait;

    /**
     * @return void
     */
    public function process(ContainerBuilder $container)
    {
        if (!$container->hasDefinition('serializer')) {
            return;
        }

        if (!$normalizers = $this->findAndSortTaggedServices('serializer.normalizer', $container)) {
            throw new RuntimeException('You must tag at least one service as "serializer.normalizer" to use the "serializer" service.');
        }

        if (!$encoders = $this->findAndSortTaggedServices('serializer.encoder', $container)) {
            throw new RuntimeException('You must tag at least one service as "serializer.encoder" to use the "serializer" service.');
        }

        if ($container->hasParameter('serializer.default_context')) {
            $defaultContext = $container->getParameter('serializer.default_context');
            foreach (array_merge($normalizers, $encoders) as $service) {
                $definition = $container->getDefinition($service);
                $definition->setBindings(['array $defaultContext' => new BoundArgument($defaultContext, false)] + $definition->getBindings());
            }

            $container->getParameterBag()->remove('serializer.default_context');
        }

        if ($container->getParameter('kernel.debug') && $container->hasDefinition('serializer.data_collector')) {
            foreach ($normalizers as $i => $normalizer) {
                $normalizers[$i] = $container->register('.debug.serializer.normalizer.'.$normalizer, TraceableNormalizer::class)
                    ->setArguments([$normalizer, new Reference('serializer.data_collector')]);
            }

            foreach ($encoders as $i => $encoder) {
                $encoders[$i] = $container->register('.debug.serializer.encoder.'.$encoder, TraceableEncoder::class)
                    ->setArguments([$encoder, new Reference('serializer.data_collector')]);
            }
        }

        $serializerDefinition = $container->getDefinition('serializer');
        $serializerDefinition->replaceArgument(0, $normalizers);
        $serializerDefinition->replaceArgument(1, $encoders);
    }
}
