<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * Modified by Beau Fiechter on 22-July-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\Symfony\Bundle\FrameworkBundle\Command;

use WRCE\Dependencies\Symfony\Component\Console\Attribute\AsCommand;
use WRCE\Dependencies\Symfony\Component\Console\Command\Command;
use WRCE\Dependencies\Symfony\Component\Console\Input\InputInterface;
use WRCE\Dependencies\Symfony\Component\Console\Input\InputOption;
use WRCE\Dependencies\Symfony\Component\Console\Output\OutputInterface;
use WRCE\Dependencies\Symfony\Component\Console\Style\SymfonyStyle;
use WRCE\Dependencies\Symfony\Component\DependencyInjection\Dumper\Preloader;
use WRCE\Dependencies\Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerAggregate;
use WRCE\Dependencies\Symfony\Component\HttpKernel\CacheWarmer\WarmableInterface;

/**
 * Warmup the cache.
 *
 * @author Fabien Potencier <fabien@symfony.com>
 *
 * @final
 */
#[AsCommand(name: 'cache:warmup', description: 'Warm up an empty cache')]
class CacheWarmupCommand extends Command
{
    private CacheWarmerAggregate $cacheWarmer;

    public function __construct(CacheWarmerAggregate $cacheWarmer)
    {
        parent::__construct();

        $this->cacheWarmer = $cacheWarmer;
    }

    protected function configure(): void
    {
        $this
            ->setDefinition([
                new InputOption('no-optional-warmers', '', InputOption::VALUE_NONE, 'Skip optional cache warmers (faster)'),
            ])
            ->setHelp(<<<'EOF'
The <info>%command.name%</info> command warms up the cache.

Before running this command, the cache must be empty.

EOF
            )
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $kernel = $this->getApplication()->getKernel();
        $io->comment(sprintf('Warming up the cache for the <info>%s</info> environment with debug <info>%s</info>', $kernel->getEnvironment(), var_export($kernel->isDebug(), true)));

        if (!$input->getOption('no-optional-warmers')) {
            $this->cacheWarmer->enableOptionalWarmers();
        }
        $cacheDir = $kernel->getContainer()->getParameter('kernel.cache_dir');

        if ($kernel instanceof WarmableInterface) {
            $kernel->warmUp($cacheDir);
        }

        $preload = $this->cacheWarmer->warmUp($cacheDir);

        $buildDir = $kernel->getContainer()->getParameter('kernel.build_dir');
        if ($preload && $cacheDir === $buildDir && file_exists($preloadFile = $buildDir.'/'.$kernel->getContainer()->getParameter('kernel.container_class').'.preload.php')) {
            Preloader::append($preloadFile, $preload);
        }

        $io->success(sprintf('Cache for the "%s" environment (debug=%s) was successfully warmed.', $kernel->getEnvironment(), var_export($kernel->isDebug(), true)));

        return 0;
    }
}
