<?php
/*
 * Copyright (c) 2023. RTM Business
 *
 * @license proprietary
 * Modified by Beau Fiechter on 22-July-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\PostType;

use WRCE\Dependencies\Doctrine\Common\Collections\ArrayCollection;
use WRCE\Dependencies\Doctrine\Common\Collections\Collection;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Entity;
use WRCE\Dependencies\Doctrine\ORM\Mapping\OneToMany;
use WRCE\Dependencies\Symfony\Component\Serializer\Annotation\Context;
use WRCE\Dependencies\Symfony\Component\Serializer\Annotation\Groups;
use WRCE\Dependencies\WordpressModels\ORM\Entity\Post;
use WRCE\Dependencies\WordpressModels\ORM\Entity\User;
use WRCE\Dependencies\WordpressModels\ORM\EntityManagerFactory;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\OrderItem;
use WRCE\Dependencies\WordpressModels\Serialization\TreeDepthObjectNormalizer;

#[Entity]
class ShopOrder extends Post
{

    // line items
    #[OneToMany(mappedBy: 'order', targetEntity: OrderItem::class)]
    #[Groups(['read_order'])]
    #[Context(normalizationContext: [
        TreeDepthObjectNormalizer::ENABLE_MAX_TREE_DEPTH => true,
        TreeDepthObjectNormalizer::TREE_DEPTH_LIMIT => 2,
        'groups' => ['read_order_item_simple']
    ])]
    private Collection $items;

    public function __construct()
    {
        parent::__construct();
        $this->items = new ArrayCollection();
    }

    #[Groups(['read_order'])]
    public function getOrderNumber(): string
    {
        return $this->getMetaValue('_order_number');
    }

    #[Groups(['read_order'])]
    public function getOrderDate(): \DateTimeInterface
    {
        return $this->getDate();
    }

    #[Groups(['read_order'])]
    public function getOrderStatus(): string
    {
        return str_replace('wc-', '', $this->getStatus());
    }

    #[
        Groups(['read_order']),
        Context(
            normalizationContext: [
                TreeDepthObjectNormalizer::ENABLE_MAX_TREE_DEPTH => true,
                TreeDepthObjectNormalizer::TREE_DEPTH_LIMIT => 2,
                'groups' => ['read_user_simple']
            ])]
    public function getCustomer(): User|null
    {
        $metaValue = $this->getMetaValue('_customer_user');
        if (!$metaValue) {
            return null;
        }
        return EntityManagerFactory::instance()->getRepository(User::class)->find($metaValue);
    }

    // order items getter, setter, adder, remover
    public function getItems(): Collection
    {
        return $this->items;
    }

    public function setItems(Collection $items): ShopOrder
    {
        $this->items = $items;
        return $this;
    }

    public function addItem(OrderItem $lineItem): ShopOrder
    {
        $this->items->add($lineItem);
        return $this;
    }

    public function removeItem(OrderItem $lineItem): ShopOrder
    {
        $this->items->removeElement($lineItem);
        return $this;
    }

}