<?php
/*
 * Copyright (c) 2023. RTM Business
 *
 * @license proprietary
 * Modified by Beau Fiechter on 22-July-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\WordpressModels\ORM;

use DateTime;
use WRCE\Dependencies\Doctrine\Common\EventSubscriber;
use WRCE\Dependencies\Doctrine\ORM\Event\OnFlushEventArgs;
use WRCE\Dependencies\Doctrine\ORM\Event\PrePersistEventArgs;
use WRCE\Dependencies\Doctrine\ORM\Event\PreRemoveEventArgs;
use WRCE\Dependencies\Doctrine\ORM\Event\PreUpdateEventArgs;
use WRCE\Dependencies\Doctrine\ORM\Events;
use WRCE\Dependencies\Doctrine\Persistence\Event\LifecycleEventArgs;
use WRCE\Dependencies\Doctrine\Persistence\Mapping\ClassMetadata;
use WRCE\Dependencies\WordpressModels\ORM\Attributes\Timestamp;

/**
 * Doctrine event subscriber to automatically set timestamps for Entities with the Timestampable Trait.
 *
 * Also facilitates soft-deletions.
 */
class TimestampListener implements EventSubscriber
{

    private array $softDeletions = [];

    public function getSubscribedEvents()
    {
        return [
            Events::prePersist,
            Events::preUpdate,
            Events::preRemove,
            Events::onFlush
        ];
    }

    public function prePersist(PrePersistEventArgs $eventArgs)
    {
        $this->updateTimestamps($eventArgs, Events::prePersist);
    }

    public function preUpdate(PreUpdateEventArgs $eventArgs)
    {
        $this->updateTimestamps($eventArgs, Events::preUpdate);
    }

    public function preRemove(PreRemoveEventArgs $eventArgs)
    {
        if ($this->updateTimestamps($eventArgs, Events::preRemove)) {
            $this->softDeletions[] = spl_object_hash($eventArgs->getObject());
        }
    }

    /**
     * Cancel scheduled deletions when soft-deletions are registered.
     *
     * @param OnFlushEventArgs $eventArgs
     * @return void
     */
    public function onFlush(OnFlushEventArgs $eventArgs)
    {
        $objectManager = $eventArgs->getObjectManager();
        $unitOfWork = $objectManager->getUnitOfWork();
        foreach ($unitOfWork->getScheduledEntityDeletions() as $deletion) {
            $classMetadata = $objectManager
                ->getClassMetadata($deletion::class);
            $hash = spl_object_hash($deletion);

            if (in_array($hash, $this->softDeletions)) {
                // cancel the deletion
                $objectManager->persist($deletion);
                $unitOfWork->computeChangeSet($classMetadata, $deletion);
            }
        }
    }

    private function updateTimestamps(LifecycleEventArgs $eventArgs, string $event): bool
    {
        $entity = $eventArgs->getObject();
        $objectManager = $eventArgs->getObjectManager();
        $props = $this->getTimestampProperties($objectManager->getClassMetadata($entity::class), $event);

        $isTimestamped = false;
        foreach ($props as $prop) {
            $prop->setValue($entity, new DateTime());
            $isTimestamped = true;
        }

        return $isTimestamped;
    }

    /**
     * @param ClassMetadata $metadata
     * @param string $event
     * @return \ReflectionProperty[]
     */
    private function getTimestampProperties(ClassMetadata $metadata, string $event): array
    {
        return array_filter($metadata->getReflectionProperties(), function (?\ReflectionProperty $property) use ($event) {
            if (!$property || !($reflectionAttribute = current($property->getAttributes(Timestamp::class)))) {
                return false;
            }
            /** @var Timestamp $timestamp */
            $timestamp = $reflectionAttribute->newInstance();
            return $timestamp->onEvent === $event;
        });
    }

}
