<?php
/**
 * @license MIT
 *
 * Modified by Beau Fiechter on 22-July-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace WRCE\Dependencies\Doctrine\ORM\Cache\Logging;

use WRCE\Dependencies\Doctrine\ORM\Cache\CollectionCacheKey;
use WRCE\Dependencies\Doctrine\ORM\Cache\EntityCacheKey;
use WRCE\Dependencies\Doctrine\ORM\Cache\QueryCacheKey;

use function array_sum;

/**
 * Provide basic second level cache statistics.
 */
class StatisticsCacheLogger implements CacheLogger
{
    /** @var array<string, int> */
    private $cacheMissCountMap = [];

    /** @var array<string, int> */
    private $cacheHitCountMap = [];

    /** @var array<string, int> */
    private $cachePutCountMap = [];

    /**
     * {@inheritDoc}
     */
    public function collectionCacheMiss($regionName, CollectionCacheKey $key)
    {
        $this->cacheMissCountMap[$regionName]
            = ($this->cacheMissCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function collectionCacheHit($regionName, CollectionCacheKey $key)
    {
        $this->cacheHitCountMap[$regionName]
            = ($this->cacheHitCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function collectionCachePut($regionName, CollectionCacheKey $key)
    {
        $this->cachePutCountMap[$regionName]
            = ($this->cachePutCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function entityCacheMiss($regionName, EntityCacheKey $key)
    {
        $this->cacheMissCountMap[$regionName]
            = ($this->cacheMissCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function entityCacheHit($regionName, EntityCacheKey $key)
    {
        $this->cacheHitCountMap[$regionName]
            = ($this->cacheHitCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function entityCachePut($regionName, EntityCacheKey $key)
    {
        $this->cachePutCountMap[$regionName]
            = ($this->cachePutCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function queryCacheHit($regionName, QueryCacheKey $key)
    {
        $this->cacheHitCountMap[$regionName]
            = ($this->cacheHitCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function queryCacheMiss($regionName, QueryCacheKey $key)
    {
        $this->cacheMissCountMap[$regionName]
            = ($this->cacheMissCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * {@inheritDoc}
     */
    public function queryCachePut($regionName, QueryCacheKey $key)
    {
        $this->cachePutCountMap[$regionName]
            = ($this->cachePutCountMap[$regionName] ?? 0) + 1;
    }

    /**
     * Get the number of entries successfully retrieved from cache.
     *
     * @param string $regionName The name of the cache region.
     *
     * @return int
     */
    public function getRegionHitCount($regionName)
    {
        return $this->cacheHitCountMap[$regionName] ?? 0;
    }

    /**
     * Get the number of cached entries *not* found in cache.
     *
     * @param string $regionName The name of the cache region.
     *
     * @return int
     */
    public function getRegionMissCount($regionName)
    {
        return $this->cacheMissCountMap[$regionName] ?? 0;
    }

    /**
     * Get the number of cacheable entries put in cache.
     *
     * @param string $regionName The name of the cache region.
     *
     * @return int
     */
    public function getRegionPutCount($regionName)
    {
        return $this->cachePutCountMap[$regionName] ?? 0;
    }

    /** @return array<string, int> */
    public function getRegionsMiss()
    {
        return $this->cacheMissCountMap;
    }

    /** @return array<string, int> */
    public function getRegionsHit()
    {
        return $this->cacheHitCountMap;
    }

    /** @return array<string, int> */
    public function getRegionsPut()
    {
        return $this->cachePutCountMap;
    }

    /**
     * Clear region statistics
     *
     * @param string $regionName The name of the cache region.
     *
     * @return void
     */
    public function clearRegionStats($regionName)
    {
        $this->cachePutCountMap[$regionName]  = 0;
        $this->cacheHitCountMap[$regionName]  = 0;
        $this->cacheMissCountMap[$regionName] = 0;
    }

    /**
     * Clear all statistics
     *
     * @return void
     */
    public function clearStats()
    {
        $this->cachePutCountMap  = [];
        $this->cacheHitCountMap  = [];
        $this->cacheMissCountMap = [];
    }

    /**
     * Get the total number of put in cache.
     *
     * @return int
     */
    public function getPutCount()
    {
        return array_sum($this->cachePutCountMap);
    }

    /**
     * Get the total number of entries successfully retrieved from cache.
     *
     * @return int
     */
    public function getHitCount()
    {
        return array_sum($this->cacheHitCountMap);
    }

    /**
     * Get the total number of cached entries *not* found in cache.
     *
     * @return int
     */
    public function getMissCount()
    {
        return array_sum($this->cacheMissCountMap);
    }
}
