<?php

namespace RTMCustomMails\WooCommerce;

use ArrayAccess;
use WC_Email;
use WRCE\Dependencies\WordpressModels\DependencyInjection\HookAttributes\Attributes\Filter;
use WRCE\Dependencies\WordpressModels\Traits\SingletonTrait;

/**
 * Class to convert WooCommerce email form fields to JSON Schema.
 */
class WooCommerceSchemaConverter
{
    use SingletonTrait;

    /**
     * Get the JSON Schema for a WooCommerce email.
     *
     * Sub-schemas are generated for each field in a similar way to the WooCommerce HTML renderer, but here
     * we use WordPress hooks to generate the sub-schema associative array.
     *
     * Conditional schemas are not supported.
     *
     * @param WC_Email $email
     * @return array an associative array representing the JSON Schema
     */
    public function getJsonSchema(WC_Email $email): array
    {
        $jsonSchema = [
            '$schema' => 'http://json-schema.org/draft-07/schema#',
            'type' => 'object',
            'properties' => [],
            'required' => [],
        ];

        foreach ($email->get_form_fields() as $fieldKey => $fieldData) {
            // Skip fields with UI properties
            if (isset($fieldData['css'])) {
                continue;
            }

            $field_type = self::determineFieldType($fieldData);

            if ($field_type) {
                // defaults
                $property = [
                        'title' => $fieldData['title'],
                        'default' => $fieldData['default']
                    ] + $field_type;

                // add description if available
                if (isset($fieldData['description'])) {
                    $property['description'] = html_entity_decode($fieldData['description']);
                }

                // add options if available
                if (isset($fieldData['options'])) {
                    // eliminate option groups
                    $flatOptions = self::flatOptions($fieldData['options']);
                    if ($field_type['type'] === 'array') {
                        $property['items']['enum'] = array_keys($flatOptions);
                        $property['items']['enumNames'] = array_values($flatOptions);
                    } else {
                        $property['enum'] = array_keys($flatOptions);
                        $property['enumNames'] = array_values($flatOptions);
                    }
                }

                // add to properties
                $jsonSchema['properties'][$fieldKey] = $property;

                // add to required if required
                if ($fieldData['required'] ?? false) {
                    $jsonSchema['required'][] = $fieldKey;
                }
            }
        }

        return $jsonSchema;
    }

    /**
     * Determine the JSON Schema type for a WooCommerce email field.
     *
     * @param array|ArrayAccess $data
     * @return array|null
     */
    private function determineFieldType(array|ArrayAccess $data): ?array
    {
        return apply_filters('wrce_woocommerce_json_schema_mapping/field_type=' . $data['type'], [
            'type' => 'string', // default (email, color, etc.)
            'format' => $data['type']
        ], $data);
    }

    /**
     * Flatten an array of options.
     *
     * @param array $options
     * @return array
     */
    private function flatOptions(array $options)
    {
        $flattened = [];

        foreach ($options as $key => $value) {
            if (is_array($value)) {
                $flattened = array_merge($flattened, self::flatOptions($value));
            } else {
                $flattened[$key] = $value;
            }
        }

        return $flattened;
    }


    /* Field Mappings */
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=checkbox')]
    public function checkboxFieldMapping()
    {
        return [
            'type' => 'boolean'
        ];
    }

    /**
     * Unformatted string
     * @return string[]
     */
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=text')]
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=textarea')]
    public function stringFieldMapping()
    {
        return [
            'type' => 'string',
        ];
    }

    /**
     * Mapping for select and radio fields for mutex values.
     *
     * @param array $fieldData
     * @return array
     */
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=radio', 10, 2)]
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=select', 10, 2)]
    public function selectFieldMapping(array $_, array|ArrayAccess $fieldData)
    {
        $options = $fieldData['options'] ?? [];
        $itemType = gettype(current(array_keys($options))) !== 'string' ? 'number' : 'string';

        return [
            'type' => $itemType,
            'enum' => array_keys($options),
            'enumNames' => array_values($options),
        ];
    }

    /**
     * Multi-select field mapping.
     * @param array $fieldData
     * @return array
     */
    #[Filter('wrce_woocommerce_json_schema_mapping/field_type=multiselect', 10, 2)]
    public function multiselectFieldMapping(array $_, array|ArrayAccess $fieldData)
    {
        $options = $fieldData['options'] ?? [];
        $itemType = gettype(current(array_keys($options))) !== 'string' ? 'number' : 'string';

        return [
            'type' => 'array',
            'items' => [
                'type' => $itemType,
                'enum' => array_keys($options),
                'enumNames' => array_values($options),
            ],
            "uniqueItems" => true
        ];
    }

}
