<?php

namespace RTMCustomMails\Renderer;

use RTMCustomMails\ContextBuilder\WPContextBuilder;
use RTMCustomMails\Email\RTMCustomEmail;
use RTMMailComposer\ContextBuilder\ContextBuilderRegistry;
use RTMMailComposer\Model\CustomMail;
use RTMMailComposer\Model\EmailSection;
use RTMMailComposer\Renderer\TwigEmailRenderer;
use WRCE\Dependencies\Timber\Loader;
use WRCE\Dependencies\Timber\Timber;
use WRCE\Dependencies\Twig\Environment;
use WRCE\Dependencies\WordpressModels\DependencyInjection\HookAttributes\Attributes\Action;
use WRCE\Dependencies\WordpressModels\DependencyInjection\Initializer\Attributes\ForceInitialize;

#[ForceInitialize]
class TimberEmailRenderer extends TwigEmailRenderer
{
    private static ?TimberEmailRenderer $instance = null;

    public static function instance(?WPContextBuilder $contextBuilder = null)
    {
        if (self::$instance === null) {
            self::$instance = new TimberEmailRenderer($contextBuilder);
        }

        return self::$instance;
    }

    public function __construct(WPContextBuilder $WPContext)
    {
        parent::__construct($WPContext->getRegistry(), (new Loader())->get_twig());
    }

    public function filterTimberContext(array $context)
    {
        if (!doing_action('wrce_email_sections')) {
            return $context;
        }

        // unset internal timber context
        foreach (['http_host', 'wp_title', 'body_class', 'request', 'user', 'theme', 'posts', 'wp_head', 'wp_footer'] as $key) {
            unset($context[$key]);
        }

        return $context;
    }
    #[Action('wrce_email_sections', 10, 2)]
    public function render(CustomMail $mail, object $mailObject, array $initialContext = []): string
    {
        do_action('wrce_register_context_builders', $this->contextBuilder);
        $currentContext = Timber::$context_cache;
        add_filter('timber/context', [$this, 'filterTimberContext']);

        $context = Timber::context() + $initialContext;
        $render = parent::render($mail, $mailObject, $context);

        // reset timber context
        Timber::$context_cache = $currentContext;
        remove_filter('timber/context', [$this, 'filterTimberContext']);

        if ($mail instanceof RTMCustomEmail) {
            $render = $mail->style_inline($render);
        }

        return $render;
    }

    /**
     * @inheritDoc
     */
    #[Action('wrce_render_email_section', 10, 3)]
    public function renderSection(EmailSection $section, array $context): string
    {
        [$action, $action_arguments] = apply_filters('wrce_email_section_action', $section->getId(), $context);

        // set default args for default hook
        if ($action === 'wrce_render_email_section') {
            $renderedSection = parent::renderSection($section, $context);
        } else {
            // used for custom hooks, like woocoomerce_email_order_details
            ob_start();
            do_action($action, ...$action_arguments);
            $renderedSection = ob_get_clean();
            ob_flush();
        }
        return $renderedSection;
    }
}
