<?php

namespace RTMCustomMails\EmailSections;

use RTMCustomMails\ContextBuilder\WPContextBuilder;
use RTMCustomMails\Util;
use RTMMailComposer\Model\EmailSection;
use WRCE\Dependencies\WordpressModels\DependencyInjection\HookAttributes\Attributes\Filter;

class SectionValidation
{
    public function __construct(protected WPContextBuilder $contextBuilder)
    {
    }


    /**
     * @param string $emailId
     * @param EmailSection[] $sections
     * @return array
     */
    public function validateEmailWithSections(string $emailId, array $sections): array
    {
        $email = Util::getEmailById($emailId);
        $errors = [];
        // Fetch allowed variables
        $allowedVariables = array_unique([
            ...apply_filters('wrce_email_allowed_variables/object_type=' . $email->getObjectType(), []),
            ...apply_filters('wrce_email_allowed_variables/email_id=' . $email->id, []),
        ]);

        foreach ($sections as $section) {
            $sectionUsedTokens = self::sectionUsedTokens($section);
            $sectionUsedVariables = $sectionUsedTokens['variables'];

            $unallowedVariables = array_diff($sectionUsedVariables, $allowedVariables);

            foreach ($unallowedVariables as $variable) {
                if (!isset($email->getRenderContext()[$variable])) {
                    $errors[] = [
                        'property' => 'text',
                        'message' => "Variable \"{$variable}\" does not exist."
                    ];
                } else {
                    $errors[] = [
                        'property' => 'text',
                        'message' => "Variable \"{$variable}\" is not allowed in this email."
                    ];
                }
            }
        }

        return $errors;
    }

    public function sectionUsedTokens(EmailSection $section): array
    {
        $text = $section->getText();

        $matches = [];
        $variablesUsed = [];
        if (preg_match_all('/\{\{\s+([\w_]+)[^}(]+}}/', $text, $matches)) {
            $variablesUsed = $matches[1];
        }

        $functionsUsed = [];
        if (preg_match_all('/\{\{\s+([\w_]+)\([^}]+}}/', $text, $matches)) {
            $functionsUsed = $matches[1];
        }

        return ['variables' => $variablesUsed, 'functions' => $functionsUsed];
    }

    #[Filter('wrce_email_allowed_variables/object_type=order')]
    #[Filter('wrce_email_allowed_variables/object_type=user')]
    public function getAllowedContextVarsOrderType($vars = []): array
    {
        do_action('wrce_register_context_builders', $this->contextBuilder->getRegistry());

        $allBuilders = $this->contextBuilder->getRegistry()->getBuilders();

        $allowedVars = [];
        foreach ($allBuilders as $builders) {
            foreach ($builders as $builder) {
                $whitelist = $builder->getWhitelist();
                foreach ($whitelist as $entry) {
                    if (isset($entry['variable'])) {
                        $allowedVars[] = $entry['variable'];
                    }
                }
            }
        }

        return array_unique([...$vars, ...$allowedVars]);
    }
}
