<?php

namespace RTMCustomMails\ContextBuilder;

use RTMMailComposer\Model\CustomMail;
use WC_Abstract_Order;
use WC_Meta_Data;

/**
 * @template T of WC_Abstract_Order
 * @extends ObjectTypeContextBuilder<T>
 */
class OrderContextBuilder extends ObjectTypeContextBuilder
{
    public function __construct()
    {
        parent::__construct('order');
    }

    /**
     * Defines what data is available to email templates when handling orders.
     */
    protected function defineWhitelist(): array
    {
        return [
            ['variable' => 'order_number', 'description' => 'Order number'],
            ['variable' => 'order_total', 'description' => 'Order total amount'],
            ['variable' => 'order_status', 'description' => 'Order status'],
            ['variable' => 'payment_method', 'description' => 'Payment method'],
            ['variable' => 'order_items', 'description' => 'Ordered items'],
            ['variable' => 'billing_name', 'description' => 'Billing name'],
            ['variable' => 'billing_address_1', 'description' => 'Full billing adress'],
            ['variable' => 'billing_address_2', 'description' => 'Billing address line 2'],
            ['variable' => 'billing_city', 'description' => 'Billing city'],
            ['variable' => 'billing_state', 'description' => 'Billing state'],
            ['variable' => 'billing_postcode', 'description' => 'Billing postcode'],
            ['variable' => 'billing_country', 'description' => 'Billing country'],
            ['variable' => 'billing_email', 'description' => 'Billing email address'],
            ['variable' => 'billing_phone', 'description' => 'Billing phone number'],
            ['variable' => 'order_payment_link', 'description' => 'Order payment link'],
            ['variable' => 'order_date', 'description' => 'Order date'],
            ['variable' => 'recipient_name', 'description' => 'Recipient name'],
            ['variable' => 'order_customer_note', 'description' => 'Order customer note'],
            ['variable' => 'shipping_address', 'description' => 'Shipping address'],
        ];
    }

    /**
     * Prepares the data context specifically for orders.
     *
     * @param CustomMail $mail
     * @param T $mailObject
     * @param array $context
     * @return array
     * @throws \ReflectionException
     */
    protected function buildContext(CustomMail $mail, $mailObject, array $context = []): array
    {
        $order = $mailObject ?? null;
        if ($order) {
            $context['order_number'] = $order->get_order_number();
            $context['order_total'] = $order->get_total();
            $context['order_status'] = $order->get_status();
            $context['payment_method'] = $order->get_payment_method_title();
            $context['order_customer_note'] = $order->get_customer_note();
            $context['order_payment_link'] = $order->get_checkout_payment_url();
            $context['order_date'] = $order->get_date_created()->date('Y-m-d H:i:s');
            // Format order items into a readable
            $context['order_items'] = implode(', ', array_map(fn($item) => $item->get_quantity() . ' x ' . $item->get_name(), $order->get_items()));

            // Billing information
            $context['billing_name'] = $order->get_formatted_billing_full_name();
            $context['billing_address_1'] = $order->get_formatted_billing_address();
            $context['billing_address_2'] = $order->get_billing_address_2();
            $context['billing_country'] = $order->get_billing_country();
            $context['billing_email'] = $order->get_billing_email();
            $context['billing_phone'] = $order->get_billing_phone();

            //shipping
            $context['shipping_address'] = $order->get_formatted_shipping_address();

            //recipient
            $context['recipient_name'] = $order->get_billing_first_name() ?: $order->get_billing_email() ?: 'Customer';
        }
        return apply_filters('wrce_inject_context/object_type=order', $context, $mail, $mailObject);
    }

    /**
     * @param WC_Abstract_Order $order
     * @return array
     */
    public function getMeta(WC_Abstract_Order $order): array
    {
        return array_reduce($order->get_meta_data(), function (array $c, WC_Meta_Data $m) {
            $meta_key = $m->get_data()['key'];
            $key = str_starts_with($meta_key, '_') ? substr($meta_key, 1) : $meta_key;
            $c[$key] = $m->get_data()['value'];
            return $c;
        }, []);
    }

    /**
     * @param array $meta
     * @return array
     */
    public function getBilling(array $meta): array
    {
        $billing_meta = array_filter($meta, fn($key) => str_starts_with($key, 'billing_'), ARRAY_FILTER_USE_KEY);
        $meta = array_filter(array_diff_assoc($meta, $billing_meta), fn($k) => !str_starts_with($k, 'buckaroo_'), ARRAY_FILTER_USE_KEY);
        return array($billing_meta, $meta);
    }
}
