<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * Modified by Beau Fiechter on 18-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\Symfony\Bridge\Doctrine\CacheWarmer;

use WRCE\Dependencies\Doctrine\Persistence\ManagerRegistry;
use WRCE\Dependencies\Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerInterface;

/**
 * The proxy generator cache warmer generates all entity proxies.
 *
 * In the process of generating proxies the cache for all the metadata is primed also,
 * since this information is necessary to build the proxies in the first place.
 *
 * @author Benjamin Eberlei <kontakt@beberlei.de>
 */
class ProxyCacheWarmer implements CacheWarmerInterface
{
    public function __construct(
        private readonly ManagerRegistry $registry,
    ) {
    }

    /**
     * This cache warmer is not optional, without proxies fatal error occurs!
     */
    public function isOptional(): bool
    {
        return false;
    }

    public function warmUp(string $cacheDir, ?string $buildDir = null): array
    {
        $files = [];
        foreach ($this->registry->getManagers() as $em) {
            // we need the directory no matter the proxy cache generation strategy
            if (!is_dir($proxyCacheDir = $em->getConfiguration()->getProxyDir())) {
                if (false === @mkdir($proxyCacheDir, 0777, true) && !is_dir($proxyCacheDir)) {
                    throw new \RuntimeException(sprintf('Unable to create the Doctrine Proxy directory "%s".', $proxyCacheDir));
                }
            } elseif (!is_writable($proxyCacheDir)) {
                throw new \RuntimeException(sprintf('The Doctrine Proxy directory "%s" is not writeable for the current system user.', $proxyCacheDir));
            }

            // if proxies are autogenerated we don't need to generate them in the cache warmer
            if ($em->getConfiguration()->getAutoGenerateProxyClasses()) {
                continue;
            }

            $classes = $em->getMetadataFactory()->getAllMetadata();

            $em->getProxyFactory()->generateProxyClasses($classes);

            foreach (scandir($proxyCacheDir) as $file) {
                if (!is_dir($file = $proxyCacheDir.'/'.$file)) {
                    $files[] = $file;
                }
            }
        }

        return $files;
    }
}
