<?php
/*
 * Copyright (c) 2023. RTM Business
 *
 * @license proprietary
 * Modified by Beau Fiechter on 18-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity;

use WRCE\Dependencies\Doctrine\Common\Collections\Collection;
use WRCE\Dependencies\Doctrine\DBAL\Types\Types;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Column;
use WRCE\Dependencies\Doctrine\ORM\Mapping\DiscriminatorColumn;
use WRCE\Dependencies\Doctrine\ORM\Mapping\DiscriminatorMap;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Entity;
use WRCE\Dependencies\Doctrine\ORM\Mapping\GeneratedValue;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Id;
use WRCE\Dependencies\Doctrine\ORM\Mapping\InheritanceType;
use WRCE\Dependencies\Doctrine\ORM\Mapping\JoinColumn;
use WRCE\Dependencies\Doctrine\ORM\Mapping\ManyToOne;
use WRCE\Dependencies\Doctrine\ORM\Mapping\MappedSuperclass;
use WRCE\Dependencies\Doctrine\ORM\Mapping\OneToMany;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Table;
use WRCE\Dependencies\Symfony\Component\Serializer\Attribute\Context;
use WRCE\Dependencies\Symfony\Component\Serializer\Attribute\Groups;
use WRCE\Dependencies\WordpressModels\ORM\Entity\GetMetaValueTrait;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\OrderItem\CouponItem;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\OrderItem\FeeItem;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\OrderItem\LineItem;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\OrderItem\ShippingItem;
use WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity\PostType\ShopOrder;
use WRCE\Dependencies\WordpressModels\Serialization\TreeDepthObjectNormalizer;

#[Entity]
#[Table(name: 'order_items')]
#[
    MappedSuperclass,
    InheritanceType('SINGLE_TABLE'),
    DiscriminatorColumn('order_item_type', type: Types::STRING, length: 200, options: ['default' => 'line_item']),
    DiscriminatorMap([
        'line_item' => LineItem::class,
        'fee' => FeeItem::class,
        'shipping' => ShippingItem::class,
        'coupon' => CouponItem::class
    ])
]
class OrderItem
{

    /**
     * @use GetMetaValueTrait<OrderItemMeta>
     */
    use GetMetaValueTrait;

    #[Id]
    #[GeneratedValue(strategy: 'IDENTITY')]
    #[Column('order_item_id', type: Types::BIGINT, options: ['unsigned' => true])]
    #[Groups(['read_order_item', 'read_order_item_simple'])]
    private ?int $id;

    #[Column('order_item_name', type: Types::STRING, nullable: true)]
    #[Groups(['read_order_item', 'read_order_item_simple'])]
    private ?string $name;

    #[ManyToOne(targetEntity: ShopOrder::class, inversedBy: 'items')]
    #[JoinColumn(name: 'order_id', referencedColumnName: 'ID')]
    #[Groups(['read_order_item'])]
    #[Context(
        normalizationContext: [
            TreeDepthObjectNormalizer::ENABLE_MAX_TREE_DEPTH => true,
            TreeDepthObjectNormalizer::TREE_DEPTH_LIMIT => 1,
            'groups' => ['read_order']
        ]
    )]
    private ShopOrder $order;

    /**
     * @var Collection<OrderItemMeta>
     */
    #[OneToMany(mappedBy: 'orderItem', targetEntity: OrderItemMeta::class)]
    #[JoinColumn(name: 'order_item_id', referencedColumnName: 'order_item_id')]
    #[Groups(['read_order_item', 'read_order_item_simple'])]
    private Collection $meta;

    /**
     * @return int|null
     */
    public function getId(): ?int
    {
        return $this->id;
    }

    /**
     * @param int|null $id
     * @return OrderItem
     */
    public function setId(?int $id): OrderItem
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param string|null $name
     * @return OrderItem
     */
    public function setName(?string $name): OrderItem
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @return ShopOrder
     */
    public function getOrder(): ShopOrder
    {
        return $this->order;
    }

    /**
     * @param ShopOrder $order
     * @return OrderItem
     */
    public function setOrder(ShopOrder $order): OrderItem
    {
        $this->order = $order;
        return $this;
    }

    /**
     * @return Collection
     */
    public function getMeta(): Collection
    {
        return $this->meta;
    }

    /**
     * @param Collection $meta
     * @return OrderItem
     */
    public function setMeta(Collection $meta): OrderItem
    {
        $this->meta = $meta;
        return $this;
    }

}