<?php
/**
 * @license MIT
 *
 * Modified by Beau Fiechter on 18-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace WRCE\Dependencies\Doctrine\ORM\Query\AST\Functions;

use WRCE\Dependencies\Doctrine\ORM\Query\AST\Node;
use WRCE\Dependencies\Doctrine\ORM\Query\Parser;
use WRCE\Dependencies\Doctrine\ORM\Query\QueryException;
use WRCE\Dependencies\Doctrine\ORM\Query\SqlWalker;
use WRCE\Dependencies\Doctrine\ORM\Query\TokenType;

use function strtolower;

/**
 * "DATE_ADD" "(" ArithmeticPrimary "," ArithmeticPrimary "," StringPrimary ")"
 *
 * @link    www.doctrine-project.org
 */
class DateAddFunction extends FunctionNode
{
    /** @var Node */
    public $firstDateExpression = null;

    /** @var Node */
    public $intervalExpression = null;

    /** @var Node */
    public $unit = null;

    /** @inheritDoc */
    public function getSql(SqlWalker $sqlWalker)
    {
        switch (strtolower($this->unit->value)) {
            case 'second':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddSecondsExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'minute':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddMinutesExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'hour':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddHourExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'day':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddDaysExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'week':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddWeeksExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'month':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddMonthExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            case 'year':
                return $sqlWalker->getConnection()->getDatabasePlatform()->getDateAddYearsExpression(
                    $this->firstDateExpression->dispatch($sqlWalker),
                    $this->intervalExpression->dispatch($sqlWalker)
                );

            default:
                throw QueryException::semanticalError(
                    'DATE_ADD() only supports units of type second, minute, hour, day, week, month and year.'
                );
        }
    }

    /** @inheritDoc */
    public function parse(Parser $parser)
    {
        $parser->match(TokenType::T_IDENTIFIER);
        $parser->match(TokenType::T_OPEN_PARENTHESIS);

        $this->firstDateExpression = $parser->ArithmeticPrimary();
        $parser->match(TokenType::T_COMMA);
        $this->intervalExpression = $parser->ArithmeticPrimary();
        $parser->match(TokenType::T_COMMA);
        $this->unit = $parser->StringPrimary();

        $parser->match(TokenType::T_CLOSE_PARENTHESIS);
    }
}
