<?php
/**
 * @license MIT
 *
 * Modified by Beau Fiechter on 18-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace WRCE\Dependencies\Doctrine\ORM\Persisters;

use WRCE\Dependencies\Doctrine\Common\Collections\Expr\Comparison;
use WRCE\Dependencies\Doctrine\Common\Collections\Expr\CompositeExpression;
use WRCE\Dependencies\Doctrine\Common\Collections\Expr\ExpressionVisitor;
use WRCE\Dependencies\Doctrine\Common\Collections\Expr\Value;
use WRCE\Dependencies\Doctrine\ORM\Mapping\ClassMetadata;
use WRCE\Dependencies\Doctrine\ORM\Persisters\Entity\BasicEntityPersister;
use RuntimeException;

use function defined;
use function implode;
use function in_array;
use function is_object;

/**
 * Visit Expressions and generate SQL WHERE conditions from them.
 */
class SqlExpressionVisitor extends ExpressionVisitor
{
    /** @var BasicEntityPersister */
    private $persister;

    /** @var ClassMetadata */
    private $classMetadata;

    public function __construct(BasicEntityPersister $persister, ClassMetadata $classMetadata)
    {
        $this->persister     = $persister;
        $this->classMetadata = $classMetadata;
    }

    /**
     * Converts a comparison expression into the target query language output.
     *
     * @return mixed
     */
    public function walkComparison(Comparison $comparison)
    {
        $field = $comparison->getField();
        $value = $comparison->getValue()->getValue(); // shortcut for walkValue()

        if (
            isset($this->classMetadata->associationMappings[$field]) &&
            $value !== null &&
            ! is_object($value) &&
            ! in_array($comparison->getOperator(), [Comparison::IN, Comparison::NIN], true)
        ) {
            throw MatchingAssociationFieldRequiresObject::fromClassAndAssociation(
                $this->classMetadata->name,
                $field
            );
        }

        return $this->persister->getSelectConditionStatementSQL($field, $value, null, $comparison->getOperator());
    }

    /**
     * Converts a composite expression into the target query language output.
     *
     * @return string
     *
     * @throws RuntimeException
     */
    public function walkCompositeExpression(CompositeExpression $expr)
    {
        $expressionList = [];

        foreach ($expr->getExpressionList() as $child) {
            $expressionList[] = $this->dispatch($child);
        }

        switch ($expr->getType()) {
            case CompositeExpression::TYPE_AND:
                return '(' . implode(' AND ', $expressionList) . ')';

            case CompositeExpression::TYPE_OR:
                return '(' . implode(' OR ', $expressionList) . ')';

            default:
                // Multiversion support for `doctrine/collections` before and after v2.1.0
                if (defined(CompositeExpression::class . '::TYPE_NOT') && $expr->getType() === CompositeExpression::TYPE_NOT) {
                    return 'NOT (' . $expressionList[0] . ')';
                }

                throw new RuntimeException('Unknown composite ' . $expr->getType());
        }
    }

    /**
     * Converts a value expression into the target query language part.
     *
     * @return string
     */
    public function walkValue(Value $value)
    {
        return '?';
    }
}
