<?php
/*
 * Copyright (c) 2023. RTM Business
 *
 * @license proprietary
 * Modified by Beau Fiechter on 17-June-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace WRCE\Dependencies\WordpressModels\ORM\WooCommerceEntity;

use WRCE\Dependencies\Doctrine\DBAL\Types\Types;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Column;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Entity;
use WRCE\Dependencies\Doctrine\ORM\Mapping\GeneratedValue;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Id;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Index;
use WRCE\Dependencies\Doctrine\ORM\Mapping\Table;

/**
 * Doctrine entity matching the `wp_woocommerce_downloadable_product_permissions` table.
 *
 * Schema:
 * ```sql
 * create table wp_woocommerce_downloadable_product_permissions
 * (
 * permission_id       bigint unsigned auto_increment
 * primary key,
 * download_id         varchar(36)                                   not null,
 * product_id          bigint unsigned                               not null,
 * order_id            bigint unsigned default '0'                   not null,
 * order_key           varchar(200)                                  not null,
 * user_email          varchar(200)                                  not null,
 * user_id             bigint unsigned                               null,
 * downloads_remaining varchar(9)                                    null,
 * access_granted      datetime        default '0000-00-00 00:00:00' not null,
 * access_expires      datetime                                      null,
 * download_count      bigint unsigned default '0'                   not null
 * )
 * collate = utf8mb4_unicode_520_ci;
 *
 * create index download_order_key_product
 * on wp_woocommerce_downloadable_product_permissions (product_id, order_id, order_key(191), download_id);
 *
 * create index download_order_product
 * on wp_woocommerce_downloadable_product_permissions (download_id, order_id, product_id);
 *
 * create index order_id
 * on wp_woocommerce_downloadable_product_permissions (order_id);
 *
 * create index user_order_remaining_expires
 * on wp_woocommerce_downloadable_product_permissions (user_id, order_id, downloads_remaining, access_expires);
 * ```
 */
#[Entity]
#[Table(name: 'downloadable_product_permissions')]
#[
    Index(
        name: 'download_order_key_product',
        columns: ['product_id', 'order_id', 'order_key', 'download_id']),
    Index(
        name: 'download_order_product',
        columns: ['download_id', 'order_id', 'product_id']),
    Index(
        name: 'user_order_remaining_expires',
        columns: ['user_id', 'order_id', 'downloads_remaining', 'access_expires'])
]
class DownloadableProductPermission
{

    #[Id]
    #[Column(name: 'permission_id', type: Types::BIGINT, options: ['unsigned' => true])]
    #[GeneratedValue(strategy: 'IDENTITY')]
    private ?int $id = null;

    #[Column(name: 'download_id', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $downloadId;

    #[Column(name: 'product_id', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $productId;

    #[Column(name: 'order_id', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $orderId;

    #[Column(name: 'order_key', type: Types::STRING, length: 200)]
    private string $orderKey;

    #[Column(name: 'user_email', type: Types::STRING, length: 200)]
    private string $userEmail;

    #[Column(name: 'user_id', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $userId;

    #[Column(name: 'downloads_remaining', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $downloadsRemaining;

    #[Column(name: 'access_granted', type: Types::DATETIME_IMMUTABLE)]
    private \DateTimeImmutable $accessGranted;

    #[Column(name: 'access_expires', type: Types::DATETIME_IMMUTABLE)]
    private \DateTimeImmutable $accessExpires;

    #[Column(name: 'download_count', type: Types::BIGINT, options: ['unsigned' => true])]
    private int $downloadCount;

    /**
     * @return int|null
     */
    public function getId(): ?int
    {
        return $this->id;
    }

    /**
     * @param int|null $id
     * @return DownloadableProductPermission
     */
    public function setId(?int $id): DownloadableProductPermission
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return int
     */
    public function getDownloadId(): int
    {
        return $this->downloadId;
    }

    /**
     * @param int $downloadId
     * @return DownloadableProductPermission
     */
    public function setDownloadId(int $downloadId): DownloadableProductPermission
    {
        $this->downloadId = $downloadId;
        return $this;
    }

    /**
     * @return int
     */
    public function getProductId(): int
    {
        return $this->productId;
    }

    /**
     * @param int $productId
     * @return DownloadableProductPermission
     */
    public function setProductId(int $productId): DownloadableProductPermission
    {
        $this->productId = $productId;
        return $this;
    }

    /**
     * @return int
     */
    public function getOrderId(): int
    {
        return $this->orderId;
    }

    /**
     * @param int $orderId
     * @return DownloadableProductPermission
     */
    public function setOrderId(int $orderId): DownloadableProductPermission
    {
        $this->orderId = $orderId;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderKey(): string
    {
        return $this->orderKey;
    }

    /**
     * @param string $orderKey
     * @return DownloadableProductPermission
     */
    public function setOrderKey(string $orderKey): DownloadableProductPermission
    {
        $this->orderKey = $orderKey;
        return $this;
    }

    /**
     * @return string
     */
    public function getUserEmail(): string
    {
        return $this->userEmail;
    }

    /**
     * @param string $userEmail
     * @return DownloadableProductPermission
     */
    public function setUserEmail(string $userEmail): DownloadableProductPermission
    {
        $this->userEmail = $userEmail;
        return $this;
    }

    /**
     * @return int
     */
    public function getUserId(): int
    {
        return $this->userId;
    }

    /**
     * @param int $userId
     * @return DownloadableProductPermission
     */
    public function setUserId(int $userId): DownloadableProductPermission
    {
        $this->userId = $userId;
        return $this;
    }

    /**
     * @return int
     */
    public function getDownloadsRemaining(): int
    {
        return $this->downloadsRemaining;
    }

    /**
     * @param int $downloadsRemaining
     * @return DownloadableProductPermission
     */
    public function setDownloadsRemaining(int $downloadsRemaining): DownloadableProductPermission
    {
        $this->downloadsRemaining = $downloadsRemaining;
        return $this;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getAccessGranted(): \DateTimeImmutable
    {
        return $this->accessGranted;
    }

    /**
     * @param \DateTimeImmutable $accessGranted
     * @return DownloadableProductPermission
     */
    public function setAccessGranted(\DateTimeImmutable $accessGranted): DownloadableProductPermission
    {
        $this->accessGranted = $accessGranted;
        return $this;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getAccessExpires(): \DateTimeImmutable
    {
        return $this->accessExpires;
    }

    /**
     * @param \DateTimeImmutable $accessExpires
     * @return DownloadableProductPermission
     */
    public function setAccessExpires(\DateTimeImmutable $accessExpires): DownloadableProductPermission
    {
        $this->accessExpires = $accessExpires;
        return $this;
    }

    /**
     * @return int
     */
    public function getDownloadCount(): int
    {
        return $this->downloadCount;
    }

    /**
     * @param int $downloadCount
     * @return DownloadableProductPermission
     */
    public function setDownloadCount(int $downloadCount): DownloadableProductPermission
    {
        $this->downloadCount = $downloadCount;
        return $this;
    }

}