<?php
/**
 * Class WC_Email_Customer_Note file.
 *
 * @package WooCommerce\Emails
 */

namespace RTMCustomMails\Email;

/**
 * Customer Note Order Email.
 *
 * Customer note emails are sent when you add a note to an order.
 *
 * @class       WC_Email_Customer_Note
 * @version     3.5.0
 * @package     WooCommerce\Classes\Emails
 * @extends     RTMCustomEmail
 */
class RTMCustomEmail_Customer_Note extends RTMCustomEmail
{

    /**
     * Customer note.
     *
     * @var string
     */
    public $customer_note;

    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->id = 'customer_note';
        $this->customer_email = true;
        $this->title = __('Customer note', 'woocommerce');
        $this->description = __('Customer note emails are sent when you add a note to an order.', 'woocommerce');
        $this->template_html = '';
        $this->template_plain = '';
        $this->placeholders = [
            '{order_date}' => '',
            '{order_number}' => '',
        ];

        // Call parent constructor.
        parent::__construct();

        $this->object_type = 'order';
    }

    /**
     * Get email subject.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_subject()
    {
        return __('Note added to your {site_title} order from {order_date}', 'woocommerce');
    }

    /**
     * Get email heading.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_heading()
    {
        return __('A note has been added to your order', 'woocommerce');
    }

    /**
     * Trigger.
     *
     * @param array $args Email arguments.
     */
    public function trigger($args = [])
    {
        $this->setup_locale();

        if (!empty($args)) {
            $defaults = [
                'order_id' => '',
                'customer_note' => '',
            ];

            $args = wp_parse_args($args, $defaults);

            $order_id = $args['order_id'];
            $customer_note = $args['customer_note'];

            if ($order_id) {
                $this->object = wc_get_order($order_id);

                if ($this->object) {
                    $this->recipient = $this->object->get_billing_email();
                    $this->customer_note = $customer_note ?: $this->object->get_customer_note();
                    $this->placeholders['{order_date}'] = wc_format_datetime($this->object->get_date_created());
                    $this->placeholders['{order_number}'] = $this->object->get_order_number();
                }
            }
        }

        if ($this->is_enabled() && $this->get_recipient()) {
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }

        $this->restore_locale();
    }

    /**
     * Get content plain.
     *
     * @return string
     */
    public function get_content_plain()
    {
        return wc_get_template_html(
            $this->template_plain,
            [
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'additional_content' => $this->get_additional_content(),
                'customer_note' => $this->customer_note,
                'sent_to_admin' => false,
                'plain_text' => true,
                'email' => $this,
            ]
        );
    }

    protected function prepareContext(array $baseContext = []): array
    {
        return [
            ...parent::prepareContext($baseContext),
            'customer_note' => $this->customer_note,
        ];
    }

    /**
     * Default content to show below main email content.
     *
     * @return string
     * @since 3.7.0
     */
    public function get_default_additional_content()
    {
        return __('Thanks for reading.', 'woocommerce');
    }

    public function getTriggerHooks(): array
    {
        return ['woocommerce_new_customer_note_notification'];
    }
}
