<?php
/**
 * Class WC_Email_Customer_On_Hold_Order file.
 *
 * @package WooCommerce\Emails
 */

namespace RTMCustomMails\Email;


use WC_Order;

/**
 * Customer On-hold Order Email.
 *
 * An email sent to the customer when a new order is on-hold for.
 *
 * @class       WC_Email_Customer_On_Hold_Order
 * @version     2.6.0
 * @package     WooCommerce\Classes\Emails
 * @extends     RTMCustomEmail
 */
class RTMCustomEmail_Customer_On_Hold_Order extends RTMCustomEmail
{

    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->id = 'customer_on_hold_order';
        $this->customer_email = true;
        $this->title = __('Order on-hold', 'woocommerce');
        $this->description = __('This is an order notification sent to customers containing order details after an order is placed on-hold from Pending, Cancelled or Failed order status.', 'woocommerce');
        $this->template_html = '';
        $this->template_plain = '';
        $this->placeholders = [
            '{order_date}' => '',
            '{order_number}' => '',
        ];

        // Call parent constructor.
        parent::__construct();
    }

    /**
     * Get email subject.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_subject()
    {
        return __('Your {site_title} order has been received!', 'woocommerce');
    }

    /**
     * Get email heading.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_heading()
    {
        return __('Thank you for your order', 'woocommerce');
    }

    /**
     * Trigger the sending of this email.
     *
     * @param int $order_id The order ID.
     * @param WC_Order|false $order Order object.
     */
    public function trigger($order_id = 0, $order = false)
    {
        $this->setup_locale();

        if ($order_id && !is_a($order, 'WC_Order')) {
            $order = wc_get_order($order_id);
        }

        if (is_a($order, 'WC_Order')) {
            $this->object = $order;
            $this->recipient = $this->object->get_billing_email();
            $this->placeholders['{order_date}'] = wc_format_datetime($this->object->get_date_created());
            $this->placeholders['{order_number}'] = $this->object->get_order_number();
        }

        if ($this->is_enabled() && $this->get_recipient()) {
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }

        $this->restore_locale();
    }

    /**
     * Get content plain.
     *
     * @return string
     */
    public function get_content_plain()
    {
        return wc_get_template_html(
            $this->template_plain,
            [
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'additional_content' => $this->get_additional_content(),
                'sent_to_admin' => false,
                'plain_text' => true,
                'email' => $this,
            ]
        );
    }

    /**
     * Default content to show below main email content.
     *
     * @return string
     * @since 3.7.0
     */
    public function get_default_additional_content()
    {
        return __('We look forward to fulfilling your order soon.', 'woocommerce');
    }

    public function getTriggerHooks(): array
    {
        return ['woocommerce_order_status_pending_to_on-hold_notification',
            'woocommerce_order_status_failed_to_on-hold_notification',
            'woocommerce_order_status_cancelled_to_on-hold_notification'];
    }
}

