<?php
/*
 * Copyright (c) 2023. RTM Business
 */

namespace WordpressModels\ORM\Entity;

use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Serializer\Attribute\Groups;

/**
 * Entity mapping for generic _meta tables.
 *
 * This class is used for all meta tables: {@see PostMeta}, {@see UserMeta}, {@see TermMeta}, {@see CommentMeta}.
 *
 * Associated entities should be defined in the extending class, for example {@see PostMeta::$post}. Ensure that the
 * serialization groups on the associated entity are set to `read_meta_details`.
 *
 * Serialization groups:
 * - read_meta -- for reading meta data
 * - read_meta_details -- for reading meta data with associated entities
 */
#[ORM\Index(name: 'meta_key', columns: ['meta_key'])]
abstract class AbstractMeta
{

    #[ORM\Column(name: 'meta_id', type: Types::BIGINT, options: ['unsigned' => true])]
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: 'IDENTITY')]
    #[Groups('read_meta')]
    protected ?int $id = null;

    /**
     * @var string|null
     */
    #[ORM\Column(name: 'meta_key', type: Types::STRING, length: 255, nullable: true)]
    #[Groups('read_meta')]
    protected ?string $key = null;

    /**
     * @var string|null
     */
    #[ORM\Column(name: 'meta_value', type: Types::TEXT, length: 0, nullable: true)]
    #[Groups('read_meta')]
    protected ?string $value = null;

    public function getId(): ?int
    {
        return $this->id;
    }

    public function setId(?int $id): self
    {
        $this->id = $id;
        return $this;
    }

    public function getKey(): ?string
    {
        return $this->key;
    }

    public function setKey(?string $key): self
    {
        $this->key = $key;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getValue(): ?string
    {
        return $this->value;
    }

    /**
     * Check the serialization (json or php) of the current value, and return the deserialized value.
     *
     * @return mixed
     */
    public function getValueDeserialized(): mixed
    {
        if ($this->isJson($this->value)) {
            return json_decode($this->value);
        } elseif (is_serialized($this->value)) {
            return unserialize($this->value);
        }

        return $this->value;
    }

    private function isJson(?string $argument, bool $ignore_scalars = true): bool
    {
        if (!is_string($argument) || '' === $argument) {
            return false;
        }

        if ($ignore_scalars && !in_array($argument[0], ['{', '['], true)) {
            return false;
        }

        json_decode($argument, true);

        return json_last_error() === JSON_ERROR_NONE;
    }

    /**
     * Check the serialization (json or php) of the current value, and replicate with the new value.
     *
     * If the current value is not serialized, the new value will be set as is.
     * If the current value is serialized and the given value is also serialized, the new value will be set as is.
     *
     * If the given value is an array or an object, it will always be serialized as php.
     *
     * @param mixed $value
     * @return $this
     */
    public function setValue(mixed $value): self
    {
        if ($this->isJson($this->value) && !$this->isJson($value)) {
            $this->value = json_encode($value);
        } elseif ((is_serialized($this->value) && !is_serialized($value))
            || is_array($value) || is_object($value)) {
            $this->value = serialize(is_object($value) ? (array)$value : $value);
        } else {
            $this->value = $value;
        }

        return $this;
    }

    /**
     * Getter for the meta type.
     *
     * Used for passing the type to hook calls (e.g. `updated_post_meta` for type 'post').
     *
     * @return string
     */
    public abstract static function getType(): string;
}