<?php

namespace RTMCustomMails\Rest\Model;

use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\EmailOptions\AbstractEmailEnabledCondition;
use RTMCustomMails\PDFInvoices\WoocommercePDFInvoices;
use RTMCustomMails\Schedule\ScheduledEmails;
use WC_Email;
use WPO\WC\PDF_Invoices\Documents\Order_Document;

class EmailModel
{

    private string $id;
    private string $title;
    private string $objectType;
    private string $recipientType;

    private array $sections;
    private array $attachments;
    private array $settings;

    private bool $schedulable;
    private bool $enabled;
    private bool $conditional = false;

    /**
     * @return string
     */
    public function getId(): string
    {
        return $this->id;
    }

    /**
     * @return array
     */
    public function getAttachments(): array
    {
        return $this->attachments;
    }

    /**
     * @return string
     */
    public function getObjectType(): string
    {
        return $this->objectType;
    }

    /**
     * @return string
     */
    public function getRecipientType(): string
    {
        return $this->recipientType;
    }

    /**
     * @return array
     */
    public function getSections(): array
    {
        return $this->sections;
    }

    /**
     * @return array
     */
    public function getSettings(): array
    {
        return $this->settings;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @return bool
     */
    public function isSchedulable(): bool
    {
        return $this->schedulable;
    }

    /**
     * @return bool
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    public static function fromCustomEmail(WC_Email $email)
    {
        $email = RTMCustomEmail::asCustomEmail($email);
        $model = new EmailModel();
        $model->id = $email->id;
        $model->title = $email->get_title();
        $model->objectType = $email->getObjectType();
        $model->recipientType = $email->getRecipientType();

        $model->sections = $email->getSections();

        $model->attachments = array_values(array_map(fn(Order_Document $document) => $document->get_type(), WoocommercePDFInvoices::getDocumentsForEmail($email->id)));

        $settings = array_filter($email->settings, fn(string $key) => !in_array($key, ['sections']), ARRAY_FILTER_USE_KEY);
        $formFields = $email->get_form_fields();
        foreach ($settings as $key => $setting) {
            if ($formFields[$key]['type'] === 'checkbox') {
                $settings[$key] = $setting === 'yes';
            }
            if ($formFields[$key]['type'] === 'multiselect') {
                $options = $formFields[$key]['options'];
                $mapperFn = gettype(array_key_first($options)) === 'string' ? 'strval' : 'intval';
                $settings[$key] = array_map($mapperFn, $setting);
            }
        }

        $model->settings = $settings;

        $model->schedulable = isset(ScheduledEmails::getSchedulableEmails()[$email->id]);
        $model->enabled = $email->get_option('enabled') === 'yes';
        $model->conditional = $model->isConditional();
        return $model;
    }

    public function isConditional(): bool
    {
        global $wp_filter;
        $hooks = $wp_filter['woocommerce_email_enabled_' . $this->id];
        foreach ($hooks as $hook) {
            foreach ($hook as $callback) {
                if (is_array($callback['function']) && $callback['function'][0] instanceof AbstractEmailEnabledCondition) {
                    $this->conditional = true;
                    break;
                }
            }
        }

        return $this->conditional;
    }

}